<?php declare(strict_types = 1);

namespace EshopCatalog\FrontModule\Model\Export;

use EshopCatalog\FrontModule\Model\Dao\Product;
use EshopCatalog\FrontModule\Model\Provider\IXmlExportService;
use Nette\Utils\Strings;

class ZboziXmlService extends XmlService implements IXmlExportService
{
	/** @var string */
	protected $name = 'zbozi';

	public function startExport(): void
	{
		parent::startExport();

		$this->writeToFile('<?xml version="1.0" encoding="utf-8"?>' . PHP_EOL .
			'<SHOP xmlns="http://www.zbozi.cz/ns/offer/1.0">' . PHP_EOL);
	}

	public function writeItem(Product $product)
	{
		$export = $this->prepareExportData($product);

		if ($export['status'] == 0)
			return;

		$dom               = new \DOMDocument();
		$dom->formatOutput = true;
		$dom->encoding     = 'utf-8';
		$shopitem          = $dom->createElement('SHOPITEM');

		$child = $dom->createElement('ITEM_ID', (string) $product->getId());
		$shopitem->appendChild($child);

		$child     = $dom->createElement('PRODUCTNAME');
		$childText = $dom->createTextNode((string) $export['productName']);
		$child->appendChild($childText);
		$shopitem->appendChild($child);

		$child     = $dom->createElement('PRODUCT');
		$childText = $dom->createTextNode((string) $export['product']);
		$child->appendChild($childText);
		$shopitem->appendChild($child);

		$child     = $dom->createElement('DESCRIPTION');
		$childText = $dom->createCDATASection(Strings::normalize(strip_tags((string) $product->getDescription())));
		$child->appendChild($childText);
		$shopitem->appendChild($child);

		$child = $dom->createElement('URL', (string) $product->link);
		$shopitem->appendChild($child);

		$child = $dom->createElement('PRICE_VAT', number_format($product->getPrice(), 2, ".", ""));
		$shopitem->appendChild($child);

		$child = $dom->createElement('DELIVERY_DATE', ($product->quantity > 0) ? '0' : '30');
		$shopitem->appendChild($child);

		if ($product->getGallery() && $product->getGallery()->cover) {
			$child = $dom->createElement('IMGURL', (string) ($this->baseUrl . $product->getGallery()->cover->file));
			$shopitem->appendChild($child);
		}

		if ($product->getEan()) {
			$child = $dom->createElement('EAN', (string) $product->getEan());
			$shopitem->appendChild($child);
		}

		if ($export['categoryText']) {
			$child = $dom->createElement('CATEGORYTEXT', (string) $export['categoryText']);
			$shopitem->appendChild($child);
		}

		if ($product->getManufacturer() && trim($product->getManufacturer()->name)) {
			$child = $dom->createElement('MANUFACTURER', (string) $product->getManufacturer()->name);
			$shopitem->appendChild($child);
		}

		if ($export['bidCpc']) {
			$child = $dom->createElement('MAX_CPC', $export['bidCpc']);
			$shopitem->appendChild($child);
		}

		foreach ($product->getGifts() as $gift) {
			$child = $dom->createElement('FREE_GIFT_TEXT', $gift->getName());
			$shopitem->appendChild($child);
		}

		$dom->appendChild($shopitem);

		$this->writeToFile($dom->saveXML($dom->documentElement) . PHP_EOL);
	}

	public function endExport()
	{
		$this->writeToFile('</SHOP>');

		parent::endExport();
	}
}
