<?php declare(strict_types = 1);

namespace EshopCatalog\FrontModule\Model;

use Core\Model\Helpers\BaseFrontEntityService;
use Doctrine\ORM\Query;
use EshopCatalog\FrontModule\Model\Dao;
use EshopCatalog\Model\Config;
use EshopCatalog\Model\Entities\ProductTag;
use EshopCatalog\Model\Entities\Tag;
use EshopOrders\Model\Entities\Spedition;
use Nette\Caching\Cache;
use Nette\Utils\DateTime;

/**
 * Class Tags
 * @package EshopCatalog\FrontModule\Model
 */
class Tags extends BaseFrontEntityService
{
	/** @var Cache */
	protected $productsCache;

	const CACHE_NAMESPACE = 'eshopCatalogTags';

	protected $entityClass = Tag::class;

	/** @var array */
	protected $cTags;

	/** @var int */
	protected $cFreeFrom;

	/** @var array */
	protected $cacheDep = [
		Cache::TAGS   => [self::CACHE_NAMESPACE, Products::CACHE_NAMESPACE],
		Cache::EXPIRE => '1 week',
	];

	public function getCache()
	{
		if ($this->cache === null)
			$this->cache = new Cache($this->cacheStorage, self::CACHE_NAMESPACE);

		return $this->cache;
	}

	public function getProductsCache()
	{
		if ($this->productsCache === null)
			$this->productsCache = new Cache($this->cacheStorage, Products::CACHE_NAMESPACE);

		return $this->productsCache;
	}

	/**
	 * @param Dao\Product|Dao\Product[] $product
	 *
	 * @throws \Doctrine\ORM\NonUniqueResultException
	 */
	public function loadTagsToProduct(&$product)
	{
		$products = is_array($product) ? $product : [$product];

		$tags = $this->getIdsForProducts();

		$tagForPrice         = Config::load('tagWhenPriceIsLowerThenRetail');
		$freeDelivery        = $this->get('freeDelivery');
		$freeDeliveryMinimum = $this->getFreeDeliveryMinimumPrice();

		foreach ($products as &$prod) {
			if (isset($tags[$prod->getId()])) {
				foreach ($tags[$prod->getId()] as $tagId)
					$prod->tags[$tagId] = $this->get($tagId);
			}

			if ($tagForPrice && $prod->getPrice() < $prod->getRetailPrice())
				$prod->tags[$tagForPrice] = $this->get($tagForPrice);

			if ($freeDelivery && $prod->getPrice() >= $freeDeliveryMinimum) {
				$freeDelivery->isAuto       = true;
				$prod->tags['freeDelivery'] = $freeDelivery;
			}
		}
	}

	public function getIdsForProducts(): array
	{
		$result = [];
		$now    = (new DateTime())->format('Y-m-d H:i:00');

		foreach ($this->getEr()->createQueryBuilder('t')
			         ->select('t.id, t.type, pt.validFrom, pt.validTo, IDENTITY(pt.product) as product')
			         ->join(ProductTag::class, 'pt', 'WITH', 'pt.tag = t')
			         ->andWhere('pt.validFrom IS NULL OR pt.validFrom >= :now')
			         ->andWhere('pt.validTo IS NULL OR pt.validTo <= :now')
			         ->setParameter('now', $now)
			         ->getQuery()->useResultCache(true, 30)->getArrayResult() as $row) {
			if (($row['validFrom'] == null || $row['validFrom']->format('Y-m-d H:i:s') >= $now)
				&& ($row['validTo'] == null || $row['validTo']->format('Y-m-d H:i:s') <= $now))
				$result[$row['product']][] = $row['type'];
		}

		return $result;
	}

	/**
	 * @param int|string $id
	 *
	 * @return Dao\Tag|null
	 * @throws \Doctrine\ORM\NonUniqueResultException
	 */
	public function get($id)
	{
		if ($this->cTags === null) {
			$this->cTags = [];

			foreach ($this->getEr()->createQueryBuilder('t')
				         ->select('t.id, t.type, t.image, tt.name, t.color, t.bgColor')
				         ->join('t.texts', 'tt', 'WITH', 'tt.lang = :lang')
				         ->setParameter('lang', $this->translator->getLocale())
				         ->getQuery()->enableResultCache(60)->getArrayResult() as $row) {
				$tag                       = $this->fillDao($row);
				$this->cTags[$row['type']] = $tag;
				$this->cTags[$row['id']]   = $tag;
			}
		}

		return $this->cTags[$id] ?? null;
	}

	/**
	 * @param array $tag
	 *
	 * @return Dao\Tag
	 */
	protected function fillDao($tag)
	{
		$t = (new Dao\Tag())
			->setId($tag['id'])
			->setType($tag['type'])
			->setName($tag['name'])
			->setIsAuto(false)
			->setImage($tag['image'])
			->setTextColor($tag['color'])
			->setBgColor($tag['bgColor']);

		return $t;
	}

	/**
	 * TODO uložit do cache doprav
	 *
	 * @return int
	 */
	protected function getFreeDeliveryMinimumPrice()
	{
		if (!$this->cFreeFrom) {
			$v = $this->em->getRepository(Spedition::class)->createQueryBuilder('s')->select('s.freeFrom')
				->andWhere('s.freeFrom > 0')
				->setMaxResults(1)->orderBy('s.freeFrom', 'ASC')
				->getQuery()->useResultCache(true, 60)->getOneOrNullResult();

			$this->cFreeFrom = $v['freeFrom'];
		}

		return $this->cFreeFrom;
	}
}
