<?php declare(strict_types = 1);

namespace EshopCatalog\AdminModule\Components\Availability;

use Core\Model\Helpers\Arrays;
use Core\Model\Helpers\FormHelper;
use Core\Model\UI\BaseControl;
use Core\Model\UI\Form\BaseForm;
use EshopCatalog\AdminModule\Model\AvailabilityService;
use EshopCatalog\Model\Config;
use EshopCatalog\Model\Entities\Availability;
use EshopCatalog\Model\Entities\AvailabilityText;
use Exception;
use Nette\InvalidArgumentException;
use Nette\Utils\ArrayHash;

class AvailabilityForm extends BaseControl
{
	public ?Availability $availability = null;

	/** @var ?int @persistent */
	public ?int $id = null;

	protected AvailabilityService $availabilityService;

	public function __construct(
		AvailabilityService $availabilityService
	)
	{
		$this->availabilityService = $availabilityService;
	}

	public function render(): void
	{
		$this->template->render($this->getTemplateFile());
	}

	protected function createComponentForm(): BaseForm
	{
		$form = $this->createForm();
		$form->setAjax();

		$form->addText('ident', 'eshopCatalog.availability.ident')
			->setMaxLength(255)
			->addCondition($form::IS_NOT_IN, [Availability::IN_STOCK, Availability::SOLD_OUT])->setRequired();
		$form->addText('name', 'eshopCatalog.availability.name')
			->setIsMultilanguage()
			->setMaxLength(255);
		$form->addTextArea('description', 'eshopCatalog.availability.description')
			->setIsMultilanguage();
		$form->addBool('canShowOnList', 'eshopCatalog.availability.canShowOnList')->setDefaultValue(1);
		$form->addBool('canShowOnSearch', 'eshopCatalog.availability.canShowOnSearch')->setDefaultValue(1);
		$form->addBool('canAddToCart', 'eshopCatalog.availability.canAddToCart')->setDefaultValue(1);

		if (Config::load('productsList.showAvailabilityFilter')) {
			$form->addBool('showInFilter', 'eshopCatalog.availability.showInFilter')->setDefaultValue(1);
		}

		if (Config::load('allowWarehouseOverdraft')) {
			$form->addBool('warehouseOverdraft', 'eshopCatalog.availability.warehouseOverdraft')
				->setDescription('eshopCatalog.availability.warehouseOverdraftDescription')
				->setDefaultValue(0);
		}

		$form->addInteger('delay', 'eshopCatalog.availability.delay')->setRequired();
		$form->addColorInput('color', 'eshopCatalog.availability.color')->setDefaultValue('#ffffff');
		$form->addColorInput('bgColor', 'eshopCatalog.availability.bgColor')->setDefaultValue('#2AABE2');
		$form->addSelect('openGraphText', 'eshopCatalog.availability.openGraphText', [
			''           => '',
			'inStock'    => 'eshopCatalog.availability.openGraphList.inStock',
			'outOfStock' => 'eshopCatalog.availability.openGraphList.outOfStock',
			'preorder'   => 'eshopCatalog.availability.openGraphList.preorder',
			'backorder'  => 'eshopCatalog.availability.openGraphList.backorder',
		]);

		$form->addSaveCancelControl();

		$form->onValidate[] = [$this, 'formValidate'];
		$form->onSuccess[]  = [$this, 'formSuccess'];

		return $form;
	}

	public function formValidate(BaseForm $form, ArrayHash $values): void
	{
		if ($values->ident) {
			/** @var ?Availability $exist */
			$exist = $this->availabilityService->getEr()->findOneBy(['ident' => $values->ident]);

			if ($exist && $exist->getId() !== $this->id) {
				$form->addError($this->t('eshopCatalog.availability.identExist', ['title' => $values->ident]), false);
			}
		}

		if ($form->hasErrors()) {
			$this->redrawControl('form');
		}
	}

	public function formSuccess(BaseForm $form, ArrayHash $values): bool
	{
		$presenter = $form->getPresenterIfExists();
		if (!$presenter) {
			return false;
		}

		try {
			$langValues = $form->convertMultilangValuesToArray();
			/** @var AvailabilityText[] $texts */
			$texts = [];

			if ($this->id) {
				$availability = $this->availabilityService->get($this->id);
				if (!$availability) {
					return false;
				}

				$texts = $availability->getTexts()->toArray();
			} else {
				$availability = new Availability($values->ident, (int) $values->hours);
			}

			$ident = $values->ident;
			unset($values->ident);

			if (!Config::load('allowWarehouseOverdraft')) {
				unset($values->warehouseOverdraft);
			}

			FormHelper::fillEntityByValues($availability, $values);
			if (!Arrays::contains([Availability::IN_STOCK, Availability::SOLD_OUT], $values->ident)) {
				$availability->setIdent($ident);
			}

			$this->em->persist($availability);

			foreach ($langValues as $l => $v) {
				if (!isset($texts[$l]) && $v['name']) {
					$texts[$l] = new AvailabilityText($availability, $l, $v['name']);
				}

				if ($texts[$l] === null) {
					unset($texts[$l]);
					continue;
				}

				if ($v['name'] === '' || $texts[$l]->name === '') {
					$this->em->remove($texts[$l]);

					unset($texts[$l]);
					continue;
				}

				$texts[$l]->name        = $v['name'];
				$texts[$l]->description = $v['description'];
				$this->em->persist($texts[$l]);
			}

			$availability->setTexts($texts);

			$this->em->persist($availability)->flush();
			$form->addCustomData('availabilityId', $availability->getIdent());
			$presenter->flashMessageSuccess('default.saved');
		} catch (Exception $e) {
			$form->addError($e->getMessage());
			$this->redrawControl('form');

			return false;
		}

		return true;
	}

	public function setAvailability(int $id): void
	{
		$this->id           = $id;
		$this->availability = $availability = $this->availabilityService->get($id);

		if (!$availability) {
			throw new InvalidArgumentException;
		}

		FormHelper::fillContainerByEntity($this['form'], $availability);

		$form = $this['form'];
		$d    = [
			'ident' => $availability->getIdent(),
		];
		foreach ($availability->getTexts() as $l => $text) {
			$d['name'][$l]        = $text->name;
			$d['description'][$l] = $text->description;
		}

		$form->setDefaults($d);

		$formIdent = $form->getComponent('ident');

		if ($formIdent) {
			if ($availability->getIdent() === Availability::IN_STOCK) {
				$form->setDescription($this->t('eshopCatalog.availability.identDescriptions.inStock'));
				$formIdent->setOmitted()
					->setDefaultValue(Availability::IN_STOCK)
					->getControlPrototype()->addClass('disabled');
			} else if ($availability->getIdent() === Availability::SOLD_OUT) {
				$form->setDescription($this->t('eshopCatalog.availability.identDescriptions.soldOut'));
				$formIdent->setOmitted()
					->setDefaultValue(Availability::SOLD_OUT)
					->getControlPrototype()->addClass('disabled');
			}
		}
	}

}
