<?php declare(strict_types = 1);

namespace EshopCatalog\AdminModule\Components\Features;

use Core\Model\Event\CreateFormEvent;
use Core\Model\Event\FormSuccessEvent;
use Core\Model\Event\SetFormDataEvent;
use Core\Model\UI\BaseControl;
use Core\Model\UI\Form\BaseForm;
use EshopCatalog\AdminModule\Model\Features;
use EshopCatalog\AdminModule\Model\FeaturesTypeService;
use EshopCatalog\FrontModule\Model\CacheService;
use EshopCatalog\Model\Config;
use EshopCatalog\Model\Entities\Feature;
use EshopCatalog\Model\Entities\FeatureTexts;
use EshopCatalog\Model\Helpers\GoogleExportEnums;
use EshopCatalog\Model\Helpers\ZboziExportEnums;
use EshopProductsComparison\DI\EshopProductsComparisonExtension;
use Exception;
use Nette\InvalidArgumentException;
use Nette\Utils\ArrayHash;

class FeatureForm extends BaseControl
{
	public ?Feature               $feature      = null;
	public ?FeatureTexts          $featureTexts = null;
	protected Features            $featureServices;
	protected FeaturesTypeService $featuresTypeService;
	protected CacheService        $cacheService;

	public function __construct(
		Features            $features,
		FeaturesTypeService $featuresTypeService,
		CacheService        $cacheService
	)
	{
		$this->featureServices     = $features;
		$this->featuresTypeService = $featuresTypeService;
		$this->cacheService        = $cacheService;
	}

	public function render(): void
	{
		$this->template->thisForm = $this['form'];
		$this->template->render($this->getTemplateFile());
	}

	protected function createComponentForm(): BaseForm
	{
		$form = $this->createForm();

		$form->addText('name', 'eshopCatalog.featureForm.name')
			->setIsMultilanguage()
			->setMaxLength(255);
		$form->addText('internalName', 'eshopCatalog.feature.internalName')
			->setIsMultilanguage()
			->setMaxLength(255);
		$form->addText('unit', 'eshopCatalog.feature.unit')
			->setMaxLength(10)
			->setNullable();
		$form->addBool('isPublished', 'eshopCatalog.featureForm.isPublished')
			->setDefaultValue(1);
		$form->addBool('showInProduct', 'eshopCatalog.defaultGrid.showInProduct')
			->setDefaultValue(1);
		$form->addBool('showInExport', 'eshopCatalog.defaultGrid.showInExport')
			->setDefaultValue(1);
		$form->addBool('useAsFilter', 'eshopCatalog.featureForm.useAsFilter')
			->setDefaultValue(1);
		$form->addBool('useForVariantDiff', 'eshopCatalog.featureForm.useForVariantDiff')
			->setDefaultValue(1);

		$form->addBool('seoNoFollow', 'eshopCatalog.featureForm.seoNoFollow')
			->setDefaultValue(0);
		$form->addCheckbox('seoNoFollowInvalidateValues', 'eshopCatalog.featureForm.seoNoFollowInvalidateValues')
			->setDefaultValue(0);

		if (Config::load('features.allowType')) {
			$form->addSelect('type', 'eshopCatalog.featureForm.type', [
				Feature::TYPE_CHECK => 'eshopCatalog.feature.type.check',
				Feature::TYPE_RANGE => 'eshopCatalog.feature.type.range',
			])
				->setDescription('eshopCatalog.feature.typeChangeWarning');
			$form->addText('decimals', 'eshopCatalog.feature.decimals')
				->setHtmlType('number')
				->setHtmlAttribute('step', 1)
				->setHtmlAttribute('min', 0)
				->setHtmlAttribute('max', 5)
				->setDefaultValue(0);
		}

		$form->addSelect('valueType', 'eshopCatalog.featureForm.valueType', [
			Feature::VALUE_TYPE_TEXT  => 'eshopCatalog.feature.valueType.text',
			Feature::VALUE_TYPE_COLOR => 'eshopCatalog.feature.valueType.color',
		])->setDefaultValue(Feature::VALUE_TYPE_TEXT);

		if (Config::load('features.allowDescription')) {
			$form->addText('productTabTitle', 'eshopCatalog.featureForm.productTabTitle')
				->setIsMultilanguage()
				->setMaxLength(255);
		}

		if (class_exists(EshopProductsComparisonExtension::class)) {
			$form->addBool('exportHeureka', 'eshopCatalog.featureForm.exportHeureka')->setDefaultValue(1);
			$form->addText('heurekaName', 'eshopCatalog.featureForm.heurekaName')
				->setIsMultilanguage()
				->setMaxLength(255);
			$form->addBool('exportZbozi', 'eshopCatalog.featureForm.exportZbozi')->setDefaultValue(1);
			$form->addText('zboziName', 'eshopCatalog.featureForm.zboziName')
				->setIsMultilanguage()
				->setMaxLength(255);
			$form->addSelect('zboziType', 'eshopCatalog.featureForm.zboziType', ['' => null] + ZboziExportEnums::getListForSelect());
			$form->addBool('exportGoogle', 'eshopCatalog.featureForm.exportGoogle')->setDefaultValue(1);
			$form->addText('googleName', 'eshopCatalog.featureForm.googleName')
				->setIsMultilanguage()
				->setMaxLength(255);
			$form->addSelect('googleType', 'eshopCatalog.featureForm.googleType', ['' => null] + GoogleExportEnums::getListForSelect());
			$form->addBool('exportIdealo', 'eshopCatalog.featureForm.exportIdealo')->setDefaultValue(1);
			$form->addText('idealoName', 'eshopCatalog.featureForm.idealoName')
				->setIsMultilanguage()
				->setMaxLength(255);
		}

		$this->eventDispatcher->dispatch(
			new CreateFormEvent($form, $this->getPresenterIfExists() ? $this->template : null),
			FeatureForm::class . '::createForm'
		);

		$form->addSaveCancelControl();

		$form->onSuccess[] = [$this, 'formSuccess'];

		return $form;
	}

	public function formSuccess(BaseForm $form, ArrayHash $values): bool
	{
		$this->em->beginTransaction();
		try {
			$oldType = $this->feature ? $this->feature->type : null;
			$feature = $this->feature ?: new Feature;

			$feature->setIsPublished($values->isPublished);
			$feature->setUseAsFilter($values->useAsFilter);
			$feature->type              = $values->type ?: Feature::TYPE_CHECK;
			$feature->unit              = $values->unit ?: null;
			$feature->decimals          = (int) $values->decimals ?: 0;
			$feature->exportHeureka     = (int) $values->exportHeureka;
			$feature->exportZbozi       = (int) $values->exportZbozi;
			$feature->exportIdealo      = (int) $values->exportIdealo;
			$feature->zboziType         = $values->zboziType ?: null;
			$feature->exportGoogle      = (int) $values->exportGoogle;
			$feature->googleType        = $values->googleType ?: null;
			$feature->useForVariantDiff = (int) $values->useForVariantDiff;
			$feature->showInProduct     = (int) $values->showInProduct;
			$feature->showInExport      = (int) $values->showInExport;
			$feature->valueType         = $values->valueType ?: Feature::VALUE_TYPE_TEXT;
			$feature->seoNoFollow       = (int) $values->seoNoFollow;

			$langValues = $form->convertMultilangValuesToArray();
			$texts      = $feature->getFeatureTexts()->toArray();

			foreach ($langValues as $lang => $vals) {
				if ($vals['name'] == '') {
					if (isset($texts[$lang])) {
						$this->em->remove($texts[$lang]);
						unset($texts[$lang]);
					}
					continue;
				}

				$text               = $texts[$lang] ?? new FeatureTexts($feature, $lang);
				$text->name         = $vals['name'];
				$text->internalName = $vals['internalName'];
				$text->heurekaName  = $vals['heurekaName'];
				$text->idealoName   = $vals['idealoName'];
				$text->zboziName    = $vals['zboziName'];
				$text->googleName   = $vals['googleName'];

				if (Config::load('features.allowDescription')) {
					$text->productTabTitle = $vals['productTabTitle'];
				}

				$this->em->persist($text);
				$texts[$lang] = $text;
			}

			$feature->setFeatureTexts($texts);
			$this->em->persist($feature);

			// Změna typu hodnot
			if (Config::load('features.allowType') && $oldType && $oldType != $this->feature->type) {
				set_time_limit(5 * 60);
				if (!$this->featuresTypeService->changeType($oldType, $this->feature->type, $this->feature->getId())) {
					throw new Exception('Change feature type failed');
				}
			}

			$event                   = new FormSuccessEvent($form, $values,
				$this->getPresenterIfExists() ? $this->template : null, $this->getPresenterIfExists() ? $this->presenter : null);
			$event->custom['entity'] = $feature;
			$this->eventDispatcher->dispatch($event, FeatureForm::class . '::formSuccess');

			$this->em->persist($feature);

			$this->em->flush();
			$this->em->commit();

			if ($values->seoNoFollowInvalidateValues) {
				$this->em->getConnection()->executeQuery("UPDATE eshop_catalog__feature_value SET seo_no_follow = NULL WHERE feature_id = ?", [$feature->getId()]);
			}

			$form->addCustomData('featureId', $feature->getId());
			$this->presenter->flashMessageSuccess('eshopCatalog.featureForm.featureSaved');

			$this->cacheService->clearFeatures();

			return true;
		} catch (Exception $e) {
			if ($this->em->getConnection()->isTransactionActive()) {
				$this->em->rollback();
			}

			$form->addError($e->getMessage());
			$this->redrawControl('formErrors');
		}

		return false;
	}

	/**
	 * @param int|string $id
	 */
	public function setFeature($id): void
	{
		$this->feature = $this->featureServices->get($id);

		if ($this->feature) {
			$form = $this['form'];
			$d    = [
				'isPublished'       => $this->feature->getIsPublished(),
				'useAsFilter'       => $this->feature->getUseAsFilter(),
				'type'              => $this->feature->type,
				'unit'              => $this->feature->unit,
				'decimals'          => $this->feature->decimals,
				'exportHeureka'     => $this->feature->exportHeureka,
				'exportZbozi'       => $this->feature->exportZbozi,
				'exportIdealo'      => $this->feature->exportIdealo,
				'zboziType'         => $this->feature->zboziType,
				'exportGoogle'      => $this->feature->exportGoogle,
				'googleType'        => $this->feature->googleType,
				'useForVariantDiff' => $this->feature->useForVariantDiff,
				'valueType'         => $this->feature->valueType,
				'showInProduct'     => $this->feature->showInProduct,
				'showInExport'      => $this->feature->showInExport,
				'seoNoFollow'       => $this->feature->seoNoFollow,
			];
			$form->setDefaults($d);

			$mTexts = $form->prepareMultilangTextsForForm([
				'name',
				'internalName',
				'productTabTitle',
				'heurekaName',
				'zboziName',
				'idealoName',
				'googleName',
			], $this->feature->getFeatureTexts()->toArray());
			$form->setDefaults($mTexts['texts']);

			$this->eventDispatcher->dispatch(new SetFormDataEvent($form, $this->feature), FeatureForm::class . '::setFeature');
		} else {
			throw new InvalidArgumentException;
		}
	}

}
