<?php declare(strict_types = 1);

namespace EshopCatalog\AdminModule\Components\Features;

use Core\Components\Flashes\Flashes;
use Core\Model\UI\BaseControl;
use Core\Model\UI\DataGrid\BaseDataGrid;
use Core\Model\UI\DataGrid\DataSource\DoctrineDataSource;
use Doctrine\ORM\Query\Expr\Join;
use EshopCatalog\AdminModule\Model\Features;
use EshopCatalog\AdminModule\Model\FeatureValues;
use EshopCatalog\Model\Entities\Feature;
use EshopCatalog\Model\Entities\FeatureProduct;
use Nette\Application\AbortException;
use Nette\Utils\Html;

class FeaturesGrid extends BaseControl
{
	protected Features      $featureServices;
	protected FeatureValues $featureValuesService;

	public function __construct(
		Features      $features,
		FeatureValues $featureValues
	)
	{
		$this->featureServices      = $features;
		$this->featureValuesService = $featureValues;
	}

	public function render(): void
	{
		$this->template->render(__DIR__ . '/FeaturesGrid.latte');
	}

	/*******************************************************************************************************************
	 * ==================  Handle
	 */

	public function handleMergeAll(): void
	{
		$this->featureServices->mergeAll();
		$this->redirect('this');
	}

	public function handleMergeAllValues(): void
	{
		$this->featureValuesService->mergeAll();
		$this->redirect('this');
	}

	public function handleRepairPositions(): void
	{
		$this->featureServices->repairPositions();
		$this->redirect('this');
	}

	public function handleRepairValuesPositions(): void
	{
		$this->featureValuesService->repairPositions();
		$this->redirect('this');
	}

	public function handleSortAlphabeticallyFeatures(): void
	{
		$this->featureServices->sortAlphabetically();
		$this->redirect('this');
	}

	public function handleSortAlphabeticallyFeatureValues(): void
	{
		$this->featureValuesService->sortAlphabetically();
		$this->redirect('this');
	}

	/*******************************************************************************************************************
	 * ==================  Components
	 */

	protected function createComponentGrid(): BaseDataGrid
	{
		$grid = $this->createGrid();
		$grid->setDefaultPerPage(50);
		$grid->setRememberState();

		$qb = $this->featureServices->getEr()->createQueryBuilder('f')->addSelect('ft')
			->join('f.featureTexts', 'ft', 'WITH', 'ft.lang = :lang')
			->setParameter('lang', $this->translator->getLocale())
			->orderBy('f.position');
		$grid->setDataSource($qb);

		$productsWithFilter = [];

		/** @var DoctrineDataSource $ds */
		$ds                 = $grid->getDataSource();
		$ds->onDataLoaded[] = function($items) use (&$productsWithFilter) {
			$ids = array_map(static function($v) { return $v->getId(); }, $items);

			foreach ($this->em->getRepository(FeatureProduct::class)->createQueryBuilder('fp')
				         ->select('count(IDENTITY(fp.product)) as count, IDENTITY(fv.feature) as feature')
				         ->innerJoin('fp.featureValue', 'fv', Join::WITH, 'fv.feature IN (:ids)')
				         ->setParameter('ids', $ids)
				         ->groupBy('fv.feature')->getQuery()->getScalarResult() as $row) {
				$productsWithFilter[$row['feature']] = (int) $row['count'];
			}
		};

		$grid->setSortable();
		$grid->setSortableHandler('featuresGrid:gridSortableRow!');

		// Columns
		$grid->addColumnText('name', 'eshopCatalog.defaultGrid.name')->setRenderer(function(Feature $row) {
			$texts = $row->getFeatureText($this->translator->getLocale());

			return $texts ? Html::el()
				->addHtml(Html::el('a', ['href' => $this->presenter->link('VariantsFeatures:editFeature', $row->getId())])
					->setText($texts->name))
				->addHtml(Html::el('div')->addHtml(Html::el('small')->setText($texts->internalName)))
				: '';
		});
		$grid->addColumnStatus('showInProduct', 'eshopCatalog.defaultGrid.showInProductShort')->setAlign('center')
			->addOption(1, 'eshopCatalog.defaultGrid.enabled')
			->setIcon('check')
			->setClass('btn-success')
			->setShowTitle(false)
			->endOption()
			->addOption(0, 'eshopCatalog.defaultGrid.unabled')
			->setIcon('times')
			->setClass('btn-danger')
			->setShowTitle(false)
			->endOption()
			->onChange[] = [$this, 'gridShowInProductChange'];
		$grid->addColumnStatus('showInExport', 'eshopCatalog.defaultGrid.showInExportShort')->setAlign('center')
			->addOption(1, 'eshopCatalog.defaultGrid.enabled')
			->setIcon('check')
			->setClass('btn-success')
			->setShowTitle(false)
			->endOption()
			->addOption(0, 'eshopCatalog.defaultGrid.unabled')
			->setIcon('times')
			->setClass('btn-danger')
			->setShowTitle(false)
			->endOption()
			->onChange[] = [$this, 'gridShowInExportChange'];
		$grid->addColumnStatus('useAsFilter', 'eshopCatalog.defaultGrid.useAsFilter')->setAlign('center')
			->addOption(1, 'eshopCatalog.defaultGrid.enabled')->setIcon('check')->setClass('btn-success')->setShowTitle(false)->endOption()
			->addOption(0, 'eshopCatalog.defaultGrid.unabled')->setIcon('times')->setClass('btn-danger')->setShowTitle(false)->endOption()
			->onChange[] = [$this, 'gridEnabledFilterChange'];
		$grid->addColumnStatus('isPublished', 'default.isPublished')->setAlign('center')
			->addOption(1, 'default.publish')->setIcon('check')->setClass('btn-success')->setShowTitle(false)->endOption()
			->addOption(0, 'default.unPublish')->setIcon('times')->setClass('btn-danger')->setShowTitle(false)->endOption()
			->onChange[] = [$this, 'gridPublishChange'];

		// Filter
		$grid->addFilterText('name', '', ['ft.name', 'ft.internalName']);
		$grid->addFilterSelect('showInProduct', '', [
			''  => '',
			'1' => $this->t('default.yes'),
			'0' => $this->t('default.no'),
		]);
		$grid->addFilterSelect('showInExport', '', [
			''  => '',
			'1' => $this->t('default.yes'),
			'0' => $this->t('default.no'),
		]);
		$grid->addFilterSelect('useAsFilter', '', [
			''  => '',
			'1' => $this->t('default.yes'),
			'0' => $this->t('default.no'),
		]);
		$grid->addFilterSelect('isPublished', '', [
			''  => '',
			'1' => $this->t('default.yes'),
			'0' => $this->t('default.no'),
		]);

		// Actions
		$grid->addAction('edit', '', 'VariantsFeatures:editFeature')->setIcon('edit')->setBsType('primary');
		$grid->addAction('delete', '', 'delete!')
			->setIcon('times')->setBsType('danger')->addClass('ajax')
			->setConfirm(function(Feature $feature) use (&$productsWithFilter) {
				$c = $productsWithFilter[$feature->getId()];

				return $this->t('eshopCatalog.featuresGrid.' . ($c ? 'reallyDeleteFeatureWithProducts' : 'reallyDeleteFeature'), [
					'feature' => $feature->getFeatureText()->name,
					'prod'    => $c ? number_format($c, 0, '', ' ') : '',
				]);
			});

		// Prototype
		$grid->getColumn('showInProduct')->getElementPrototype('th')->addClass('w1nw');
		$grid->getColumn('showInExport')->getElementPrototype('th')->addClass('w1nw');
		$grid->getColumn('useAsFilter')->getElementPrototype('th')->addClass('w1nw');
		$grid->getColumn('isPublished')->getElementPrototype('td')->addClass('w1nw');

		return $grid;
	}

	/*******************************************************************************************************************
	 * ==================  Handle
	 */

	/**
	 * @param int|string $id
	 *
	 * @throws AbortException
	 */
	public function handleDelete($id): void
	{
		$presenter = $this->presenter;
		if ($this->featureServices->remove($id)) {
			$presenter->flashMessage('eshopCatalog.defaultGrid.removed', Flashes::FLASH_SUCCESS);
		} else {
			$presenter->flashMessage('eshopCatalog.defaultGrid.removeFailed', Flashes::FLASH_DANGER);
		}

		if ($presenter->isAjax()) {
			$this['grid']->reload();
			$presenter->redrawControl('flashes');
		} else {
			$presenter->redirect('this');
		}
	}

	/*******************************************************************************************************************
	 * =================  Grid function
	 */

	/**
	 * @param int      $id
	 * @param int|bool $newStatus
	 *
	 * @throws AbortException
	 */
	public function gridPublishChange($id, $newStatus): void
	{
		$presenter = $this->presenter;

		if ($this->featureServices->setPublish($id, $newStatus)) {
			$presenter->flashMessageSuccess('eshopCatalog.defaultGrid.publishChanged');
		} else {
			$presenter->flashMessageDanger('eshopCatalog.defaultGrid.publishChangeFailed');
		}

		if ($presenter->isAjax()) {
			$this['grid']->redrawItem($id);
			$presenter->redrawControl('flashes');
		} else {
			$presenter->redirect('this');
		}
	}

	/**
	 * @param int      $id
	 * @param int|bool $newStatus
	 *
	 * @throws AbortException
	 */
	public function gridEnabledFilterChange($id, $newStatus): void
	{
		$presenter = $this->presenter;

		if ($this->featureServices->setUseAsFilter($id, $newStatus)) {
			$presenter->flashMessageSuccess('eshopCatalog.defaultGrid.filterChanged');
		} else {
			$presenter->flashMessageDanger('eshopCatalog.defaultGrid.filterChangeFailed');
		}

		if ($presenter->isAjax()) {
			$this['grid']->redrawItem($id);
			$presenter->redrawControl('flashes');
		} else {
			$presenter->redirect('this');
		}
	}

	public function gridShowInProductChange(string $id, string $newStatus): void
	{
		if ($this->featureServices->setShowInProduct((int) $id, (int) $newStatus)) {
			$this->presenter->flashMessageSuccess('default.saved');
		} else {
			$this->presenter->flashMessageDanger('default.error');
		}

		$this['grid']->redrawItem($id);
		$this->presenter->redrawControl('flashes');
	}

	public function gridShowInExportChange(string $id, string $newStatus): void
	{
		if ($this->featureServices->setShowInExport((int) $id, (int) $newStatus)) {
			$this->presenter->flashMessageSuccess('default.saved');
		} else {
			$this->presenter->flashMessageDanger('default.error');
		}

		$this['grid']->redrawItem($id);
		$this->presenter->redrawControl('flashes');
	}

	public function handleGridSortableRow(): void
	{
		$presenter = $this->presenter;
		$request   = $presenter->getHttpRequest();
		$id        = $request->getPost('id');
		$position  = $request->getPost('position');

		if ($id != null && $position != null && $this->featureServices->setPosition($id, $position)) {
			$presenter->flashMessageSuccess('default.positionChanged');
		} else {
			$presenter->flashMessageDanger('default.positionChangeFailed');
		}

		if ($presenter->isAjax()) {
			$presenter->redrawControl('flashes');
			$this['grid']->reload();
		} else {
			$this->redirect('this');
		}
	}

}
