<?php declare(strict_types = 1);

namespace EshopCatalog\AdminModule\Components\Manufacturers;

use Core\Model\UI\BaseControl;
use Core\Model\UI\DataGrid\BaseDataGrid;
use Core\Model\UI\DataGrid\DataSource\DoctrineDataSource;
use EshopCatalog\AdminModule\Model\AvailabilityService;
use EshopCatalog\AdminModule\Model\Manufacturers;
use EshopCatalog\Model\Entities\Availability;
use EshopCatalog\Model\Entities\Manufacturer;
use Nette\Application\AbortException;
use Nette\Utils\Html;

class ManufacturersGrid extends BaseControl
{
	protected Manufacturers       $manufacturerServices;
	protected AvailabilityService $availabilityService;

	public function __construct(
		Manufacturers       $manufacturers,
		AvailabilityService $availabilityService
	)
	{
		$this->manufacturerServices = $manufacturers;
		$this->availabilityService  = $availabilityService;
	}

	public function render(): void
	{
		$this->template->render($this->getTemplateFile());
	}

	/*******************************************************************************************************************
	 * ==================  Handle
	 */

	/**
	 * @param int|string $id
	 *
	 * @throws AbortException
	 */
	public function handleDelete($id): void
	{
		$presenter = $this->presenter;
		if ($this->manufacturerServices->remove($id)) {
			$presenter->flashMessageSuccess('eshopCatalog.defaultGrid.removed');
		} else {
			$presenter->flashMessageDanger('eshopCatalog.defaultGrid.removeFailed');
		}

		if ($presenter->isAjax()) {
			$this['grid']->reload();
			$presenter->redrawControl('flashes');
		} else {
			$presenter->redirect('this');
		}
	}

	public function handleMergeAll(): void
	{
		$this->manufacturerServices->mergeAll();
		$this->redirect('this');
	}

	public function handleRepairPositions(): void
	{
		$this->manufacturerServices->repairPositions();
		$this->redirect('this');
	}

	public function handleSortAlphabetically(): void
	{
		$this->manufacturerServices->sortAlphabetically();
		$this->redirect('this');
	}

	/*******************************************************************************************************************
	 * ==================  Components
	 */

	protected function createComponentGrid(): BaseDataGrid
	{
		$grid = $this->createGrid();
		$grid->setDefaultPerPage(50);
		$grid->setRememberState();

		$soldOut   = $this->availabilityService->getAllByIdent()[Availability::SOLD_OUT] ?? null;
		$soldOutId = $soldOut ? $soldOut->getId() : null;

		$products = [];

		$qb = $this->manufacturerServices->getEr()->createQueryBuilder('m')->addSelect('mt')
			->leftJoin('m.texts', 'mt', 'WITH', 'mt.lang = :lang')
			->setParameter('lang', $this->translator->getLocale())
			->orderBy('m.position');

		$dataSource = new DoctrineDataSource($qb, 'm.id');

		$dataSource->onDataLoaded[] = function(array $items) use (&$products, $soldOutId) {
			$ids = [];

			/** @var Manufacturer[] $items */
			foreach ($items as $item) {
				$ids[] = $item->getId();
			}

			if (!empty($ids)) {
				foreach ($this->em->getConnection()->fetchAllAssociative("SELECT id, id_manufacturer as manu, id_availability as av
							FROM eshop_catalog__product
							WHERE is_deleted = 0
							AND id_manufacturer IN (" . implode(',', $ids) . ")") as $row) {
					$isSoldOut = (int) $row['av'] === $soldOutId;

					if (!isset($products[$row['manu']])) {
						$products[$row['manu']] = [
							'soldOut' => [],
							'inStock' => [],
						];
					}

					$products[$row['manu']][$isSoldOut ? 'soldOut' : 'inStock'][] = $row['id'];
				}
			}
		};

		$grid->setDataSource($dataSource);

		$grid->setSortable();
		$grid->setSortableHandler('manufacturersGrid:gridSortableRow!');

		// Columns
		$grid->addColumnLink('name', 'eshopCatalog.defaultGrid.name', 'ManufacturersSuppliers:editManufacturer');
		$grid->addColumnText('products', 'eshopCatalog.manufacturer.products')->setRenderer(function(Manufacturer $row) use (&$products): Html {
			$inStock = count($products[$row->getId()]['inStock'] ?? []);
			$soldOut = count($products[$row->getId()]['soldOut'] ?? []);

			$inStockEl = Html::el('span', [
				'class' => 'btn btn-xs ' . ($inStock > 0 ? 'btn-success' : 'btn-danger'),
			])->setText(number_format($inStock, 0, ',', ' '));
			$soldOutEl = Html::el('span', [
				'class' => 'btn btn-xs btn-danger',
			])->setText(number_format($soldOut, 0, ',', ' '));

			$separatorEl = Html::el('span', [
				'style' => 'padding-left: 4px',
			])->setText('/ ');

			return Html::el()
				->addHtml($inStockEl)
				->addHtml($separatorEl)
				->addHtml($soldOutEl);
		})->setAlign('center');
		$grid->addColumnStatus('isPublished', 'eshopCatalog.defaultGrid.isPublished')->setAlign('center')
			->addOption(1, 'eshopCatalog.defaultGrid.publish')->setIcon('check')->setClass('btn-success')->setShowTitle(false)->endOption()
			->addOption(0, 'eshopCatalog.defaultGrid.unPublish')->setIcon('times')->setClass('btn-danger')->setShowTitle(false)->endOption()
			->onChange[] = [$this, 'gridPublishChange'];

		$grid->getColumn('products')->getElementPrototype('td')->addClass('w1nw');

		// Filter
		$grid->addFilterText('name', '');

		// Actions
		$grid->addAction('edit', '', 'ManufacturersSuppliers:editManufacturer')->setIcon('edit')->setBsType('primary');
		$grid->addAction('delete', '', 'delete!')->setIcon('times')->setBsType('danger')->addClass('ajax')->setConfirm('default.reallyDelete');

		return $grid;
	}

	/*******************************************************************************************************************
	 * =================  Grid function
	 */

	/**
	 * @param int      $id
	 * @param int|bool $newStatus
	 *
	 * @throws AbortException
	 */
	public function gridPublishChange($id, $newStatus): void
	{
		$presenter = $this->presenter;

		if ($this->manufacturerServices->setPublish($id, $newStatus)) {
			$presenter->flashMessageSuccess('eshopCatalog.defaultGrid.publishChanged');
		} else {
			$presenter->flashMessageDanger('eshopCatalog.defaultGrid.publishChangeFailed');
		}

		if ($presenter->isAjax()) {
			$this['grid']->redrawItem($id);
			$presenter->redrawControl('flashes');
		} else {
			$presenter->redirect('this');
		}
	}

	public function handleGridSortableRow(): void
	{
		$presenter = $this->presenter;
		$request   = $presenter->getHttpRequest();
		$id        = $request->getPost('id');
		$position  = $request->getPost('position');

		if ($id != null && $position != null && $this->manufacturerServices->setPosition($id, $position)) {
			$presenter->flashMessageSuccess('default.positionChanged');
		} else {
			$presenter->flashMessageDanger('default.positionChangeFailed');
		}

		if ($presenter->isAjax()) {
			$presenter->redrawControl('flashes');
			$this['grid']->reload();
		} else {
			$this->redirect('this');
		}
	}

}
