<?php declare(strict_types = 1);

namespace EshopCatalog\AdminModule\Components\Products;

use Core\Model\UI\BaseControl;
use Core\Model\UI\Form\BaseForm;
use EshopCatalog\AdminModule\Model\Products;
use EshopCatalog\FrontModule\Model\CacheService;
use EshopCatalog\Model\Entities\Product;
use Nette\Caching\Cache;
use Nette\Http\Request;
use Nette\Utils\ArrayHash;

class CreateVariants extends BaseControl
{
	protected Products     $products;
	protected CacheService $cacheService;
	protected Request      $request;

	/** @var Product[]|null */
	protected ?array $cProducts = null;

	public function __construct(
		Products     $products,
		CacheService $cacheService,
		Request      $request
	)
	{
		$this->products     = $products;
		$this->cacheService = $cacheService;
		$this->request      = $request;
	}

	public function render(): void
	{
		$this->template->thisForm = $this['form'];
		$this->template->products = $this->getProducts();
		$this->template->render($this->getTemplateFile());
	}

	public function createComponentForm(): BaseForm
	{
		$form = $this->createForm();

		$prods = [];
		foreach ($this->getProducts() as $p) {
			$prods[$p->getId()] = $p->getText()->name;
		}

		$form->addText('name', 'eshopCatalog.createVariants.name')
			->setIsMultilanguage();
		$form->addRadioList('main', 'eshopCatalog.createVariants.mainProduct', $prods)
			->setRequired();

		$form->addSaveCancelControl();
		$form->getComponent('saveControl')->removeComponent($form->getComponent('saveControl')['save']);

		$form->onSuccess[] = [$this, 'formSuccess'];

		return $form;
	}

	public function formSuccess(BaseForm $form, ArrayHash $values): bool
	{
		try {
			$main = (int) $values->main;


			foreach ($this->getProducts() as $p) {
				if ($main !== $p->getId()) {
					$this->products->setAsVariantFor($p->getId(), $main, $values->name);
					$this->em->clear();
				}
			}

			foreach ($this->getProducts() as $p) {
				$this->products->clearProductCache($p->getId());
			}

			$this->presenter->flashMessageSuccess('default.saved');

			$this->cacheService->productCache->clean([
				Cache::Tags => ['variants'],
			]);

			return true;
		} catch (\Exception $e) {
			$form->addError($e->getMessage());

			$this->redrawControl('form');
		}

		return false;
	}

	protected function getProducts(): array
	{
		if ($this->cProducts === null) {
			$this->cProducts = [];

			$ids = explode('-', $this->request->getQuery('ids'));

			foreach ($this->products->getEr()->findBy(['id' => $ids]) as $p) {
				/** @var Product $p */
				$this->cProducts[$p->getId()] = $p;
			}
		}

		return $this->cProducts;
	}
}
