<?php declare(strict_types = 1);

namespace EshopCatalog\AdminModule\Components\Tags;

use Core\Components\Flashes\Flashes;
use Core\Model\UI\BaseControl;
use Core\Model\UI\DataGrid\BaseDataGrid;
use Nette\Utils\Html;
use EshopCatalog\Model\Entities\Tag;
use EshopCatalog\AdminModule\Model\Tags;

class TagsGrid extends BaseControl
{
	protected Tags            $tagServices;
	protected ITagFormFactory $tagFormFactory;

	public function __construct(
		Tags            $tags,
		ITagFormFactory $tagFormFactory
	)
	{
		$this->tagServices    = $tags;
		$this->tagFormFactory = $tagFormFactory;
	}

	public function render(): void
	{
		$this->template->render($this->getTemplateFile());
	}

	/*******************************************************************************************************************
	 * ==================  Handle
	 */

	public function handleEdit(int $id): void
	{
		$presenter = $this->presenter;
		$tag       = $this->tagServices->get($id);

		if (!$tag) {
			$presenter->error();
		}

		$this['tagForm']->setTag($id);
		$this->template->modalTitle = $this->t('eshopCatalog.title.editTag',
			['tag' => $tag->getText() ? $tag->getText()->name : '']);
		$this->template->modal      = 'tagForm';
		$this->redrawControl('modal');
	}

	public function handleDelete(int $id): void
	{
		$presenter = $this->presenter;
		if ($this->tagServices->remove($id)) {
			$presenter->flashMessage('eshopCatalog.defaultGrid.removed', Flashes::FLASH_SUCCESS);
		} else {
			$presenter->flashMessage('eshopCatalog.defaultGrid.removeFailed', Flashes::FLASH_DANGER);
		}

		if ($presenter->isAjax()) {
			$this['grid']->reload();
			$presenter->redrawControl('flashes');
		} else {
			$presenter->redirect('this');
		}
	}

	/*******************************************************************************************************************
	 * ==================  Components
	 */

	protected function createComponentGrid(): BaseDataGrid
	{
		$grid = $this->createGrid();
		$grid->setDefaultPerPage(20);

		$qb = $this->tagServices->getEr()->createQueryBuilder('t')
			->leftJoin('t.texts', 'tt', 'WITH', 'tt.lang = :lang')
			->setParameter('lang', $this->translator->getLocale())
			->getQuery()->getResult();

		$grid->setDataSource($qb);

		// Columns
		$grid->addColumnText('type', 'eshopCatalog.defaultGrid.type');
		$grid->addColumnLink('name', 'eshopCatalog.defaultGrid.name', 'Products:editTag')->setRenderer(function(Tag $row) {
			return Html::el('span', [
				'class' => 'btn btn-xs',
				'style' => sprintf('color: %s; background-color: %s; cursor: default;', $row->color, $row->bgColor),
			])->setText($row->getText($this->translator->getLocale())->name);
		});

		// Actions
		$grid->addAction('edit', '', 'edit!')->setIcon('edit')->setBsType('primary')->addClass('ajax');
		$grid->addAction('delete', '', 'delete!')->setIcon('times')->setBsType('danger')
			->setConfirm('default.reallyDelete')
			->addClass('ajax');

		return $grid;
	}

	protected function createComponentTagForm(): TagForm
	{
		$control = $this->tagFormFactory->create();

		if ($this->getParameter('id')) {
			$control->setTag((int) $this->getParameter('id'));
		}

		$control['form']->onSuccessSave[]         = function() {
			$this->presenter->redrawControl('flashes');
			$this['grid']->reload();
		};
		$control['form']->onSuccessSaveAndClose[] = function() {
			$this->presenter->payload->hideModal = true;
			$this->presenter->redrawControl('flashes');
			$this['grid']->reload();
		};
		$control['form']->getComponent('saveControl')->closeModalOnCancel();

		return $control;
	}

}
