<?php declare(strict_types = 1);

namespace EshopCatalog\AdminModule\Model\FormContainers;

use Contributte\Translation\Translator;
use Core\Model\Entities\EntityManagerDecorator;
use Core\Model\UI\AbstractPresenter;
use Core\Model\UI\Form\BaseContainer;
use EshopCatalog\AdminModule\Components\Products\ProductForm;
use EshopCatalog\AdminModule\Model\Products;
use EshopCatalog\AdminModule\Model\Repository\RelatedGroups;
use EshopCatalog\Model\Entities\Product;
use EshopCatalog\Model\Entities\RelatedProduct;
use Nette\Application\IPresenter;

class RelatedProductContainer
{
	protected Translator             $translator;
	protected EntityManagerDecorator $em;
	protected Products               $products;
	protected RelatedGroups          $relatedGroups;

	public function __construct(
		Translator             $translator,
		EntityManagerDecorator $em,
		Products               $products,
		RelatedGroups          $relatedGroups
	)
	{
		$this->translator    = $translator;
		$this->em            = $em;
		$this->products      = $products;
		$this->relatedGroups = $relatedGroups;
	}

	public function getContainer(bool $multilang = false): BaseContainer
	{
		$container = new BaseContainer();
		$container->addText('product', 'eshopCatalog.productForm.findProducts')
			->setHtmlAttribute('data-autocomplete-name', 'eshopCatalogRelatedProducts')
			->setHtmlAttribute('data-autocomplete-keys', 'id,code1,ean,name');
		$container->addCustomData('template', __DIR__ . '/RelatedProductContainer.latte');
		$container->addCustomData('groups', $this->relatedGroups->getOptionsForSelect());

		$container->monitor(ProductForm::class, function(ProductForm $control) use ($container) {
			$product = $control->product;
			$related = [];

			$control->monitor(IPresenter::class, function(AbstractPresenter $presenter) use ($container) {
				$container->getComponent('product')
					->setHtmlAttribute('data-autocomplete-url', $presenter->presenter->link(':EshopCatalog:Cron:Products:loadAll'));
			});

			if ($product) {
				foreach ($this->em->createQueryBuilder()
					         ->select('rp, p, pt')
					         ->from(RelatedProduct::class, 'rp')
					         ->innerJoin('rp.product', 'p')
					         ->innerJoin('p.productTexts', 'pt')
					         ->where('rp.origin = :prod')
					         ->setParameter('prod', $product->getId())
					         ->getQuery()->getResult() as $row) {
					/** @var RelatedProduct $row */
					$related[$row->getProduct()->getId()] = [
						'name'  => $row->getProduct()->getText()->name,
						'group' => $row->getGroup()->getId(),
					];
				}
			}

			$container->addCustomData('related', $related);
		});

		return $container;
	}

	/**
	 * @param Product[] $relatedProducts
	 */
	public function setDefaults(BaseContainer $container, array $relatedProducts): void
	{
	}

	public function saveData(array $data, Product $product): void
	{
		$data           = $data['relatedProductsIds'];
		$currentRelated = $product->getRelatedProducts()->toArray();

		unset($data['__id__']);

		foreach (array_diff_key($currentRelated, $data) as $k => $entity) {
			/** @var RelatedProduct $entity */
			$this->em->getConnection()->executeStatement('DELETE FROM eshop_catalog__related_product WHERE id = ?', [$entity->getId()]);
			unset($currentRelated[$k]);
		}

		foreach (array_diff_key($data, $currentRelated) as $prodId => $vals) {
			if (!is_numeric($prodId)) {
				continue;
			}

			$group   = $this->relatedGroups->getReference($vals['group']);
			$related = new RelatedProduct($product, $this->products->getReference($prodId), $group);

			$this->em->persist($related);
		}

		foreach ($currentRelated as $k => $entity) {
			$dataRow = $data[$k] ?? null;

			if (!$dataRow) {
				continue;
			}

			if ((int) $dataRow['group'] !== $entity->getGroup()->getId()) {
				$entity->setGroup($this->relatedGroups->getReference((int) $dataRow['group']));
				$this->em->persist($entity);
			}
		}
	}

}
