<?php declare(strict_types = 1);

namespace EshopCatalog\ApiModule\Api\V1\Controllers;

use Apitte\Core\Annotation\Controller\Method;
use Apitte\Core\Annotation\Controller\Path;
use Apitte\Core\Annotation\Controller\RequestParameters;
use Apitte\Core\Annotation\Controller\RequestParameter;
use Apitte\Core\Http\ApiRequest;
use Apitte\Core\Http\ApiResponse;
use Contributte\Translation\Translator;
use Core\Model\Entities\EntityManagerDecorator;
use Core\Model\Entities\ExtraField;
use Core\Model\Entities\QueryBuilder;
use Core\Model\Parameters;
use Doctrine\ORM\AbstractQuery;
use Doctrine\ORM\Query;
use EshopCatalog\Model\Entities\PackageItem;
use EshopCatalog\FrontModule\Model\Dao\Product;
use EshopCatalog\FrontModule\Model\Products;
use EshopCatalog\FrontModule\Model\ProductsFacade;
use EshopCatalog\AdminModule\Model\Products as AdminProducts;
use EshopCatalog\Model\Packages;
use Nette\Application\LinkGenerator;
use Nette\Utils\Arrays;
use Nette\Utils\Validators;
use Throwable;

/**
 * @Path("/product")
 */
class ProductsController extends BaseController
{
	protected Products               $productsDaoService;
	protected ProductsFacade         $productsFacade;
	protected AdminProducts          $products;
	protected Translator             $translator;
	protected LinkGenerator          $linkGenerator;
	protected EntityManagerDecorator $em;
	protected Packages               $packages;

	public function __construct(Products $productsDaoService, ProductsFacade $productsFacade, AdminProducts $products, Translator $translator, LinkGenerator $linkGenerator, EntityManagerDecorator $em,
								Packages $packages)
	{
		$this->productsDaoService = $productsDaoService;
		$this->productsFacade     = $productsFacade;
		$this->products           = $products;
		$this->translator         = $translator;
		$this->linkGenerator      = $linkGenerator;
		$this->em                 = $em;
		$this->packages           = $packages;

		ProductsFacade::setMode(ProductsFacade::MODE_CHECKOUT);
	}

	/**
	 * @Path("/")
	 * @Method("GET")
	 * @RequestParameters({
	 *      @RequestParameter(name="start", type="int", description="start", required=false, in="query"),
	 *      @RequestParameter(name="limit", type="int", description="limit", required=false, in="query"),
	 *      @RequestParameter(name="filter", type="string", description="filter", required=false, in="query")
	 * })
	 */
	public function index(ApiRequest $request, ApiResponse $response): array
	{
		$offset = $request->getQueryParam('offset', 0);
		$limit  = $request->getQueryParam('limit', 0);
		$sort   = $request->getQueryParam('sort', 'ASC');

		$productIds = $this->productsDaoService->getProductsIdAll($offset, $limit, $sort);

		return $productIds ? array_values($this->productsFacade->getProducts($productIds)) : [];
	}

	/**
	 * @return Product[]
	 * @throws Throwable
	 * @Path("/get/{id}")
	 * @Method("GET")
	 * @RequestParameters({
	 *      @RequestParameter(name="id", type="int")
	 * })
	 */
	public function product(ApiRequest $request, ApiResponse $response): array
	{
		$id      = $request->getParameter('id');
		$product = $this->productsFacade->getProduct($id);

		if (!$product) {
			$this->sendError($response, "Product with this id doesn´t exist");
		}

		return [$product];
	}

	/**
	 * @Path("/search-all")
	 * @Method("GET")
	 * @RequestParameters({
	 *      @RequestParameter(name="userId", type="int", description="userId", required=false, in="query"),
	 *      @RequestParameter(name="enableListing", type="bool", required=false, in="query"),
	 * })
	 */
	public function searchAll(ApiRequest $request, ApiResponse $response): ApiResponse
	{
		$userId = $request->getQueryParam('userId', null);
		$enableListing = $request->getQueryParam('enableListing', null);

		$qb = $this->prepareSearchQueryBuilder(!$enableListing);
		$result = $qb->getQuery()->getArrayResult();

		if (class_exists('EshopCheckout\DI\EshopCheckoutExtension') && Parameters::load('eshopCheckout.allowPriceLevels', false) && !is_null($userId)) {
			$this->loadPriceLevels($result, (int) $userId);
		}

		$this->loadPackages($result, !$enableListing, !is_null($userId) ? ((int) $userId) : null);

		return $response->writeJsonBody((array) $result);
	}

	/**
	 * @Path("/searchByCode/{code}")
	 * @Method("GET")
	 * @RequestParameters({
	 *      @RequestParameter(name="code", type="string", required=true),
	 *      @RequestParameter(name="userId", type="int", description="userId", required=false, in="query"),
	 *      @RequestParameter(name="enableListing", type="bool", required=false, in="query")
	 * })
	 */
	public function searchByCode(ApiRequest $request, ApiResponse $response): ApiResponse
	{
		$query  = urldecode($request->getParameter('code'));
		$userId = $request->getQueryParam('userId', null);
		$enableListing = $request->getQueryParam('enableListing', null);
		$qb = $this->prepareSearchQueryBuilder(!$enableListing);
		$qb->andWhere('p.ean = :ean')
		   ->setParameter('ean', $query);

		$result = $qb->getQuery()->setMaxResults(1)->getOneOrNullResult(AbstractQuery::HYDRATE_ARRAY);

		if (!$result) {
			return $response->writeJsonBody(['status' => 'error', 'data' => null]);
		}

		$result['link'] = '#';

		if (!Validators::isNone($result['coverPath']) && !Validators::isNone($result['coverFilename'])) {
			$result['path']     = $result['coverPath'];
			$result['filename'] = $result['coverFilename'];
		}
		unset($result['coverPath'], $result['coverFilename']);

		$rows = [$result];
		if (class_exists('EshopCheckout\DI\EshopCheckoutExtension') && Parameters::load('eshopCheckout.allowPriceLevels', false) && !is_null($userId)) {
			$this->loadPriceLevels($rows, (int) $userId);
		}

		$this->loadPackages($rows, !$enableListing, !is_null($userId) ? ((int) $userId) : null);

		return $response->writeJsonBody(Arrays::first((array) $rows));
	}

	/**
	 * @Path("/search/{query}")
	 * @Method("GET")
	 * @RequestParameters({
	 *      @RequestParameter(name="query", type="string", required=true),
	 *      @RequestParameter(name="userId", type="int", description="userId", required=false, in="query"),
	 *      @RequestParameter(name="start", type="int", description="start", required=false, in="query"),
	 *      @RequestParameter(name="limit", type="int", description="limit", required=false, in="query"),
	 *      @RequestParameter(name="filter", type="string", description="filter", required=false, in="query"),
	 *      @RequestParameter(name="sort", type="string", description="sort", required=false, in="query"),
	 *      @RequestParameter(name="enableListing", type="bool", required=false, in="query")
	 * })
	 */
	public function search(ApiRequest $request, ApiResponse $response): array
	{
		$query  = urldecode($request->getParameter('query'));
		$offset = $request->getQueryParam('offset', 0);
		$limit  = $request->getQueryParam('limit', null);
		$filter = $request->getQueryParam('filter', null);
		$sort   = $request->getQueryParam('sort', null);
		$userId = $request->getQueryParam('userId', null);
		$enableListing = $request->getQueryParam('enableListing', null);

		$qb = $this->prepareSearchQueryBuilder(!$enableListing);

		if ($query) {
			$qb->andWhere('pt.name LIKE :term OR p.code1 LIKE :term OR p.code2 LIKE :term OR p.ean LIKE :term OR p.id LIKE :term')
				->setParameter('term', '%' . $query . '%');
		}

		$result = $qb->getQuery()->getArrayResult();
		foreach ($result as &$row) {
			$link = '#';

			$row['link'] = $link;

			if (!Validators::isNone($row['coverPath']) && !Validators::isNone($row['coverFilename'])) {
				$row['path']     = $row['coverPath'];
				$row['filename'] = $row['coverFilename'];
			}
			unset($row['coverPath'], $row['coverFilename']);
		}

		if (class_exists('EshopCheckout\DI\EshopCheckoutExtension') && Parameters::load('eshopCheckout.allowPriceLevels', false) && !is_null($userId)) {
			$this->loadPriceLevels($result, (int) $userId);
		}

		$this->loadPackages($result, !$enableListing, !is_null($userId) ? ((int) $userId) : null);

		return $result;
	}

	protected function prepareSearchQueryBuilder(bool $disableListing = true): QueryBuilder
	{
		// TODO do modelu
		/** @var QueryBuilder $qb */
		$qb = $this->productsDaoService->getEr()->createQueryBuilder('p', 'p.id')
			->select('p.id, IDENTITY(p.package) package, pt.name, p.price, p.code1, p.code2, p.ean, p.quantity, p.isPublished, pt.isPublished, vat.rate as vatRate, i.path, i.filename, cover.path as coverPath, cover.filename as coverFilename')
			->join('p.productTexts', 'pt', 'WITH', 'pt.lang = :lang')
			->join('p.vatRate', 'vat')
			->leftJoin('p.gallery', 'g')
			->leftJoin('g.images', 'cover', Query\Expr\Join::WITH, 'cover.isCover = 1')
			->leftJoin('g.images', 'i', Query\Expr\Join::WITH, 'i.position = 0')
			->orderBy('pt.name')
			->setParameter('lang', $this->translator->getLocale());

		$qb->andWhere('p.isDeleted = 0');

		if ($disableListing) {
			$qb->andWhere('p.disableListing = 0');
		}

		return $qb;
	}

	protected function loadPackages(array &$products, bool $disableListing = true, int $userId = null): void
	{
		$decorate = function(&$row) {
			$link = '#';

			$row['link'] = $link;

			if (!Validators::isNone($row['coverPath']) && !Validators::isNone($row['coverFilename'])) {
				$row['path']     = $row['coverPath'];
				$row['filename'] = $row['coverFilename'];
			}
			unset($row['coverPath'], $row['coverFilename']);
		};

		$editablePackageIds = $this->packages->getEditablePackageIds();

		foreach ($products as $k => $product) {
			if (in_array((int) $product['id'], $editablePackageIds, true)) { // editovatelne balicky
				$products[$k]['packageId'] = $product['id'];
				$products[$k]['packageParentId'] = null;
				$products[$k]['isEditablePackage'] = true;
				$products[$k]['boxName'] = $this->translator->translate('eshopPackagesFront.editablePackage.box');

			} else { // balicky definovane z adminu a ostatni produkty
				$products[$k]['packageId'] = null;
				$products[$k]['packageParentId'] = null;
				$products[$k]['isEditablePackage'] = false;
				$products[$k]['boxName'] = null;

				if (!$product['package']) {
					continue;
				}

				$products[$k]['packageId'] = $product['package'];

				$qbPackages = $this->em->getRepository(PackageItem::class)->createQueryBuilder('packItem');
				$qbPackages->select('p.id, packItem.id as packageId, packItem.quantity')
						   ->join('packItem.product', 'p')
						   ->andWhere('packItem.package = :package')
						   ->setParameter('package', $product['package']);

				$packageItems = [];
				foreach ($qbPackages->getQuery()->getScalarResult() as $pack) {
					$packageItems[$pack['id']] = $pack;
				}

				if (!$packageItems) {
					continue;
				}

				$qb = $this->prepareSearchQueryBuilder($disableListing);
				$qb->andWhere('p.id IN (:productPackageIds)')
				   ->setParameter('productPackageIds', array_keys($packageItems));

				$result = [];
				foreach ($qb->getQuery()->getArrayResult() as $row) {
					$decorate($row);

					$row['quantity'] = $packageItems[$row['id']]['quantity'];
					$row['price'] = '0';
					$row['packageParentId'] = $products[$k]['packageId'];

					$result[$row['id']] = $row;
				}

				if (class_exists('EshopCheckout\DI\EshopCheckoutExtension') && Parameters::load('eshopCheckout.allowPriceLevels', false) && !is_null($userId)) {
					$this->loadPriceLevels($result, $userId);
				}

				$products[$k]['package'] = $result;
			}
		}
	}

	protected function loadPriceLevels(array &$products, int $userId = null): void
	{
		$daoProducts = [];
		foreach ($products as $product) {
			$dao = new Product($product['id']);
			$dao->setQuantity((int) $product['quantity']);
			$dao->setVatRate($product['vatRate']);
			$dao->setPrice((float) $product['price']);
			$dao->setBasePrice((float) $product['price']);
			$dao->priceInBaseCurrency = $dao->getPrice();
			$dao->basePriceInBaseCurrency = $dao->basePrice;
			$daoProducts[$product['id']]  = $dao;
		}

		$this->productsDaoService->loadPriceLevels($daoProducts, $userId);

		foreach ($products as $k => $v) {
			$product = $daoProducts[$v['id']];
			if (((int) $v['price']) !== 0) {
				$products[$k]['price'] = $product->price;
			}
		}
	}

}
