<?php declare(strict_types = 1);

namespace EshopCatalog\FrontModule\Model\Dao;

use Core\Model\Dao\Country;
use Core\Model\Entities\TSeo;
use Core\Model\Helpers\Strings;
use EshopCatalog\Model\Config;
use EshopCatalog\Model\Entities\Feature;
use EshopGifts\FrontModule\Model\Dao\Gift;
use Gallery\FrontModule\Model\Dao\Album;
use Gallery\FrontModule\Model\Dao\Image;

class Product
{
	use TSeo;

	public int $id;

	public int $quantity         = 0;
	public int $quantityExternal = 0;

	public int  $minimumAmount = 1;
	public ?int $maximumAmount = null;

	public ?string $currency        = null;
	public ?array  $cFeaturesForTab = null;

	/**
	 * Základní cena bez slev
	 *
	 * @var float
	 */
	public $basePrice;

	/** @var float */
	public $price;

	/** @var float|null */
	public $retailPrice;

	/** @var float */
	public $basePriceInBaseCurrency;

	/** @var float */
	public $priceInBaseCurrency;

	/** @var float|null */
	public $retailPriceInBaseCurrency;

	/** @var float */
	public $recyclingFee;

	public ?int          $manufacturerId = null;
	public ?Manufacturer $manufacturer   = null;

	public ?string $name  = null;
	public ?string $name2 = null;

	/**
	 * Základní název pro skupinu variant
	 */
	public ?string $variantName = null;

	/**
	 * Základní obrázek pro skupinu variant
	 */
	public ?Image $variantImage = null;

	public ?string $shortDescription = null;
	public ?string $description      = null;

	public ?\DateTimeInterface $created  = null;
	public ?\DateTimeInterface $modified = null;

	public ?int   $galleryId = null;
	public ?Album $gallery   = null;

	/** @var FeatureProduct[] */
	public array $features = [];

	/** @var int[] */
	public array $featureValuesIds = [];

	/** @var RelatedGroup[] */
	protected array $related = [];

	/** @var self[] */
	public $alternative;

	public ?int      $defaultCategoryId = null;
	public ?Category $defaultCategory   = null;

	/** @var int[] */
	public array $categories = [];

	/** @var Tag[] */
	public array $tags = [];

	/** @var array<int, array{tagId: int, validFrom: \DateTimeInterface|null, validTo: \DateTimeInterface|null}> */
	public array $tagsRaw = [];

	public ?string $link = '#';

	public ?int    $vatRate = null;
	public ?string $ean     = null;
	public ?string $code1   = null;
	public ?string $code2   = null;

	public ?Availability $availability             = null;
	public ?Availability $availabilityAfterSoldOut = null;
	public ?Availability $externalAvailability     = null;

	public ?string $preorderText = null;

	public int $unlimitedQuantity = 0;

	protected array $extraFields          = [];
	public bool     $discountDisabled     = false;
	public bool     $categoryGiftsAllowed = true;
	public bool     $orderGiftsAllowed    = true;

	/** @var Gift[] */
	public array $gifts = [];

	/** @var int[] */
	public array $giftsIds = [];

	public bool $isActive = true;

	public bool $canAddToCart = true;

	public bool $isAssort = false;

	public bool $isDiscount = false;

	/** @var self[] */
	public array $variants = [];

	public ?string $variantId = null;

	public ?string $variantOf = null;

	public array $variantProductIds = [];

	public array $helperData = [];

	public array $moreData = [];

	/** @var FeatureProduct[] */
	protected array $cVariantDifferences = [];

	public string  $condition            = 'new';
	public ?string $conditionDescription = null;

	/** @var Document[] */
	protected array $documents = [];

	/** @var Video[] */
	protected array $videos = [];

	/** @var bool */
	public bool $disablePickUpSpedition = false;

	public bool $disableCalculateFreeSpedition = false;

	/** @var bool */
	public bool $isOversize = false;

	public array $inSites = [];

	public array $disabledSpeditions = [];
	public array $disabledPayments   = [];

	/** @var int|null Package width */
	public ?int $width = null;

	/** @var int|null Package height */
	public ?int $height = null;

	/** @var int|null Package depth */
	public ?int $depth = null;

	/** @var int|null Package weight */
	public ?int $weight = null;

	public ?int     $hsCustomsCode   = null;
	public ?Country $countryOfOrigin = null;

	public ?int     $packageId = null;
	public ?Package $package   = null;
	public int      $verifyAge = 0;

	public function __construct(
		int $id
	)
	{
		$this->id = $id;
	}

	public function getId(): int
	{
		return $this->id;
	}

	public function isInSite(string $siteIdent): bool
	{
		return in_array($siteIdent, $this->inSites);
	}

	/*******
	 * == Quantity
	 */

	public function getQuantityExternal(): int { return $this->quantityExternal; }

	public function getQuantity(): int
	{
		if (!Config::wareHouseExist()) {
			if ($this->getAvailability() && $this->getAvailability()->canAddToCart()) {
				if (Config::load('noWarehouseRealQuantity')) {
					return $this->quantity;
				}

				return 500;
			}

			return 0;
		}

		if (Config::load('pseudoWarehouse') && $this->unlimitedQuantity === 1) {
			return 500;
		}

		return (int) $this->quantity;
	}

	public function setQuantityExternal(int $quantity): self
	{
		$this->quantityExternal = $quantity;

		return $this;
	}

	public function setQuantity(int $quantity): self
	{
		$this->quantity = $quantity;

		return $this;
	}

	public function getQuantityWithExternal(): int
	{
		return $this->getQuantity() + $this->getQuantityExternal();
	}

	/**
	 * Vrátí text pro překlad s počtem ks skladem
	 */
	public function getQuantityText(bool $useExternal = false): string
	{
		$q     = $useExternal ? $this->quantityExternal : $this->getQuantity();
		$limit = Config::load('quantityTextLimit', 15);

		if (Config::load('pseudoWarehouse') && $this->unlimitedQuantity === 1) {
			$text = 'inStock';
		} else if ($q <= 0) {
			$text = 'soldOut';
		} else if ($q <= 4 && $q <= $limit) {
			$text = 'lastInStock1';
		} else if ($q <= 15 && $q <= $limit) {
			$text = 'lastInStock2';
		} else {
			$text = 'inStock';
		}

		return 'eshopCatalogFront.product.' . $text;
	}

	public function getQuantityTextWithExternal(): string
	{
		$q     = $this->quantityExternal + $this->getQuantity();
		$limit = Config::load('quantityTextLimit', 15);

		if (Config::load('pseudoWarehouse') && $this->unlimitedQuantity === 1) {
			$text = 'inStock';
		} else if ($q <= 0) {
			$text = 'soldOut';
		} else if ($q <= 4 && $q <= $limit) {
			$text = 'lastInStock1';
		} else if ($q <= 15 && $q <= $limit) {
			$text = 'lastInStock2';
		} else {
			$text = 'inStock';
		}

		return 'eshopCatalogFront.product.' . $text;
	}

	/**
	 * Vrátí stav počtu ks skladem
	 */
	public function getQuantityColor(bool $useExternal = false): string
	{
		if (Config::load('pseudoWarehouse') && $this->unlimitedQuantity === 1) {
			return 'green';
		}

		$q     = $useExternal ? $this->quantityExternal : $this->getQuantity();
		$limit = Config::load('quantityTextLimit', 15);
		if ($q <= 0) {
			return 'red';
		}

		if ($q <= $limit) {
			return 'yellow';
		}

		return 'green';
	}

	public function getQuantityColorWithExternal(): string
	{
		if (Config::load('pseudoWarehouse') && $this->unlimitedQuantity === 1) {
			return 'green';
		}

		$q     = $this->quantityExternal + $this->getQuantity();
		$limit = Config::load('quantityTextLimit', 15);
		if ($q <= 0) {
			return 'red';
		}

		if ($q <= $limit) {
			return 'yellow';
		}

		return 'green';
	}

	/*******
	 * == Price
	 */

	public function getPrice(): float { return (float) $this->price; }

	public function setPrice(float $price): self
	{
		$this->price = $price;

		return $this;
	}

	public function getBasePrice(): float { return (float) $this->basePrice; }

	public function setBasePrice(float $price): self
	{
		$this->basePrice = $price;

		return $this;
	}

	public function getPriceWithoutVat(): float { return ($this->getPrice() / (1 + ($this->vatRate / 100))); }

	public function getRetailPrice(): float
	{
		return (float) $this->retailPrice;
	}

	public function setRetailPrice(float $retailPrice): void
	{
		$this->retailPrice = $retailPrice;
	}

	public function getRetailPriceWithoutVat(): float { return ($this->getRetailPrice() / (1 + ($this->vatRate / 100))); }

	public function getRecyclingFee(): float
	{
		return (float) $this->recyclingFee;
	}

	public function setRecyclingFee(float $recyclingFee): void
	{
		$this->recyclingFee = $recyclingFee;
	}

	public function getManufacturer(): ?Manufacturer
	{
		return $this->manufacturer;
	}

	/**
	 * @param Manufacturer|null $manufacturer
	 */
	public function setManufacturer($manufacturer): void
	{
		$this->manufacturer = $manufacturer;
	}

	public function getName(): string
	{
		$str = $this->name;

		if (Config::load('addManufacturerBeforeProductName', false) && $this->getManufacturer()) {
			$str = trim($this->getManufacturer()->name . ' ' . $str);
		}

		$variantDiff = $this->getVariantDifferences();
		if (Config::load('addOneVariantDiffToProductName', false) && count($variantDiff) === 1) {
			$value = array_values($variantDiff)[0]->value;

			if (!Strings::endsWith($str, $value)) {
				$str .= ' - ' . $value;
			}
		}

		return $str;
	}

	public function getVariantImage(): ?Image
	{
		return $this->variantImage;
	}

	public function getVariantName(): ?string
	{
		$str = $this->variantName;

		if (!$str) {
			foreach ($this->variants as $var) {
				if ($var->variantName) {
					$str = $var->variantName;
					break;
				}
			}

			if (!$str) {
				return null;
			}
		}

		if (Config::load('addManufacturerBeforeProductName') && $this->getManufacturer()) {
			$str = trim($this->getManufacturer()->name . ' ' . $str);
		}

		return $str;
	}

	public function setName(?string $name = null): void
	{
		$this->name = $name;
	}

	public function setName2(?string $name = null): void
	{
		$this->name2 = $name;
	}

	public function getName2(): ?string { return Config::load('enableProductName2') ? $this->name2 : $this->name; }

	public function getDescription(): ?string { return $this->description; }

	/**
	 * @param string $description
	 */
	public function setDescription($description): void
	{
		$this->description = $description;
	}

	public function getGallery(): ?Album
	{
		if (!$this->gallery && $this->variantOf && $this->variants) {
			$this->gallery = isset($this->variants[$this->variantOf]) ? $this->variants[$this->variantOf]->gallery : null;
		}

		return $this->gallery;
	}

	/**
	 * @param Album $gallery
	 */
	public function setGallery($gallery): void
	{
		$this->gallery = $gallery;
	}

	public function addFeature(FeatureProduct $feature): void
	{
		$this->features[$feature->idFeature] = $feature;
	}

	/**
	 * @param FeatureProduct[] $features
	 */
	public function setFeatures(array $features): void
	{
		$this->features += $features;
	}

	/**
	 * @return FeatureProduct[]
	 */
	public function getFeatures(): array { return $this->features; }

	/**
	 * @return FeatureProduct[]
	 */
	public function getFeaturesForTags(): array
	{
		$arr = [];
		foreach ($this->getFeatures() as $feature) {
			if ($feature->showAsTag) {
				$arr[$feature->idFeatureValue] = $feature;
			}
		}

		return $arr;
	}

	/**
	 * @return FeatureProduct[]
	 */
	public function getFeaturesForTab(): array
	{
		if ($this->cFeaturesForTab === null) {
			$this->cFeaturesForTab = [];

			foreach ($this->getFeatures() as $feature) {
				if ($feature->productTabTitle && ($feature->longDescription || $feature->moreLink)) {
					$this->cFeaturesForTab[$feature->idFeatureValue] = $feature;
				}
			}
		}

		return $this->cFeaturesForTab;
	}

	/**
	 * @return FeatureProduct[]
	 */
	public function getFeaturesById(int $id): array
	{
		$list = [];

		foreach ($this->getFeatures() as $feature) {
			if ($feature->idFeature === $id) {
				$list[$feature->idFeatureValue] = $feature;
			}
		}

		return $list;
	}

	/**
	 * @param int|string $id
	 */
	public function getFeatureById($id): ?FeatureProduct
	{
		foreach ($this->getFeatures() as $feature) {
			if ($feature->idFeature === $id) {
				return $feature;
			}
		}

		return null;
	}

	/**
	 * @return FeatureProduct[]
	 */
	public function getGroupedFeatures(bool $onlyShowInProduct = true): array
	{
		/** @var FeatureProduct[] $result */
		$result = [];

		foreach ($this->getFeatures() as $feature) {
			if ($onlyShowInProduct && !$feature->showInProduct) {
				continue;
			}

			if ($feature->type === Feature::TYPE_RANGE) {
				$feature->value = str_replace(',', '.', (string) $feature->rawValue);
			}

			if (!isset($result[$feature->idFeature])) {
				if ($feature->type === Feature::TYPE_RANGE && $feature->unit) {
					$feature->name .= ' (' . $feature->unit . ')';
				}

				$result[$feature->idFeature] = $feature;
			} else {
				$result[$feature->idFeature]->value = trim($result[$feature->idFeature]->value) . ', ' . trim($feature->value);
			}
		}

		return $result;
	}

	/**
	 * @param string $type
	 */
	public function getTag($type): ?Tag { return $this->tags[$type] ?? null; }

	/**
	 * @param self[] $array
	 */
	public function setAlternatives(array $array): self
	{
		$this->alternative = $array;

		return $this;
	}

	public function getVatRate(): ?int { return $this->vatRate; }

	/**
	 * @param int $vatRate
	 */
	public function setVatRate($vatRate): Product
	{
		$this->vatRate = $vatRate;

		return $this;
	}

	public function getEan(): ?string { return $this->ean; }

	public function setEan(?string $ean): Product
	{
		$this->ean = $ean;

		return $this;
	}

	public function getCode1(): ?string { return $this->code1; }

	public function setCode1(?string $code1): Product
	{
		$this->code1 = $code1;

		return $this;
	}

	public function getCode2(): ?string { return $this->code2; }

	public function setCode2(?string $code2): Product
	{
		$this->code2 = $code2;

		return $this;
	}

	/**
	 * @param float|int|string $key
	 * @param mixed|null       $value
	 */
	public function addExtraField($key, $value): self
	{
		$this->extraFields[$key] = $value;

		return $this;
	}

	public function setExtraFields(array $array): self
	{
		$this->extraFields = $array;

		return $this;
	}

	public function getExtraFields(): array { return $this->extraFields; }

	/**
	 * @param string     $key
	 * @param mixed|null $default
	 *
	 * @return mixed|null
	 */
	public function getExtraField(string $key, $default = null) { return $this->extraFields[$key] ?? $default; }

	/**
	 * @return Gift[]
	 */
	public function getGifts(): array { return $this->gifts; }

	public function getAvailability(): ?Availability { return $this->availability; }

	public function setAvailability(?Availability $availability = null): self
	{
		$this->availability = $availability;

		return $this;
	}

	/**
	 * ====== Variants
	 */

	public function getSortedVariantsByFeature(?int $variantId = null, bool $addSelf = false): array
	{
		if (!$this->features) {
			return $this->variants;
		}

		if ($variantId === null) {
			$variantId = array_values($this->features)[0]->idFeature;
		}

		$tmp = [];
		$not = [];

		$variants = $this->variants;
		if ($addSelf) {
			$variants[$this->getId()] = $this;
		}

		foreach ($variants as $variant) {
			$feature = $variant->getFeatureById($variantId);

			if ($feature) {
				$tmp[$feature->valuePosition][] = $variant;
			} else {
				$not[] = $variant;
			}
		}
		ksort($tmp);

		$tmp2 = [];
		foreach ($tmp as $v) {
			foreach ($v as $v2) {
				$tmp2[$v2->getId()] = $v2;
			}
		}

		return array_merge($tmp2, $not);
	}

	public function hasVariants(): bool { return $this->variants ? true : false; }

	public function getVariantDiffFeatures(): array
	{
		$result = [];

		foreach ($this->variants as $v) {
			foreach ($v->features as $f) {
				if ($f->useForVariantDiff) {
					$result[] = $f->idFeature;
				}
			}
		}

		return array_unique($result);
	}

	public function getVariantDifferences(): array
	{
		if (empty($this->variants)) {
			return [];
		}

		if (empty($this->cVariantDifferences)) {
			$used     = [];
			$features = [];
			foreach ($this->features as $v) {
				if (!$v->useForVariantDiff) {
					continue;
				}

				$features[$v->getKey()] = $v;
			}

			foreach ($this->variants as $v) {
				foreach ($v->features as $f) {
					if (!$f->useForVariantDiff) {
						continue;
					}

					$used[$f->getKey()]++;
				}
			}

			foreach ($used as $k => $v) {
				if ($v === count($this->variants)) {
					unset($features[$k]);
				}
			}

			if (count($features) === 1) {
				$featureId = array_values($features)[0]->idFeature;
				uksort($this->variants, function($a, $b) use ($featureId) {
					$a = isset($this->variants[$a]) ? $this->variants[$a]->getFeatureById($featureId) : null;
					$b = isset($this->variants[$b]) ? $this->variants[$b]->getFeatureById($featureId) : null;

					if (!$a) {
						$a                = new FeatureProduct;
						$a->valuePosition = 999;
					}
					if (!$b) {
						$b                = new FeatureProduct;
						$b->valuePosition = 999;
					}

					return $a->valuePosition <=> $b->valuePosition;
				});
			}

			$this->cVariantDifferences = $features;
		}

		return $this->cVariantDifferences;
	}

	public function getVariantsDiffSteps(): array
	{
		$arr              = [
			'diffs' => [],
			'data'  => [],
		];
		$requiredFeatures = [];
		$showAllSteps     = (bool) Config::load('productDetail.variants.showAllSteps');

		foreach ($this->getVariantDifferences() as $feature) {
			if (!$feature->showInProduct) {
				continue;
			}

			$row = [
				'feature'  => $feature,
				'variants' => [],
			];

			foreach ($this->getSortedVariantsByFeature($feature->idFeature, true) as $variant) {
				$hasCount = 0;

				if (!$showAllSteps) {
					foreach ($requiredFeatures as $requiredFeatureValue => $requiredFeature) {
						foreach ($variant->getFeatures() as $variantFeature) {
							if ($variantFeature->idFeatureValue === $requiredFeatureValue) {
								$hasCount++;
								break;
							}
						}
					}
				}

				if ($showAllSteps || $hasCount === count($requiredFeatures)) {
					/** @var FeatureProduct $variantFeature */
					$variantFeature = array_values($variant->getFeaturesById($feature->idFeature))[0];
					$key            = $variantFeature->idFeatureValue ?: $variantFeature->rawValue;

					if (!$key) {
						continue;
					}

					if (!isset($row['variants'][$key])) {
						$row['variants'][$key]['variant']  = $variant;
						$row['variants'][$key]['variants'] = [];
						$row['variants'][$key]['feature']  = $variantFeature;
					}

					$row['variants'][$key]['variants'][] = $variant;
				}
			}

			$key = $feature->idFeatureValue ?: $feature->rawValue;

			if (count($row['variants']) <= 1) {
				continue;
			}

			$requiredFeatures[$key] = $feature;
			$arr['data'][$key]      = $row;
			$arr['diffs'][$key]     = $feature;
		}

		if (count($requiredFeatures) > 1) {
			foreach ($arr['data'] as &$row) {
				$checkRequireFeatures = $requiredFeatures;
				unset($checkRequireFeatures[$row['feature']->idFeatureValue]);

				foreach ($row['variants'] as &$vals) {
					foreach ($vals['variants'] as $variant) {
						if (count(array_intersect_key($checkRequireFeatures, array_flip($variant->featureValuesIds))) === count($checkRequireFeatures)) {
							$vals['variant'] = $variant;
							break;
						}
					}
				}
			}
		}

		return $arr;
	}

	public function getFromPrice(): float
	{
		$min = $this->getPrice();

		foreach ($this->variants as $v) {
			if ($min === 0.0 || ($v->getPrice() < $min && $v->getAvailability() && $v->getAvailability()->canAddToCart())) {
				$min = $v->getPrice();
			}
		}

		return $min;
	}

	public function getFromPriceWithoutVat(): float
	{
		$min = $this->getPriceWithoutVat();

		foreach ($this->variants as $v) {
			if ($min === 0.0 || ($v->getPriceWithoutVat() < $min && $v->getAvailability() && $v->getAvailability()->canAddToCart())) {
				$min = $v->getPriceWithoutVat();
			}
		}

		return $min;
	}

	/**
	 * @return RelatedGroup[]
	 */
	public function getRelated(): array
	{
		return $this->related;
	}

	/**
	 * @param RelatedGroup[] $products
	 */
	public function setRelated(array $products): void
	{
		$this->related = $products;
	}

	public function addDocument(Document $document): void { $this->documents[] = $document; }

	/**
	 * @return Document[]
	 */
	public function getDocuments(): array { return $this->documents; }

	public function addVideo(Video $video): void { $this->videos[] = $video; }

	/**
	 * @return Video[]
	 */
	public function getVideos(): array { return $this->videos; }

	public function setMaximumAmount(?int $max): void { $this->maximumAmount = $max; }

	public function getMaximumAmount(): ?int { return Config::load('product.allowMaximumQuantityPerOrder') ? $this->maximumAmount : null; }

	/**
	 * @param mixed $default
	 *
	 * @return mixed|null
	 */
	public function getMoreDataValue(string $key, $default = null) { return $this->moreData[$key] ?? $default; }

	public function getVariantsStockTableData(?int $row = null, ?int $column = null, string $quantityType = 'default'): array
	{
		$result   = [
			'rows'    => [],
			'columns' => [],
			'data'    => [],
		];
		$diffKeys = array_keys($this->getVariantDifferences());

		if (!$row) {
			$row = (int) (explode('-', (string) $diffKeys[0])[0] ?? 0);
		}

		if (!$column) {
			$column = (int) (explode('-', (string) $diffKeys[1])[0] ?? 0);
		}

		if (!$row || !$column) {
			return $result;
		}

		foreach ($this->variants as $variant) {
			if ($quantityType === 'external') {
				$quantity = $variant->getQuantityExternal();
			} else if ($quantityType === 'all') {
				$quantity = $variant->getQuantityWithExternal();
			} else {
				$quantity = $variant->getQuantity();
			}

			$rowFeature    = $variant->getFeatureById($row);
			$columnFeature = $variant->getFeatureById($column);

			if (!$rowFeature || !$columnFeature) {
				continue;
			}

			$result['data'][$rowFeature->idFeatureValue][$columnFeature->idFeatureValue] = $quantity;

			if (!isset($result['rows'][$rowFeature->valuePosition])) {
				$result['rows'][$rowFeature->valuePosition] = $rowFeature;
			}

			if (!isset($result['columns'][$columnFeature->valuePosition])) {
				$result['columns'][$columnFeature->valuePosition] = $columnFeature;
			}
		}

		ksort($result['rows']);
		ksort($result['columns']);

		return $result;
	}

	public function hasDisabledPickupPoints(): bool
	{
		return $this->disablePickUpSpedition
			|| ($this->defaultCategory && $this->defaultCategory->hasDisabledPickupPoints());
	}

	public function getSafetyWarningImage(): ?string
	{
		return $this->defaultCategory ? $this->defaultCategory->getSafetyWarningImage() : null;
	}

	public function getSafetyWarningText(): ?string
	{
		return $this->defaultCategory ? $this->defaultCategory->getSafetyWarningText() : null;
	}
}
