<?php declare(strict_types = 1);

namespace EshopCatalog\Model\Entities;

use Core\Model\Entities\TId;
use Core\Model\Entities\TTranslateListener;
use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\Common\Collections\Collection;
use Doctrine\ORM\Mapping as ORM;
use Gedmo\Mapping\Annotation as Gedmo;

/**
 * @ORM\Entity
 * @ORM\Table(name="eshop_catalog__availability")
 * @ORM\EntityListeners({"Core\Model\Entities\TranslateListener", "AvailabilityListener"})
 */
class Availability
{
	use TId;
	use TTranslateListener;

	public const IN_STOCK = 'inStock';
	public const SOLD_OUT = 'soldOut';
	public const PREORDER = 'preorder';
	public const RESERVED = 'reserved';

	/**
	 * @ORM\Column(type="string", length=255, nullable=false)
	 */
	protected string $ident;

	/**
	 * @ORM\Column(type="smallint", options={"unsigned":true, "default":0}, nullable=false)
	 */
	public int $delay;

	/**
	 * @var bool|int
	 * @ORM\Column(type="smallint", options={"default": 1}, nullable=false)
	 */
	public $canShowOnList;

	/**
	 * @var bool|int
	 * @ORM\Column(type="smallint", options={"default": 1}, nullable=false)
	 */
	public $canShowOnSearch;

	/**
	 * @var bool|int
	 * @ORM\Column(type="smallint", options={"default": 1}, nullable=false)
	 */
	public $canAddToCart;

	/**
	 * @ORM\Column(type="string", nullable=true)
	 */
	public ?string $color = null;

	/**
	 * @ORM\Column(type="string", nullable=true)
	 */
	public ?string $bgColor = null;

	/**
	 * @var Collection<string, AvailabilityText>
	 * @ORM\OneToMany(targetEntity="AvailabilityText", mappedBy="availability", indexBy="lang")
	 */
	protected Collection $texts;

	/**
	 * @var bool|int
	 * @ORM\Column(type="smallint", options={"default": 1}, nullable=false)
	 */
	protected $canDelete;

	/**
	 * @ORM\Column(type="string", length=255, nullable=true)
	 */
	public ?string $openGraphText = null;

	/**
	 * @Gedmo\SortablePosition
	 * @ORM\Column(type="integer")
	 */
	protected int $position;

	/**
	 * @ORM\Column(type="smallint", options={"unsigned": true, "default": 0}, nullable=false)
	 */
	public int $warehouseOverdraft = 0;

	/**
	 * @ORM\Column(type="smallint", options={"unsigned": true, "default": 1}, nullable=false)
	 */
	public int $showInFilter = 1;

	public function __construct(string $ident, int $delay)
	{
		$this->setIdent($ident);
		$this->delay     = $delay;
		$this->texts     = new ArrayCollection;
		$this->canDelete = 1;
		$this->setPosition(-1);
	}

	public function getIdent(): string { return $this->ident; }

	public function setIdent(string $ident): self
	{
		$this->ident = $ident;

		return $this;
	}

	public function getText(string $lang = ''): ?AvailabilityText { return $this->texts[$lang ?: $this->locale]; }

	/**
	 * @return Collection<string, AvailabilityText>
	 */
	public function getTexts(): Collection { return $this->texts; }

	/**
	 * @param AvailabilityText[] $texts
	 */
	public function setTexts(array $texts): self
	{
		$this->texts = new ArrayCollection($texts);

		return $this;
	}

	public function canDelete(): bool { return (bool) $this->canDelete; }

	public function getPosition(): int { return $this->position; }

	public function setPosition(int $position): self
	{
		$this->position = $position;

		return $this;
	}

}
