<?php declare(strict_types = 1);

namespace EshopCatalog\Model\Entities;

use Core\Model\Entities\TSeo;
use Doctrine\ORM\Mapping as ORM;
use Nette\Utils\Strings;

/**
 * @ORM\Table(name="eshop_catalog__category_texts")
 * @ORM\Entity
 * @ORM\EntityListeners({"CategoryTextListener"})
 */
class CategoryTexts
{
	use TSeo;

	/**
	 * @ORM\Id
	 * @ORM\ManyToOne(targetEntity="Category", inversedBy="categoryTexts", cascade={"persist"})
	 * @ORM\JoinColumn(name="id", referencedColumnName="id", onDelete="CASCADE")
	 * )
	 */
	public Category $id;

	/**
	 * @ORM\Id
	 * @ORM\Column(type="string", length=2)
	 */
	private string $lang;

	/**
	 * @ORM\Column(type="smallint", length=1, options={"default": 0})
	 */
	public int $isPublished = 0;

	/**
	 * @ORM\Column(type="string", length=255)
	 */
	public string $name;

	/**
	 * @ORM\Column(name="name_h1", type="string", length=255, nullable=true)
	 */
	public ?string $nameH1 = null;

	/**
	 * @ORM\Column(type="string", length=255)
	 */
	public string $alias;

	/**
	 * @ORM\Column(type="text", nullable=true)
	 */
	public ?string $description = null;

	/**
	 * @ORM\Column(type="text", nullable=true)
	 */
	public ?string $shortDescription = null;

	/**
	 * @ORM\Column(type="text", nullable=true)
	 */
	public ?string $emptyText = null;

	/**
	 * @ORM\Column(name="image", type="string", nullable=true, length=255)
	 */
	public ?string $image = null;

	/**
	 * @ORM\Column(name="safety_warning_text", type="text", nullable=true)
	 */
	public ?string $safetyWarningText = null;

	public function __construct(
		Category $id,
		string   $lang
	)
	{
		$this->id    = $id;
		$this->name  = '';
		$this->alias = '';
		$this->lang  = $lang;
		$this->seo   = [];
	}

	public function setName(string $name): self
	{
		$this->name = $name;
		if (!$this->alias) {
			$this->setAlias($name);
		}

		return $this;
	}

	public function getName(): string { return $this->name; }

	public function setAlias(string $alias): self
	{
		$alias       = $alias != '' ? $alias : $this->name;
		$this->alias = Strings::webalize($alias);

		return $this;
	}

	public function getCategory(): Category { return $this->id; }

	public function getLang(): string
	{
		return $this->lang;
	}

	public function setNameH1(string $nameH1): self
	{
		$this->nameH1 = $nameH1;

		return $this;
	}

	public function getNameH1(): string { return $this->nameH1 ?: $this->name; }

}
