<?php declare(strict_types = 1);

namespace EshopCatalog\Model\Personalization;

use Contributte\Translation\Translator;
use Core\Model\Application\AppState;
use Core\Model\Sites;
use Currency\Model\Entities\Currency;
use EshopCatalog\FrontModule\Model\Categories;
use EshopCatalog\FrontModule\Model\Dao\Category;
use EshopCatalog\FrontModule\Model\Dao\Product;
use EshopCatalog\FrontModule\Model\ProductsFacade;
use EshopCatalog\Model\Entities\CategoryProduct;
use EshopCatalog\Model\Entities\ProductInSite;
use EshopOrders\Model\Entities\Customer;
use EshopOrders\Model\Entities\Order;
use Nette\Caching\Cache;
use Nette\Caching\Storage;
use Nette\Http\IRequest;
use Nette\Http\IResponse;
use Nette\Http\Url;
use Nette\InvalidStateException;
use Nette\Utils\ArrayHash;
use Nette\Utils\Json;
use Nette\Utils\Strings;
use Nette\Utils\Validators;
use stdClass;
use Tracy\Debugger;

class Personalization
{
	protected ArrayHash $config;
	protected ProductsFacade $productsFacade;
	protected Categories $categories;
	protected Translator $translator;
	protected IResponse $response;
	protected IRequest $request;
	protected Sites $sites;
	protected Cache $cache;
	protected PersonalizationAccessControl $personalizationAccessControl;

	public const CACHE_NAMESPACE = 'personalization';

	public function __construct(
		array $config, IResponse $response, IRequest $request, ProductsFacade $productsFacade,
		Translator $translator, Sites $sites, Storage $storage, PersonalizationAccessControl $personalizationAccessControl,
		Categories $categories)
	{
		$this->config = ArrayHash::from($config);
		$this->response = $response;
		$this->request = $request;
		$this->productsFacade = $productsFacade;
		$this->translator = $translator;
		$this->sites = $sites;
		$this->categories = $categories;
		$this->personalizationAccessControl = $personalizationAccessControl;
		$this->cache = new Cache($storage, self::CACHE_NAMESPACE);
	}

	protected function getApiToken(): string
	{
		if ($token = $this->config->apiToken) {
			return $token;
		}

		$site = $this->sites->getCurrentSite();
		if ($site && array_key_exists($site->getIdent(), (array) $this->config->apiTokens) ) {
			return $this->config->apiTokens[$site->getIdent()];
		}

		throw new InvalidStateException('Api token not set for this site');
	}

	protected function getCurrencyCode(): string
	{
		/** @var Currency|null $currency */
		$currency = AppState::getState('currency');

		return $currency ? Strings::lower($currency->getCode()) : 'czk';
	}

	protected function getUserToken(): ?string
	{
		$token = $this->request->getCookie('personalization-user-token');

		if (!$token) {
			$token = md5(uniqid((string) mt_rand(), true) . time());
			$this->response->setCookie('personalization-user-token', $token, '1 year');
		}

		return $token;
	}

	protected function getBaseUrl(): Url
	{
		return new Url($this->config->baseApiUrl);
	}

	protected function getCacheKey(string $uniqueKey): string
	{
		return sprintf('%s-%s-%s-%s', $uniqueKey, $this->getApiToken(), $this->translator->getLocale(), $this->getCurrencyCode());
	}

	protected function prepareUrl(Url $baseUrl, bool $enableCurrency = true): Url
	{
		$baseUrl->setQueryParameter('lang', $this->translator->getLocale());
		$baseUrl->setQueryParameter('currency', $this->getCurrencyCode());
		$baseUrl->setQueryParameter('_access_token', $this->getApiToken());

		return $baseUrl;
	}

	/**
	 * @return array{response: bool|string, httpCode: int, error: string}
	 */
	protected function get(Url $url): array
	{
		$url = $this->prepareUrl($url);

		$ch = curl_init();

		curl_setopt($ch, CURLOPT_URL, $url);
		curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
		curl_setopt($ch, CURLOPT_HEADER, false);
		curl_setopt($ch, CURLOPT_CUSTOMREQUEST, 'GET');
		curl_setopt($ch, CURLOPT_CONNECTTIMEOUT, 0);
		curl_setopt($ch, CURLOPT_TIMEOUT, 0);

		$response = curl_exec($ch);
		$httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
		$error = curl_error($ch);

		curl_close($ch);

		return ['response' => $response, 'httpCode' => (int) $httpCode, 'error' => $error];
	}

	/**
	 * @return array{response: bool|string, httpCode: int, error: string}
	 */
	protected function post(string $url, array $data): array
	{
		$ch = curl_init();

		$headers = [
			'X-Token: ' . $this->getApiToken(),
			'Content-Type: application/json'
		];

		curl_setopt($ch, CURLOPT_URL, $url);
		curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
		curl_setopt($ch, CURLOPT_HEADER, false);
		curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
		curl_setopt($ch, CURLOPT_POSTFIELDS, Json::encode($data));
		curl_setopt($ch, CURLOPT_CUSTOMREQUEST, 'POST');
		curl_setopt($ch, CURLOPT_CONNECTTIMEOUT, 0);
		curl_setopt($ch, CURLOPT_TIMEOUT, 0);

		$response = curl_exec($ch);
		$httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
		$error = curl_error($ch);

		curl_close($ch);

		return ['response' => $response, 'httpCode' => (int) $httpCode, 'error' => $error];
	}

	/**
	 * @return int[]
	 */
	public function getAlternativeProductIds(int $forProductId): array
	{
		$cacheKey = $this->getCacheKey('alternativeProducts-' . $forProductId);

		return $this->cache->load($cacheKey, function(&$dep) use ($forProductId) {
			$dep = [
				Cache::EXPIRATION => '5 min',
			];

			$url = $this->getBaseUrl();
			$url->setPath('/api/v1/recommended-products/alternatives/' . $forProductId);
			$url->setQueryParameter('size', $this->config->alternativeProducts->size);

			$result = $this->get($url);
			$response = (string) $result['response'];
			$httpCode = $result['httpCode'];
			$error = $result['error'];

			if ($httpCode === 200) {
				$productIds = [];
				foreach (Json::decode($response) as $value) {
					if ($this->productsFacade->getProduct((int) $value->databaseId)) {
						$productIds[] = (int) $value->databaseId;
					}
				}

				return $productIds;
			}

			Debugger::log(Validators::isNone($error) ? 'Api not working' : $error);
			return [];
		});
	}

	public function putOrder(Order $order): void
	{
		if (!$this->personalizationAccessControl->isPutOrderAllowed()) {
			return;
		}

		$url = $this->getBaseUrl();
		$url->setPath('/api/v1/orders/put');

		$data = [];
		$data['id'] = (string) $order->getId();

		foreach ($order->getOrderItems() as $item) {
			$productStatus = 'active';
			$availabilityIdent = null;
			$productCategories = [];

			if ($item->getProduct()) {
				if ($item->getProduct()->getAvailability()) {
					$availabilityIdent = $item->getProduct()->getAvailability()->getIdent();
				}

				$productCategories = array_values(array_map(static fn(CategoryProduct $cp) => $cp->getIdCategory(), $item->getProduct()->getCategoryProducts()->toArray()));

				/** @var ProductInSite|null $productInSite */
				$productInSite = $item->getProduct()->sites->get($order->site->getIdent());
				if ($productInSite->category) {
					$productCategories[] = $productInSite->category->getId();
				}

				if ($daoProduct = $this->productsFacade->getProduct((int) $item->getProduct()->getId())) {
					$productStatus = $daoProduct->isActive && $daoProduct->canAddToCart ? 'active' : 'disabled';
				}
			}

			$data['items'][] = [
				'productId'         => (string) $item->getProductId(),
				'quantity'          => $item->getQuantity(), // deprecated
				'availabilityIdent' => $availabilityIdent, // deprecated
				'productCategories' => array_unique($productCategories),
				'productStatus'     => $productStatus
			];
		}

		$this->post((string) $url, $data);
	}

	public function increaseCtr(int $fromProductId, int $productId): void
	{
		$url = $this->getBaseUrl();
		$url->setPath('/api/v1/products/increase-ctr');

		$this->post((string) $url, ['productId' => $productId, 'fromProductId' => $fromProductId]);
	}

	public function notifyView(int $productId): void
	{
		if (!$this->personalizationAccessControl->isRecentlyViewedProductsAllowed()) {
			return;
		}

		$url = $this->getBaseUrl();
		$url->setPath('/api/v1/products/notify-view');

		$this->post((string) $url, ['productId' => $productId, 'token' => $this->getUserToken()]);
	}

	/**
	 * @return int[]
	 */
	public function getCrossSellProducts(int $productId): array
	{
		$cacheKey = $this->getCacheKey('crossSellProducts-' . $productId);

		return $this->cache->load($cacheKey, function(&$dep) use ($productId) {
			$dep = [
				Cache::EXPIRATION => '5 min',
			];

			$url = $this->getBaseUrl();
			$url->setPath('/api/v1/recommended-products/cross-sell/' . $productId);
			$url->setQueryParameter('size', $this->config->crossSellProducts->size);

			$result = $this->get($url);
			$response = (string) $result['response'];
			$httpCode = $result['httpCode'];
			$error = $result['error'];

			if ($httpCode === 200) {
				$productIds = [];
				foreach (Json::decode($response) as $value) {
					if ($this->productsFacade->getProduct((int) $value->databaseId)) {
						$productIds[] = (int) $value->databaseId;
					}
				}

				return $productIds;
			}

			Debugger::log(Validators::isNone($error) ? 'Api not working' : $error);
			return [];
		});
	}

	public function getBestSellingProducts(int $inCategoryId = null): array
	{
		$cacheKey = $this->getCacheKey('bestSellingProducts-' . $inCategoryId);

		return $this->cache->load($cacheKey, function(&$dep) use ($inCategoryId) {
			$dep = [
				Cache::EXPIRATION => '5 min',
			];

			$url = $this->getBaseUrl();
			$url->setPath('/api/v1/recommended-products/best-selling');
			$url->setQueryParameter('size', $this->config->bestSellingProducts->size);

			if ($inCategoryId) {
				$url->setQueryParameter('categoryId', $inCategoryId);
			}

			$result = $this->get($url);
			$response = (string) $result['response'];
			$httpCode = $result['httpCode'];
			$error = $result['error'];

			if ($httpCode === 200) {
				return Json::decode($response);
			}

			Debugger::log(Validators::isNone($error) ? 'Api not working' : $error);
			return [];
		});
	}

	/**
	 * @param int[] $exclude
	 * @return int[]
	 */
	public function getRecentlyViewedProducts(array $exclude = []): array
	{
		$token = $this->getUserToken();

		$url = $this->getBaseUrl();
		$url->setPath('/api/v1/recommended-products/recently-viewed');
		$url->setQueryParameter('size', $this->config->recentlyViewedProducts->size);
		$url->setQueryParameter('userToken', $token);

		if ($exclude) {
			$url->setQueryParameter('exclude', implode(',', $exclude));
		}

		$result = $this->get($url);
		$response = (string) $result['response'];
		$httpCode = $result['httpCode'];
		$error = $result['error'];

		if ($httpCode === 200) {
			$productIds = [];
			foreach (Json::decode($response) as $value) {
				if ($this->productsFacade->getProduct((int) $value->databaseId)) {
					$productIds[] = (int) $value->databaseId;
				}
			}
			return $productIds;
		}

		Debugger::log(Validators::isNone($error) ? 'Api not working' : $error);
		return [];
	}

	/**
	 * @param string $query
	 * @return array{result: Category[], total: int}
	 */
	public function getCategoriesByFilter(string $query): array
	{
		$url = $this->getSearchCategoriesUrl($query);

		$result = $this->get($url);
		$response = (string) $result['response'];
		$httpCode = $result['httpCode'];
		$error = $result['error'];

		if ($httpCode === 200) {
			$categories = [];
			$data = Json::decode($response);

			/** @var stdClass $value */
			foreach ($data->result as $value) {
				if ($category = $this->categories->get((int) $value->databaseId)) {
					$categories[] = $category;
				}
			}

			return ['result' => $categories, 'total' => $data->total];
		}

		Debugger::log(Validators::isNone($error) ? 'Api not working' : $error);
		return ['result' => [], 'total' => 0];
	}

	/**
	 * @return array{result: Product[], total: int}
	 */
	public function getProductsByFilter(string $query, int $limit, int $offset, array $categories = [], ?Customer $customer = null): array
	{
		$url = $this->getSearchProductsUrl($query, $limit, $offset, $categories, $customer);

		$result = $this->get($url);
		$response = (string) $result['response'];
		$httpCode = $result['httpCode'];
		$error = $result['error'];

		if ($httpCode === 200) {
			$products = [];
			$data = Json::decode($response);
			/** @var stdClass $value */
			foreach ($data->result as $value) {
				if ($product = $this->productsFacade->getProduct((int) $value->databaseId)) {
					$products[] = $product;
				}
			}

			return ['result' => $products, 'total' => $data->total];
		}

		Debugger::log(Validators::isNone($error) ? 'Api not working' : $error);
		return ['result' => [], 'total' => 0];
	}

	public function getAutocompleteUrl(string $query, ?Customer $customer = null): Url
	{
		$url = $this->getBaseUrl();
		$url->setPath('/api/v1/recommended-products/autocomplete');
		$url->setQueryParameter('q', $query);
		$url->setQueryParameter('sizeProducts', $this->config->autocompleteForm->sizeProducts);
		$url->setQueryParameter('sizeCategories', $this->config->autocompleteForm->sizeCategories);

		$url = $this->prepareUrl($url);

		if ($customer && $customer->getGroupCustomers()) {
			$url->setQueryParameter('groupId', $customer->getGroupCustomers()->getId());
		}

		return $url;
	}

	/**
	 * @param int[] $categories
	 */
	public function getSearchProductsUrl(string $query, int $limit, int $offset, array $categories = [], ?Customer $customer = null): Url
	{
		$url = $this->getBaseUrl();
		$url->setPath('/api/v1/recommended-products/search-products');
		$url->setQueryParameter('q', $query);
		$url->setQueryParameter('limit', $limit);
		$url->setQueryParameter('offset', $offset);

		if ($categories) {
			$url->setQueryParameter('categories', implode(',', $categories));
		}

		$url = $this->prepareUrl($url);

		if ($customer && $customer->getGroupCustomers()) {
			$url->setQueryParameter('groupId', $customer->getGroupCustomers()->getId());
		}

		return $url;
	}

	public function getSearchCategoriesUrl(string $query): Url
	{
		$url = $this->getBaseUrl();
		$url->setPath('/api/v1/recommended-products/search-categories');
		$url->setQueryParameter('q', $query);

		return $this->prepareUrl($url, false);
	}

}