<?php declare(strict_types = 1);

namespace EshopCatalog\Model\Subscribers;

use Core\Model\Entities\EntityManagerDecorator;
use Core\Model\Entities\ExtraField;
use Core\Model\Helpers\Strings;
use Core\Model\Sites;
use EshopCatalog\FrontModule\Model\Categories;
use EshopCatalog\Model\Config;
use EshopCatalog\Model\Entities\Category;
use EshopCatalog\Model\Event\RouteInFindAliasEvent;
use EshopCatalog\Model\Navigation\BaseNavigation;
use EshopCatalog\Model\Navigation\Home;
use EshopCatalog\Model\NavigationFilter;
use Navigations\Model\Event\RouteInEvent;
use Navigations\Model\Event\RouteOutEvent;
use Navigations\Model\Event\RouteUpdateCacheDep;
use Navigations\Model\Navigations;
use Nette\Caching\Cache;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;

class NavigationSubscriber implements EventSubscriberInterface
{
	protected EntityManagerDecorator $entityManager;
	protected NavigationFilter       $navigationFilter;
	protected Categories             $categories;
	protected Navigations            $navigations;
	protected Sites                  $sites;
	protected static array           $outFilter        = [];
	protected array                  $cRouteInNotFound = [];

	public function __construct(
		EntityManagerDecorator $entityManager,
		NavigationFilter       $navigationFilter,
		Categories             $categories,
		Navigations            $navigations,
		Sites                  $sites
	)
	{
		$this->entityManager    = $entityManager;
		$this->navigationFilter = $navigationFilter;
		$this->categories       = $categories;
		$this->navigations      = $navigations;
		$this->sites            = $sites;
	}

	public static function getSubscribedEvents(): array
	{
		return [
			BaseNavigation::class . '::routeOut'                => ['routeOut', 150],
			BaseNavigation::class . '::routeUpdateCacheDep'     => ['routeUpdateCacheDep', 150],
			Navigations::class . '::routeInNotFoundPreRedirect' => ['routeInNotFound', 150],
			Home::class . '::routeInStart'                      => ['routeInNotFound', 150],
			BaseNavigation::class . '::routeInFindAlias'        => ['routeInNotFound', 150],
			Navigations::class . '::routeInNotFound'            => ['routeInNotFoundRedirect', 110],
		];
	}

	public function routeUpdateCacheDep(RouteUpdateCacheDep $event): void
	{
		if (!empty(self::$outFilter)) {
			foreach (self::$outFilter as $type => $v) {
				foreach ($v as $k => $vv) {
					$event->cache[Cache::TAGS][] = 'categoriesFilter_' . $type . '_' . $k;
				}
			}

			self::$outFilter = [];
		}
	}

	public function routeOut(RouteOutEvent $event): void
	{
		if ($event->disableDefaultFilter === true) {
			return;
		}

		$urlParams        = &$event->urlParams;
		$event->urlParams = $this->navigationFilter->processRouterOut($urlParams);

		if ($urlParams['productsFilter-filter']) {
			self::$outFilter = $urlParams['productsFilter-filter'];
		}
	}

	/**
	 * @param RouteInEvent|RouteInFindAliasEvent $event
	 */
	public function routeInNotFound(object $event): void
	{
		if (!Config::load('allowCategoryFullUrlField', false)) {
			return;
		}

		$key = $event->urlParams['path'] . '_' . $event->urlParams['locale'];
		if (!isset($this->cRouteInNotFound[$key])) {
			$this->cRouteInNotFound[$key] = [];
			$resultCache                  = Config::load('productsList.enableResultCacheOther');

			$query = $this->entityManager->getRepository(ExtraField::class)->createQueryBuilder('ef')
				->select('ef.sectionKey')
				->where('ef.sectionName = :sn')
				->andWhere('ef.key = :key')
				->andWhere('ef.value IN (:value)')
				->andWhere('ef.lang = :lang')
				->setParameters([
					'sn'    => Category::EXTRA_FIELD_SECTION,
					'key'   => 'fullUrl',
					'value' => [
						ltrim($event->urlParams['path'], '/'),
						'/' . ltrim($event->urlParams['path'], '/'),
					],
					'lang'  => $event->urlParams['locale'],
				])->getQuery();

			if ($resultCache) {
				$cacheKey = md5(serialize(implode([
					ltrim($event->urlParams['path'], '/'),
					'/' . ltrim($event->urlParams['path'], '/'),
					$event->urlParams['locale'],
				])));
				$query->enableResultCache($resultCache, $cacheKey);
			}

			$catIds = $query->getScalarResult();

			foreach ($catIds as $catId) {
				$catId = $catId['sectionKey'];

				if ($catId) {
					$catId = (int) $catId;
					$cat   = $this->categories->get($catId);

					if ($cat) {
						$r   = $cat->getParent();
						$nav = null;
						$i   = 0;
						while ($r && $i <= 10) {
							$nav = $this->categories->findNavigationId($r->getId());

							if ($nav) {
								break;
							}

							$r = $r->getParent();
							$i++;
						}

						if ($nav) {
							$this->cRouteInNotFound[$key] = [
								'nav'              => $this->navigations->getNavigation($nav),
								'presenter'        => 'EshopCatalog:Front:Default',
								'action'           => 'category',
								'id'               => $catId,
								'activeNavigation' => $nav,
							];
						}
					}
				}
			}
		}

		if (!empty($this->cRouteInNotFound[$key])) {
			$event->return['presenter']        = $this->cRouteInNotFound[$key]['presenter'];
			$event->return['action']           = $this->cRouteInNotFound[$key]['action'];
			$event->return['id']               = $this->cRouteInNotFound[$key]['id'];
			$event->return['activeNavigation'] = $this->cRouteInNotFound[$key]['nav'];
			$event->stopPropagation();
		}
	}

	public function routeInNotFoundRedirect(RouteInEvent $event): void
	{
		if (isset($event->urlParams['productsFilter-filter'], $event->urlParams['path']) && !isset($event->urlParams['list-limit'])) {
			$tmpPath    = explode('/', $event->urlParams['path']);
			$endPath    = array_pop($tmpPath);
			$preEndPath = implode('/', $tmpPath);
			$siteIdent  = $this->sites->getCurrentSite()->getIdent();

			if ($siteIdent) {
				$categoryRoot = $this->categories->getRootIdForSite($siteIdent);
				$categories   = $this->categories->getCategories($categoryRoot, $event->urlParams['locale'] ?? 'cs');

				$possibleCategory = null;
				foreach ($categories as $cat) {
					$link       = explode('/', $cat->link, 4);
					$catPath    = end($link);
					$linkTmp    = explode('/', $catPath);
					$endLink    = array_pop($linkTmp);
					$preEndLink = implode('/', $linkTmp);
					if (Strings::contains($endPath, $endLink)) {
						if ($possibleCategory === null || (strlen($endLink) > strlen($possibleCategory['endLink']) && $preEndLink === $preEndPath)) {
							$possibleCategory = [
								'cat'     => $cat,
								'endLink' => $endLink,
								'depth'   => count($linkTmp),
							];
						}
					}
				}

				if ($possibleCategory) {
					header('HTTP/1.1 301 Moved Permanently');
					header('Location: ' . $possibleCategory['cat']->link);
					exit;
				}
			}
		}
	}
}
