<?php declare(strict_types = 1);

namespace EshopCatalog\AdminModule\Components\Products;

use Core\Model\UI\AbstractPresenter;
use Core\Model\UI\BaseControl;
use Core\Model\UI\Form\BaseForm;
use EshopCatalog\AdminModule\Model\Products;
use Nette\Application\IPresenter;
use Nette\Utils\ArrayHash;

class SetAsVariantsFor extends BaseControl
{
	/** @var Products */
	protected Products $products;

	public function __construct(Products $products)
	{
		$this->products = $products;
	}

	public function render(): void
	{
		$this->template->thisForm = $this['form'];
		$this->template->render($this->getTemplateFile());
	}

	public function handleLoadProducts()
	{
		$output = [];

		foreach ($this->products->getEr()->createQueryBuilder('p')
			         ->select('p.id, p.code1, p.ean, pt.name')
			         ->innerJoin('p.productTexts', 'pt', 'with', 'pt.lang = :lang')
			         ->leftJoin('p.isVariant', 'isVar')
			         ->where('isVar IS NULL OR isVar.isDefault = 1')
			         ->setParameters([
				         'lang' => $this->translator->getLocale(),
			         ])->getQuery()->getArrayResult() as $row) {
			$output[$row['id']] = [
				'id'    => (string) $row['id'],
				'name'  => (string) $row['name'],
				'code1' => (string) $row['code1'],
				'ean'   => (string) $row['ean'],
			];
		}

		$this->getPresenter()->sendJson($output);
	}

	/**
	 * @return BaseForm
	 */
	public function createComponentForm(): BaseForm
	{
		$form = $this->createForm();

		$form->addHidden('productId');
		$form->addText('product', 'eshopCatalog.product.searchProduct')
			->setHtmlAttribute('data-autocomplete-name', 'productsSetAsVariantsFor')
			->setHtmlAttribute('data-autocomplete-target', $form['productId']->getHtmlId())
			->setHtmlAttribute('data-autocomplete-keys', 'id,code1,ean,name');

		$this->monitor(IPresenter::class, function(AbstractPresenter $presenter) use ($form) {
			$form['product']->setHtmlAttribute('data-autocomplete-url', $this->link('loadProducts!'));
		});

		$form->addSaveCancelControl('saveControl');

		$form->onSuccess[] = [$this, 'formSuccess'];

		return $form;
	}

	/**
	 * @param BaseForm  $form
	 * @param ArrayHash $values
	 *
	 * @return bool
	 * @throws \Doctrine\ORM\ORMException
	 */
	public function formSuccess(BaseForm $form, ArrayHash $values): bool
	{
		$productsIds = $this->presenter->getParameter('ids');
		$form->addCustomData('productIds', $productsIds);
		$presenter = $this->getPresenter();

		if ($productsIds) {
			foreach (explode('-', $productsIds) as $prodId) {
				try {
					$r = $this->products->setAsVariantFor((int) $prodId, (int) $values->productId);
				} catch (\Exception $e) {
					$presenter->flashMessageDanger($e->getMessage());
				}
			}

			$presenter->flashMessageSuccess('default.saved');

			$this->em->flush();
		}

		return true;
	}
}
