<?php declare(strict_types = 1);

namespace EshopCatalog\Model\Entities;

use Core\Model\Entities\Country;
use Currency\Model\Entities\Currency;
use Doctrine\ORM\Mapping as ORM;

/**
 * @ORM\Entity
 * @ORM\Table(name="eshop_catalog__product_price")
 */
class ProductPrice
{
	/**
	 * @ORM\Id
	 * @ORM\ManyToOne(targetEntity="Product", inversedBy="prices")
	 * @ORM\JoinColumn(name="product", referencedColumnName="id", onDelete="CASCADE")
	 */
	protected Product $product;

	/**
	 * @ORM\Id
	 * @ORM\ManyToOne(targetEntity="Core\Model\Entities\Country")
	 * @ORM\JoinColumn(name="country", referencedColumnName="id", onDelete="CASCADE")
	 */
	protected Country $country;

	/**
	 * @ORM\Column(name="currency_code", type="string", length=255, nullable=true)
	 */
	public ?string $currency = null;

	/**
	 * @ORM\Column(name="price", type="decimal", precision=10, scale=2, nullable=true)
	 */
	public $price = null;

	/**
	 * @var double
	 * @ORM\Column(name="retail_price", type="decimal", precision=10, scale=2, nullable=true)
	 */
	public $retailPrice = null;

	/**
	 * @ORM\ManyToOne(targetEntity="VatRate")
	 * @ORM\JoinColumn(name="vat_rate", referencedColumnName="id", onDelete="SET NULL")
	 */
	public ?VatRate $vatRate = null;

	public function __construct(Product $product, Country $country, ?string $currency)
	{
		$this->product  = $product;
		$this->country  = $country;
		$this->currency = $currency;
	}

	public function getProduct(): Product { return $this->product; }

	public function getCountry(): Country { return $this->country; }

	public function toArray(): array
	{
		return [
			'country'     => $this->getCountry()->getId(),
			'product'     => $this->getProduct()->getId(),
			'currency'    => $this->currency,
			'price'       => $this->price,
			'retailPrice' => $this->retailPrice,
			'vatRate'     => $this->vatRate ? $this->vatRate->getId() : null,
		];
	}
}
