<?php declare(strict_types = 1);

namespace EshopCatalog\AdminModule\Components\ImportExport;

use Core\Model\Entities\ExtraField;
use Core\Model\Helpers\Strings;
use Core\Model\Http\CsvResponse;
use Core\Model\UI\BaseControl;
use Core\Model\UI\Form\BaseForm;
use EshopCatalog\AdminModule\Model\Manufacturers;
use EshopCatalog\AdminModule\Model\Products;
use EshopCatalog\FrontModule\Model\CacheService;
use EshopCatalog\Model\Entities\Product;
use EshopCatalog\Model\Entities\ProductPriceLevel;
use EshopCatalog\Model\Entities\ProductTexts;
use EshopOrders\Model\Entities\GroupCustomers;
use EshopStock\DI\EshopStockExtension;
use EshopStock\Model\Entities\SupplyProduct;
use Nette\Http\FileUpload;
use Nette\Utils\ArrayHash;
use Tracy\Debugger;

class ProductsByManufacturer extends BaseControl
{
	protected Products $productsService;

	protected Manufacturers $manufacturers;

	protected CacheService $cacheService;

	protected array $customFields = [];

	public function __construct(Products $productsService, Manufacturers $manufacturers, CacheService $cacheService)
	{
		$this->productsService = $productsService;
		$this->manufacturers   = $manufacturers;
		$this->cacheService    = $cacheService;
	}

	public function render(): void
	{
		$this->template->render($this->getTemplateFile());
	}

	protected function createComponentForm(): BaseForm
	{
		$form = $this->createForm();
		$form->setShowLangSwitcher(false);

		$form->addSelect('manufacturer', 'eshopCatalog.product.manufacturer', [null => 'default.all'] + $this->manufacturers->getOptionsForSelect());
		$form->addSubmit('export', 'eshopCatalog.importExport.exportData')
			->setValidationScope(null);

		$form->addUpload('upload', 'eshopCatalog.importExport.dataForImport');

		$baseFields    = ['name', 'code1', 'code2', 'ean', 'retailPrice', 'price', 'quantity', 'width', 'height',
			'depth', 'weight'];
		$fields        = [];
		$defaultFields = [];

		foreach ($baseFields as $v) {
			$fields[$v] = $this->t('eshopCatalog.importExport.fields.' . $v);
			if (!in_array($v, ['quantity']))
				$defaultFields[] = $v;
		}

		$customerGroups = $this->em->getRepository(GroupCustomers::class)->createQueryBuilder('gc', 'gc.id')
			->select('gc.id, gc.name')->getQuery()->getArrayResult();

		foreach ($customerGroups as $row) {
			$fields['cg' . $row['id']] = $this->t('eshopCatalog.importExport.fields.priceLevel') . ' - ' . $row['name'];
			$defaultFields[]           = 'cg' . $row['id'];
		}

		foreach ($this->customFields as $row) {
			$fields[$row['name']] = $this->t('eshopCatalog.importExport.fields.' . $row['name']);
		}

		$form->addCheckboxList('useFields', $this->t('eshopCatalog.importExport.useFields'), $fields)
			->setDefaultValue($defaultFields)
			->setTranslator(null);

		$form->addSubmit('import', 'eshopCatalog.importExport.importData');

		$form->onSuccess[] = [$this, 'formOnSuccess'];

		return $form;
	}

	public function formOnSuccess(BaseForm $form, ArrayHash $values): bool
	{
		set_time_limit(300);
		try {
			// Pokud se data exportují tak načteme data a odešleme do CSV response
			if ($form->isSubmitted()->name === 'export') {
				$filename = 'import-export-' . date('Y-m-d') . '.csv';

				// Základní data
				$head = [];
				foreach (['id', 'lang', 'name', 'code1', 'code2', 'ean', 'retailPrice', 'price',
					         'width', 'height', 'depth', 'weight', 'quantity'] as $v)
					$head[] = $this->t('eshopCatalog.importExport.head.' . $v);

				$csv = '"' . implode('";"', $head) . '"';

				// Sklad
				$stockData = [];
				if (class_exists(EshopStockExtension::class)) {
					$csv .= ';' . $this->t('eshopCatalog.importExport.head.stock');

					foreach ($this->em->getRepository(SupplyProduct::class)->createQueryBuilder('sp')
						         ->select('IDENTITY(sp.product) as product, IDENTITY(sp.order) as order, sp.writtenOffDate')
						         ->getQuery()->getScalarResult() as $row) {
						if (!isset($stockData[$row['product']]))
							$stockData[$row['product']] = 0;
						$stockData[$row['product']]++;

						if ($row['order'] || $row['writtenOffDate'])
							$stockData[$row['product']]--;
					}
				}

				// Skupiny zákazníků
				$customerGroups = $this->em->getRepository(GroupCustomers::class)->createQueryBuilder('gc', 'gc.id')
					->select('gc.id, gc.name')->getQuery()->getArrayResult();

				foreach ($customerGroups as $row) {
					$csv .= ';"cg|' . $row['id'] . '|' . $row['name'] . '"';
				}

				foreach ($this->customFields as $cs)
					$csv .= ';"' . $this->t('eshopCatalog.importExport.head.' . $cs['name']) . '"';

				$csv .= PHP_EOL;

				$qb = $this->productsService->getEr()->createQueryBuilder('p')
					->select('p.id, p.code1, pt.lang, pt.name, p.code2, p.ean, p.quantity, p.retailPrice, p.price, p.width, p.height, p.depth, p.weight')
					->addSelect('GROUP_CONCAT(CONCAT(IDENTITY(pl.groupId), \'|\' , pl.price)) as priceLevels')
					->innerJoin('p.productTexts', 'pt')
					->leftJoin('p.priceLevels', 'pl')
					->groupBy('p.id');

				if ($values->manufacturer)
					$qb->where('p.manufacturer = :manufacturer')->setParameter('manufacturer', $values->manufacturer);

				$data = [];
				foreach ($qb->getQuery()->getArrayResult() as $row)
					$data[$row['id']] = $row;

				$extraFields = [];
				foreach ($this->em->getRepository(ExtraField::class)->createQueryBuilder('ef')
					         ->select('ef.sectionKey, ef.key, ef.value')
					         ->where('ef.sectionName = :sectionName')
					         ->andWhere('ef.sectionKey IN (' . implode(',', array_keys($data)) . ')')
					         ->setParameters([
						         'sectionName' => Product::EXTRA_FIELD_SECTION,
					         ])->getQuery()->getArrayResult() as $row) {
					$extraFields[$row['sectionKey']][$row['key']] = $row['value'];
				}

				foreach ($data as $row) {
					$isVar = $row['isDefault'] === 0;

					// Základní data
					$csv .= '"' . $row['id'] . '";';
					$csv .= '"' . $row['lang'] . '";';
					$csv .= '"' . str_replace('"', '""', (string) $row['name']) . '";';
					$csv .= '"' . $row['code1'] . '";';
					$csv .= '"' . $row['code2'] . '";';
					$csv .= '="' . $row['ean'] . '";';
					$csv .= '"' . str_replace('.', ',', $row['retailPrice']) . '";';
					$csv .= '"' . str_replace('.', ',', $row['price']) . '";';
					$csv .= '"' . $row['width'] . '";';
					$csv .= '"' . $row['height'] . '";';
					$csv .= '"' . $row['depth'] . '";';
					$csv .= '"' . $row['weight'] . '";';
					$csv .= '"' . $row['quantity'] . '";';

					// Sklad
					if (class_exists(EshopStockExtension::class)) {
						$csv .= '"' . ($stockData[$row['id']] ?? 0) . '";';;
					}

					// Skupiny zákazníků
					$priceLevels = [];
					if ($row['priceLevels'])
						foreach (explode(',', $row['priceLevels']) as $v) {
							$v                  = explode('|', $v);
							$priceLevels[$v[0]] = $v[1] ?? '';
						}

					foreach ($customerGroups as $k => $v)
						$csv .= '"' . str_replace('.', ',', ($priceLevels[$k] ?? '')) . '";';

					foreach ($this->customFields as $cs) {
						$csValue = null;
						if ($cs['source'] === 'ef')
							$csValue = $extraFields[$row['id']][$cs['name']];

						if ($cs['type'] === 'price')
							$csValue = str_replace('.', ',', $csValue);

						if ($csValue)
							$csv .= '"' . $csValue . '";';
					}

					$csv .= PHP_EOL;
				}

				CsvResponse::sendResponse($filename, $csv);
			} elseif ($values->upload && $values->upload->hasFile()) {
				// Při importu připravíte názvy tabulek a další hodnoty
				$productTextsTableName = $this->em->getClassMetadata(ProductTexts::class)->getTableName();
				$priceLevelsTableName  = $this->em->getClassMetadata(ProductPriceLevel::class)->getTableName();

				/** @var FileUpload $upload */
				$upload = $values->upload;
				$fr     = fopen($upload->getTemporaryFile(), 'r');
				$this->em->beginTransaction();
				$customerGroups    = [];
				$csvData           = [];
				$priceLevels       = [];
				$allowedProductsId = [];
				$keysForClean      = [];
				$canUse            = $values->useFields;

				// Zjistíme produkty patřící k výrobci nebo všechny
				$qb = $this->productsService->getEr()->createQueryBuilder('p')->select('p.id');

				if ($values->manufacturer)
					$qb->where('p.manufacturer = :manu')->setParameter('manu', $values->manufacturer);

				foreach ($qb->getQuery()->getArrayResult() as $row)
					$allowedProductsId[] = $row['id'];

				$extraFields = [];
				foreach ($this->em->getRepository(ExtraField::class)->createQueryBuilder('ef')
					         ->select('ef.sectionKey, ef.key, ef.value')
					         ->where('ef.sectionName = :sectionName')
					         ->andWhere('ef.sectionKey IN (' . implode(',', $allowedProductsId) . ')')
					         ->setParameters([
						         'sectionName' => Product::EXTRA_FIELD_SECTION,
					         ])->getQuery()->getArrayResult() as $row) {
					$extraFields[$row['sectionKey']][$row['key']] = $row['value'];
				}

				// Přečteme nahraný csv soubor
				while (!feof($fr) && $line = fgetcsv($fr, null, ';')) {
					// V prvním řádku načteme dynamické sloupečky
					if (Strings::toAscii($line[0]) == 'ID') {
						foreach ($line as $k => $v) {
							$tmp = explode('|', $v);

							if ($tmp[0] == 'cg') {
								$customerGroups[$k] = $tmp[1];
							}
						}

						continue;
					}

					if (!$line[0])
						continue;

					$productId = (int) $line[0];

					// Zkontrolujeme existenci produktu
					if (!in_array($productId, $allowedProductsId))
						continue;

					$csvData[$productId] = array_map('utf8_encode', $line);
				}

				// Načteme cenové hladiny
				foreach ($this->em->getRepository(ProductPriceLevel::class)->createQueryBuilder('pl')->select('IDENTITY(pl.productId) as productId, IDENTITY(pl.groupId) as groupId, pl.price')
					         ->innerJoin('pl.productId', 'p')
					         ->getQuery()->getArrayResult() as $row) {
					$priceLevels[$row['groupId'] . '-' . $row['productId']] = $row['price'];
				}

				foreach ($csvData as $productId => $line) {
					$lang = trim($line[1]);
					$name = trim($line[2]);

					if ($lang)
						$keysForClean[] = 'product/' . $productId . '/' . $lang;

					// Aktualizace názvu
					if (in_array('name', $canUse) && $lang && $name) {
						$name = preg_replace("/u([0-9a-fA-F]{4})/", "&#x\\1;", $name);
						$name = iconv("UTF-8", "ISO-8859-1//TRANSLIT", $name);
						$this->em->getConnection()->update($productTextsTableName, [
							'name' => $name,
						], [
							'id'   => $productId,
							'lang' => $lang,
						]);
					}

					$product = $this->em->getReference(Product::class, $productId);

					$code1       = trim($line[3]);
					$code2       = trim($line[4]);
					$ean         = trim($line[5]);
					$ean         = (int) trim($ean, '"=');
					$retailPrice = strtr(trim($line[6]), [' ' => '', ',' => '.']);
					$price       = strtr(trim($line[7]), [' ' => '', ',' => '.']);
					$width       = $line[8] !== '' && $line[8] !== null ? (int) $line[8] : null;
					$height      = $line[9] !== '' && $line[9] !== null ? (int) $line[9] : null;
					$depth       = $line[10] !== '' && $line[10] !== null ? (int) $line[10] : null;
					$weight      = $line[11] !== '' && $line[11] !== null ? (int) $line[11] : null;
					$quantity    = $line[12] !== '' && $line[12] !== null ? (int) $line[12] : null;

					// Aktulizace hodnot pokud jsou validní
					if ($code1 && in_array('code1', $canUse))
						$product->code1 = $code1;
					if ($code2 && in_array('code2', $canUse))
						$product->code2 = $code2;
					if ($ean && in_array('ean', $canUse))
						$product->ean = (string) $ean;
					if ($quantity !== null && is_numeric($quantity) && in_array('quantity', $canUse))
						$product->quantity;
					if ($retailPrice !== null && is_numeric($retailPrice) && in_array('retailPrice', $canUse))
						$product->retailPrice = $retailPrice;
					if ($price !== null && is_numeric($price) && in_array('price', $canUse))
						$product->price = $price;
					foreach (['width', 'height', 'depth', 'weight'] as $c) {
						if (${$c} !== null && in_array($c, $canUse))
							$product->{$c} = ${$c};
					}

					$this->em->persist($product);
					$this->em->flush($product);

					// Aktualizace cenových hladin
					$lineK = 12;
					foreach ($customerGroups as $lineK => $customerGroupId) {
						if (!in_array('cg' . $customerGroupId, $canUse))
							continue;

						$price = strtr(trim($line[$lineK]), [' ' => '', ',' => '.']);
						$exist = isset($priceLevels[$customerGroupId . '-' . $productId]);

						if (is_numeric($price)) {
							// Aktualizace nebo vytvoření
							if ($exist)
								$this->em->getConnection()->update($priceLevelsTableName, [
									'price' => $price,
								], [
									'product_id' => $productId,
									'group_id'   => $customerGroupId,
								]);
							else
								$this->em->getConnection()->insert($priceLevelsTableName, [
									'product_id' => $productId,
									'group_id'   => $customerGroupId,
									'price'      => $price,
								]);
						} else if ($exist) {
							// Smazání
							$this->em->getConnection()->delete($priceLevelsTableName, [
								'product_id' => $productId,
								'group_id'   => $customerGroupId,
							]);
						}
					}

					foreach ($this->customFields as $cs) {
						$lineK++;
						$value = $line[$lineK];

						if ($cs['type'] === 'price')
							$value = str_replace(',', '.', $value);

						if ($cs['source'] === 'ef') {
							$ef = $extraFields[$productId][$cs['name']] ?? null;
							if ($ef) {
								if ($value == '')
									$this->em->getConnection()->delete('core__extra_field', [
										'section_name' => Product::EXTRA_FIELD_SECTION,
										'section_key'  => $productId,
										'`key`'        => $cs['name'],
									]);
								else
									$this->em->getConnection()->executeStatement("UPDATE core__extra_field SET `value` = ? WHERE section_name = ? AND section_key = ? AND `key` = ?", [
										$value, Product::EXTRA_FIELD_SECTION, $productId, $cs['name'],
									]);
							} else if ($value) {
								$this->em->getConnection()->insert('core__extra_field', [
									'section_name' => Product::EXTRA_FIELD_SECTION,
									'section_key'  => $productId,
									'`key`'        => $cs['name'],
									'`value`'      => $value,
									'`lang`'       => null,
								]);
							}
						}
					}
				}

				$this->em->commit();
				if ($keysForClean)
					$this->cacheService->productCache->remove($keysForClean);
				$this->getPresenter()->flashMessageSuccess('default.saved');
				$this->getPresenter()->redrawControl('flashes');
				$this->template->messageOk = $this->t('default.saved');
				$this->redrawControl('formErrors');
			}
		} catch (\Exception $e) {
			$this->getPresenter()->flashMessageDanger($e->getMessage());
			$this->getPresenter()->redrawControl('flashes');

			$this->template->messageError = $e->getMessage();
			$this->redrawControl('formErrors');

			if ($this->em->getConnection()->isTransactionActive())
				$this->em->rollback();

			return false;
		}

		return true;
	}
}
