<?php declare(strict_types = 1);

namespace EshopCatalog\AdminModule\Model;

use Core\Model\Entities\Repository\NestedTreeRepository;
use Core\Model\Helpers\Arrays;
use Core\Model\Helpers\BaseEntityService;
use Core\Model\Helpers\Traits\TPublish;
use Core\Model\Http\Session;
use Core\Model\Lang\Langs;
use Doctrine\ORM\Query\Expr\Join;
use EshopCatalog\FrontModule\Model\CacheService;
use EshopCatalog\Model\Entities;
use EshopCatalog\Model\Entities\Category;
use Nette\Caching\Cache;
use Nette\Localization\ITranslator;
use Tracy\Debugger;

/**
 * Class Categories
 * @package EshopCatalog\AdminModule\Model
 *
 * @method Category|object|null getReference($id)
 * @method Category[]|null getAll()
 * @method NestedTreeRepository getEr()
 */
class Categories extends BaseEntityService
{
	use TPublish;

	public static bool $hidePrefixAndSuffix = false;

	/** @var Session @inject */
	public $session;

	protected $entityClass = Category::class;

	/** @var CacheService */
	protected $cacheService;

	/** @var ITranslator */
	protected $translator;

	/** @var array */
	protected $cForTree;

	/** @var Langs */
	protected $langs;

	protected ?array $cRootIds = null;

	public function __construct(CacheService $cacheService, ITranslator $translator, Langs $langs)
	{
		$this->cacheService = $cacheService;
		$this->translator   = $translator;
		$this->langs        = $langs;
	}

	public function getSessionSection()
	{
		return $this->session->getSection('EshopCatalog/Admin/Categories');
	}

	public function get($id): ?Category
	{
		return $this->getEr()->createQueryBuilder('c')->addSelect('ct')
			->leftJoin('c.categoryTexts', 'ct')
			->where('c.id = :id')->setParameter('id', $id)
			->getQuery()->getOneOrNullResult();
	}

	public function getRootIds(): array
	{
		if ($this->cRootIds === null) {
			$this->cRootIds = [];

			foreach ($this->em->getRepository(Entities\Category::class)->createQueryBuilder('c')
				         ->select('c.id, ct.alias')
				         ->where('c.lvl = 0 AND c.id = c.root')
				         ->innerJoin('c.categoryTexts', 'ct')
				         ->getQuery()->getArrayResult() as $row) {
				if ($row['alias'])
					$this->cRootIds[$row['id']] = $row['alias'];
			}
		}

		return $this->cRootIds;
	}

	public function getSiteRootId(?string $siteIdent = null): int
	{
		if ($siteIdent)
			$rootId = $this->em->getRepository(Entities\CategoryTexts::class)->createQueryBuilder('ct')
					->select('IDENTITY(ct.id) as id')
					->where('ct.alias = :alias')
					->setParameter('alias', $siteIdent)
					->getQuery()->setMaxResults(1)->getArrayResult()[0]['id'] ?? null;
		else
			$rootId = $this->em->getRepository(Entities\Category::class)->createQueryBuilder('c')
					->select('c.id')
					->where('c.lvl = 0 AND c.id = c.root')
					->getQuery()->setMaxResults(1)->getArrayResult()[0]['id'] ?? null;

		if (!$rootId) {
			$category = new Category();
			$this->em->persist($category);

			foreach ($this->langs->getLangs(false) as $lang) {
				$categoryText = new Entities\CategoryTexts($category, $lang->getTag());
				$categoryText->setAlias($siteIdent);
				$this->em->persist($categoryText);
			}

			$this->em->flush();
			$rootId = $category->getId();
		}

		return (int) $rootId;
	}

	/**
	 * @param $id
	 *
	 * @return bool
	 * @throws \Exception
	 */
	public function removeCategory($id)
	{
		try {
			foreach ($this->em->getRepository(Entities\CategoryTexts::class)->findBy(['id' => $id]) as $ct) {
				$this->em->remove($ct);
			}
			$this->em->flush();
			$this->em->remove($this->getReference($id));
			$this->em->flush();

			return true;
		} catch (\Exception $e) {
		}

		return false;
	}

	public function cleanCacheDeep($categoryId): void
	{
		$cat = $this->get($categoryId);

		if (!$cat)
			return;

		$tags = ['category/' . $cat->getId(), 'hierarchy'];
		$loop = function($childs) use (&$loop, &$ids) {
			foreach ($childs as $child) {
				$tags[] = 'category/' . $child->getId();

				if ($child->children->toArray())
					$loop($child->children->toArray());
			}
		};

		$loop($cat->children->toArray());

		$this->cacheService->categoryCache->clean([Cache::TAGS => $tags]);
	}

	protected function prepareDataForTree(?string $siteIdent = null): array
	{
		$cKey = $siteIdent ?: 'default';
		if ($this->cForTree[$cKey] === null) {
			$rootId = null;

			if ($siteIdent)
				$rootId = $this->getSiteRootId($siteIdent);

			$roots      = [];
			$categories = [];
			$flat       = [];

			$qb = $this->getEr()->createQueryBuilder('c')
				->select('c.id, ct.name, c.isPublished, c.lvl, IDENTITY(c.parent) as parent')
				->leftJoin('c.categoryTexts', 'ct', 'WITH', 'ct.lang = :lang')
				->setParameter('lang', $this->translator->getLocale())
				->groupBy('c.id')
				->addOrderBy('c.root')->addOrderBy('c.lft');

			if ($rootId)
				$qb->andWhere('c.root = :root')->setParameter('root', $rootId);

			foreach ($qb->getQuery()->getResult() as $c) {
				if ($c['lvl'] == 0) {
					$roots[$c['id']] = [
						'id'   => $c['id'],
						'name' => $c['name'],
					];
					continue;
				}

				$name = !$c['isPublished'] && !self::$hidePrefixAndSuffix ? '[x ' . $c['name'] . ' x]' : $c['name'];

				$categories[$c['id']] = $name;
				$flat[]               = [
					'id'     => $c['id'],
					'parent' => $c['parent'],
					'name'   => $name,
				];
			}

			$this->cForTree[$cKey] = [
				'roots'      => $roots,
				'categories' => $categories,
				'flat'       => $flat,
			];
		}

		return $this->cForTree[$cKey];
	}

	public function getOptionsForSelect(?string $siteIdent = null): array { return $this->prepareDataForTree($siteIdent)['categories']; }

	public function getFlatTree(?string $siteIdent = null): array { return $this->prepareDataForTree($siteIdent)['flat']; }

	public function getTreeForSelect(?string $siteIdent = null): array
	{
		$data = $this->prepareDataForTree($siteIdent);

		$trees = [];
		foreach ($data['roots'] as $id => $root) {
			$trees[$id] = [
				'name'     => $root['name'],
				'children' => Arrays::buildTree($data['flat'], 'parent', 'id', $root['id']),
			];
		}

		return $trees;
	}
}
