<?php declare(strict_types = 1);

namespace EshopCatalog\AdminModule\Presenters;

use EshopCatalog\AdminModule\Components\Features\IFeatureFormFactory;
use EshopCatalog\AdminModule\Components\Features\IFeatureValueFormFactory;
use EshopCatalog\AdminModule\Components\Features\IFeaturesGridFactory;
use EshopCatalog\AdminModule\Components\Features\IFeatureValuesGridFactory;
use EshopCatalog\Model\Entities\Feature;
use EshopCatalog\Model\Entities\FeatureValue;
use Core\Model\UI\Form\BaseForm;
use Nette\Http\IResponse;
use Nette\Utils\ArrayHash;

/**
 * Class VariantsFeaturesPresenter
 * @package EshopCatalog\AdminModule\Presenters
 */
class VariantsFeaturesPresenter extends BasePresenter
{

	/*******************************************************************************************************************
	 * ==================================   Actions
	 */

	public function actionDefault()
	{
		$this->setTitle($this->translator->translate('eshopCatalog.title.variantsFeatures'));
		$this->setHeader('eshopCatalog.title.variantsFeatures', 'fa fa-list');

		$this['navigation']->setData(['header' => [
			[
				'title' => 'eshopCatalog.menu.addFeature',
				'link'  => 'VariantsFeatures:addFeature',
				'ico'   => 'plus',
			], [
				'title' => 'eshopCatalog.menu.addFeatureValue',
				'link'  => 'VariantsFeatures:addFeatureValue',
				'ico'   => 'plus',
			],
		]]);
	}

	public function actionAddFeature()
	{
		$this->setTitle($this->translator->translate('eshopCatalog.title.newFeature'));
		$this->setHeader('eshopCatalog.title.newFeature', 'fa fa-list-alt');
		$link = null;

		if ($this->getParameter('id'))
			$link = $this->link('VariantsFeatures:addFeatureValue', ['featureId' => $this->getParameter('id')]);

		$this['navigation']->setData(['header' => [
			[
				'title' => 'eshopCatalog.menu.addFeatureValue',
				'link'  => $link ?: 'VariantsFeatures:addFeatureValue',
				'ico'   => 'plus',
			],
		]]);
	}

	public function actionAddFeatureValue()
	{
		$this->setTitle($this->translator->translate('eshopCatalog.title.newFeatureValue'));
		$this->setHeader('eshopCatalog.title.newFeatureValue', 'fa fa-list-alt');
	}

	public function actionEditFeature($id)
	{
		$feature = $this->em->getRepository(Feature::class)->find($id);
		$link    = null;

		if ($this->getParameter('id'))
			$link = $this->link('VariantsFeatures:addFeatureValue', ['featureId' => $this->getParameter('id')]);

		if (!$feature)
			$this->error(null, IResponse::S404_NOT_FOUND);

		$this->setTitle($this->translator->translate('eshopCatalog.title.editFeature'));
		$this->setHeader('eshopCatalog.title.editFeature', 'fa fa-list-alt');

		$this['navigation']->setData(['header' => [
			[
				'title' => 'eshopCatalog.menu.addFeatureValue',
				'link'  => $link ?: 'VariantsFeatures:addFeatureValue',
				'ico'   => 'plus',
			],
		]]);
	}

	public function actionEditFeatureValue($id)
	{
		$featureValue = $this->em->getRepository(FeatureValue::class)->find($id);

		if (!$featureValue)
			$this->error(null, IResponse::S404_NOT_FOUND);

		$this->setTitle($this->translator->translate('eshopCatalog.title.editFeatureValue'));
		$this->setHeader('eshopCatalog.title.editFeatureValue', 'fa fa-list-alt');
	}

	public function actionClose($id)
	{
		$this->redirect('Default:');
	}

	/*******************************************************************************************************************
	 * ==================================   Components
	 */
	protected function createComponentFeatureForm(IFeatureFormFactory $feature)
	{
		$control = $feature->create();

		if ($this->getParameter('id'))
			$control->setFeature($this->getParameter('id'));

		$control['form']->onSuccessSave[]         = function(BaseForm $form) {
			$this->redirect('VariantsFeatures:editFeature', $form->getCustomData('featureId'));
		};
		$control['form']->onSuccessSaveAndClose[] = function(BaseForm $form) {
			$this->redirect('VariantsFeatures:default');
		};
		$control['form']->onCancel[]              = function(BaseForm $form) {
			$this->redirect('VariantsFeatures:default');
		};

		return $control;
	}

	protected function createComponentFeatureValueForm(IFeatureValueFormFactory $factory)
	{
		$control = $factory->create();
		if ($this->getParameter('id'))
			$control->setFeatureValue($this->getParameter('id'));
		if ($this->getParameter('featureId'))
			$control->setFeature((int) $this->getParameter('featureId'));

		$control['form']->onSuccessSave[]         = function(BaseForm $form) {
			$this->redirect('VariantsFeatures:editFeatureValue', $form->getCustomData('featureValueId'));
		};
		$control['form']->onSuccessSaveAndClose[] = function(BaseForm $form) {
			$this->redirect('VariantsFeatures:editFeature', $form->getCustomData('featureId'));
		};

		if ($this->getHttpRequest()->getQuery('feature'))
			$control['form']['feature']->setDefaultValue($this->getHttpRequest()->getQuery('feature'));

		$control['form']->onCancel[]              = function(BaseForm $form) {
			$featureId = $form->getValues()['feature'];
			$valueId   = $this->getParameter('id');
			if ($valueId && !$featureId) {
				$featureId = $this->em->getRepository(FeatureValue::class)->createQueryBuilder('fv')->select('IDENTITY(fv.feature) as feature')
						->where('fv.id = :id')->setParameter('id', $valueId)
						->getQuery()->getArrayResult()[0]['feature'] ?? null;
			}

			if ($featureId)
				$this->redirect('VariantsFeatures:editFeature', $featureId);
			else
				$this->redirect('VariantsFeatures:default');
		};
		$control['form']->onSuccess[]             = function(BaseForm $form, ArrayHash $values) {
			if ($form->isSubmitted()->name == 'saveAndNew')
				$this->redirect('VariantsFeatures:addFeatureValue', ['feature' => $values->feature]);
		};

		return $control;
	}

	protected function createComponentFeaturesGrid(IFeaturesGridFactory $factory)
	{
		return $factory->create();
	}

	protected function createComponentFeatureValuesGrid(IFeatureValuesGridFactory $factory)
	{
		$control = $factory->create();

		if ($this->getParameter('id'))
			$control->setFeature($this->getParameter('id'));

		return $control;
	}
}
