<?php declare(strict_types = 1);

namespace EshopCatalog\CronModule\Model;

use Core\Model\Entities\EntityManagerDecorator;
use Core\Model\Images\ImagePipe;
use Core\Model\Sites;
use Currency\Model\Currencies;
use Currency\Model\Exchange;
use Doctrine\ORM\Query\Expr\Join;
use EshopCatalog\FrontModule\Model\AvailabilityService;
use EshopCatalog\FrontModule\Model\Categories;
use EshopCatalog\FrontModule\Model\Dao\Product;
use EshopCatalog\FrontModule\Model\ProductsFacade;
use EshopCatalog\Model\Config;
use EshopCatalog\Model\Entities\ProductPriceLevel;
use Nette\Utils\FileSystem;
use Nette\Utils\Json;

class ProductsFeed
{
	protected EntityManagerDecorator $em;

	protected ProductsFacade $productsFacade;

	protected Categories $categories;

	protected Sites $sites;

	protected AvailabilityService $availabilityService;

	protected Exchange $exchange;

	protected Currencies $currencies;

	protected ImagePipe $imagePipe;

	public string $baseUrl = '';

	public ?string $generate = null;

	public function __construct(EntityManagerDecorator $em, ProductsFacade $productsFacade, Categories $categories, Sites $sites,
	                            AvailabilityService    $availabilityService, Exchange $exchange, Currencies $currencies, ImagePipe $imagePipe)
	{
		$this->em                  = $em;
		$this->productsFacade      = $productsFacade;
		$this->categories          = $categories;
		$this->sites               = $sites;
		$this->availabilityService = $availabilityService;
		$this->exchange            = $exchange;
		$this->currencies          = $currencies;
		$this->imagePipe           = $imagePipe;
	}

	protected function getAllSiteProductsId(): array
	{
		$ids = [];
		foreach ($this->productsFacade->productsService->getEr()->createQueryBuilder('p')
			         ->select('p.id')
			         ->innerJoin('p.sites', 's', Join::WITH, 's.site = :site AND s.isActive = 1')
			         ->where('p.isPublished = :pub')
			         ->setParameters([
				         'site' => $this->sites->getCurrentSite()->getIdent(),
				         'pub'  => 1,
			         ])
			         ->getQuery()->getScalarResult() as $row) {
			$ids[] = $row['id'];
		}

		return $ids;
	}

	protected function getAvailabilityData(Product $product): array
	{
		$item = [
			'availability' => $product->getAvailability()->getName(),
		];

		$item['available'] = $product->getQuantity() + $product->getQuantityExternal();

		return $item;
	}

	public function getProductsAvailability(): array
	{
		$data = [];
		$avs  = $this->availabilityService->getAll();

		foreach ($this->productsFacade->productsService->getEr()->createQueryBuilder('p')
			         ->select('p.id, p.quantity, IDENTITY(p.availability) as availability, SUM(ps.quantity) as qeSum')
			         ->innerJoin('p.sites', 's', Join::WITH, 's.site = :site AND s.isActive = 1')
			         ->leftJoin('p.suppliers', 'ps')
			         ->groupBy('p.id')
			         ->setParameter('site', $this->sites->getCurrentSite()->getIdent())
			         ->getQuery()->getScalarResult() as $row) {
			if (!$row['qeSum'])
				$row['qeSum'] = 0;
			$av = $avs[$row['availability']];

			$item = [
				'id'           => $row['id'],
				'availability' => $av->getName(),
				'available'    => $row['quantity'] + $row['qeSum'],
			];

			$data[] = $item;
		}

		return $data;
	}

	public function getProducts(): array
	{
		$data = [
			'count' => 0,
		];

		$hasCzk = $this->currencies->getActive()['CZK'] ?? null;
		$hasEur = $this->currencies->getActive()['EUR'] ? (float) $this->currencies->getActive()['EUR']->rate : null;

		$priceLevels = [];
		foreach ($this->em->createQueryBuilder()
			         ->select('IDENTITY(pl.productId) as product, IDENTITY(pl.groupId) as group, pl.price')
			         ->from(ProductPriceLevel::class, 'pl')
			         ->getQuery()->getArrayResult() as $row) {
			$priceLevels[$row['product']][$row['group']] = (float) $row['price'];
		};

		$currentSite   = $this->sites->getCurrentSite();
		$site          = $currentSite->getIdent();
		$baseFile      = WWW_DIR . '/feed-' . $site . '-' . $currentSite->getCurrentDomain()->getLang();
		$tmpFile       = $baseFile . '.tmp';
		$file          = $baseFile . '.json';
		$this->baseUrl = 'https://' . $currentSite->getCurrentDomain()->getDomain() . '/';

		FileSystem::delete($tmpFile);
		FileSystem::createDir(dirname($tmpFile));
		file_put_contents($tmpFile, '{');
		$isFirstRow = true;

		$this->em->getConfiguration()->setSQLLogger(null);
		foreach (array_chunk($this->getAllSiteProductsId(), 250) as $chunk) {
			foreach ($this->productsFacade->getProducts($chunk) as $product) {
				if (!$product->getAvailability() || !$product->getAvailability()->canAddToCart())
					continue;

				$link    = explode('/', $product->link);
				$link[2] = $currentSite->getCurrentDomain()->getDomain();
				$link    = implode('/', $link);

				$item = [
					'id'                  => $product->getId(),
					'code1'               => $product->getCode1(),
					'ean'                 => $product->getEan(),
					'title'               => $product->getName(),
					'description'         => strip_tags((string) $product->shortDescription),
					'longDescription'     => $product->getDescription(),
					'link'                => $link,
					'deliveryDate'        => $product->getAvailability()->getDelay(),
					'categoryId'          => [],
					'category'            => [],
					'categoryHierarchyId' => [],
					'categoryHierarchy'   => [],
					'width'               => $product->width,
					'height'              => $product->height,
					'depth'               => $product->depth,
					'weight'              => $product->weight,
				];

				$usedCats = [];
				foreach (array_merge([$product->defaultCategoryId], $product->categories) as $catId) {
					$cat = $this->categories->get((int) $catId);

					if (!$cat || in_array($cat->getId(), $usedCats))
						continue;

					$usedCats[]           = $cat->getId();
					$item['categoryId'][] = $cat->getId();
					$item['category'][]   = $cat->name;

					$parents = array_reverse($cat->getParentPath());
					$tmp1    = [];
					$tmp2    = [];
					foreach ($parents as $parent) {
						$usedCats[] = $parent->getId();
						$tmp1[]     = $parent->getId();
						$tmp2[]     = $parent->name;
					}

					$tmp1[] = $cat->getId();
					$tmp2[] = $cat->name;

					$item['categoryHierarchyId'][] = $tmp1;
					$item['categoryHierarchy'][]   = $tmp2;
				}

				if ($product->variantId) {
					$item['itemGroupId'] = $product->variantId;

					if ($product->variantOf == $product->getId())
						$item['isMaster'] = 1;
				}

				if ($product->getManufacturer())
					$item['brand'] = $product->getManufacturer()->name;

				$addOriginalPrice = $product->getRetailPrice() && $product->getRetailPrice() > $product->getPrice();
				if ($hasCzk) {
					$item['priceCzk'] = $this->exchange->change($product->getPrice(), 'CZK');
					if ($addOriginalPrice)
						$item['priceOriginalCzk'] = $this->exchange->change($product->getRetailPrice(), 'CZK');
				}
				if ($hasEur !== null) {
					$item['rateEur'] = $hasEur;
				}

				foreach ($priceLevels[$product->getId()] ?? [] as $group => $price) {
					if ($hasCzk)
						$item['priceLevels'][$group . '_czk'] = $this->exchange->change($price, 'CZK');
				}

				$gallery = $product->getGallery();
				if ($gallery && $gallery->getCover()) {

					$item['imageLink'] = Config::load('productsFeed.makeThumbs')
						? $this->imagePipe->request($gallery->getCover()->getFilePath(), Config::load('productsFeed.thumbsSize'), 'fit')
						: $gallery->getCover()->getFilePath();

					foreach ($gallery->getImages() as $image) {
						if ($image->id === $gallery->getCover()->id || isset($item['additionalImageLink']))
							continue;
						$item['additionalImageLink'] = Config::load('productsFeed.makeThumbs')
							? $this->imagePipe->request($image->getFilePath(), Config::load('productsFeed.thumbsSize'), 'fit')
							: $image->getFilePath();
					}
				}

				$item += $this->getAvailabilityData($product);

				foreach ($product->getRelated() as $related) {
					$item['relatedProducts'][] = $related->getId();
				}

				$item['tags'] = [];
				if ($product->tags) {
					foreach ($product->tags as $tag) {
						$item['tags'][] = $tag->name;
					}
				}

				if ($product->isAssort)
					$item['tags'][] = 'Assort';

				if ($product->getFeatures()) {
					$item['features'] = [];
					foreach ($product->getFeatures() as $feature) {
						$item['features'][$feature->name] = $feature->value;
					}
				}

				file_put_contents($tmpFile, (!$isFirstRow ? ',"' . $item['id'] . '":' : '"' . $item['id'] . '":') . Json::encode($item), FILE_APPEND);
				$isFirstRow = false;
				$data['count']++;
			}
		}

		file_put_contents($tmpFile, '}', FILE_APPEND);
		FileSystem::rename($tmpFile, $file);
		FileSystem::delete($tmpFile);

		return $data;
	}

	public function loadAndParseProducts()
	{
		$currentSite = $this->sites->getCurrentSite();
		$site        = $currentSite->getIdent();
		$file        = WWW_DIR . '/feed-' . $site . '-' . $currentSite->getCurrentDomain()->getLang() . '.json';

		try {
			if (file_exists($file))
				return (array) Json::decode(file_get_contents($file));
		} catch (\Exception $e) {
		}

		return [];
	}
}
