<?php declare(strict_types = 1);

namespace EshopCatalog\CronModule\Model\Subscribers;

use Core\Model\Entities\EntityManagerDecorator;
use Core\Model\Helpers\Strings;
use Core\Model\Sites;
use EshopCatalog\FrontModule\Model\Categories;
use EshopCatalog\Model\Event\ProductsFeedEvent;
use Nette\Utils\Json;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;

class AdsProductsFeedSubscriber implements EventSubscriberInterface
{
	protected EntityManagerDecorator $em;

	protected Categories $categories;

	protected Sites $sites;

	public function __construct(EntityManagerDecorator $em, Categories $categories, Sites $sites)
	{
		$this->em         = $em;
		$this->categories = $categories;
		$this->sites      = $sites;
	}

	public static function getSubscribedEvents(): array
	{
		return [
			'eshopCatalog.productsFeed'   => 'productsFeed',
			'eshopCatalog.categoriesFeed' => 'categoriesFeed',
		];
	}

	public function productsFeed(ProductsFeedEvent $event): void
	{
		if ($event->customerName !== 'ads')
			return;

		$event->outputFormat = 'csv';

		$fp = fopen('php://memory', 'wb+');
		fputcsv($fp, ['Page URL', 'Custom label'], ',');
		foreach ($event->data as $row) {
			$label   = array_merge(['product'], array_map(fn($str) => Strings::webalize((string) $str), $row->category));
			$label[] = mb_strtolower($row->availability);
			if ($row->brand)
				$label[] = Strings::webalize($row->brand);

			fputcsv($fp, [
				$row->link,
				implode('-', $label),
			], ',');
		}

		rewind($fp);
		$event->data = [
			'output' => stream_get_contents($fp),
		];

		fclose($fp);
	}

	public function categoriesFeed(ProductsFeedEvent $event): void
	{
		if ($event->customerName !== 'ads')
			return;

		$event->outputFormat = 'csv';

		$currentSite = $this->sites->getCurrentSite();
		$site        = $currentSite->getIdent();
		$file        = WWW_DIR . '/feed-' . $site . '-' . $currentSite->getCurrentDomain()->getLang() . '.json';
		$hasProducts = [];

		try {
			if (file_exists($file)) {
				foreach ((array) Json::decode(file_get_contents($file)) as $row) {
					foreach ($row->categoryId as $catId)
						$hasProducts[$catId] = true;
				}
			}
		} catch (\Exception $e) {
		}

		$fp = fopen('php://memory', 'wb+');
		fputcsv($fp, ['Page URL', 'Custom label', 'Custom label 2'], ',');
		foreach ($event->data as $row) {
			$alias = Strings::webalize($row['title']);
			$stock = isset($hasProducts[$row['category_id']]) ? 'skladem' : 'vyprodane';

			fputcsv($fp, [
				$row['link'],
				implode('-', [
					'category',
					$alias,
					$stock,
				]),
				implode('-', [
					$alias,
					$stock,
				]),
			], ',');
		}

		rewind($fp);
		$event->data = [
			'output' => stream_get_contents($fp),
		];

		fclose($fp);
	}
}
