<?php declare(strict_types = 1);

namespace EshopCatalog\FrontModule\Components;

use Core\Model\Event\ControlEvent;
use Core\Model\Event\Event;
use Core\Model\Helpers\Strings;
use Core\Model\UI\BaseControl;
use Currency\Model\Exchange;
use EshopCatalog\FrontModule\Model\Categories;
use EshopCatalog\FrontModule\Model\Dao\FilterGroup;
use EshopCatalog\FrontModule\Model\Dao\FilterItem;
use EshopCatalog\FrontModule\Model\Event\FilterEvent;
use EshopCatalog\FrontModule\Model\FeatureProducts;
use EshopCatalog\FrontModule\Model\FilterService;
use EshopCatalog\FrontModule\Model\Products;
use EshopCatalog\FrontModule\Model\ProductsFacade;
use EshopCatalog\Model\Config;
use EshopCatalog\Model\Entities\ProductVariant;

/**
 * Class ProductsFilter
 * @package EshopCatalog\FrontModule\Components
 */
class ProductsFilter extends BaseControl
{
	/** @var int[] */
	public $productIds = [];

	/**
	 * ID produktů po vyfiltrování
	 *
	 * @var int[]
	 */
	protected $productIdsAfterFilter = [];

	protected ?array $resultAfterFilter = null;

	/**
	 * Vyfiltrované produkty ale bez filtru cena
	 *
	 * @var int[]
	 */
	public $productIdsNoPriceFilter;

	protected array $categories = [];

	/** @var string */
	public $searchQuery;

	/** @var FeatureProducts */
	protected $featureProducts;

	/** @var Products */
	protected $productsService;

	/** @var FilterService */
	protected $filterService;

	/** @var Categories */
	protected $categoriesService;

	/** @var ProductsFacade */
	protected $productsFacade;

	protected Exchange $exchange;

	/** @var callable */
	public $onFilter = [];

	/** @var array */
	protected $cFeatures, $cMinMax, $cManu;

	protected ?FilterGroup $cTags = null;

	protected array $sortValues = [];

	public ?array $allowedFeatures = null;

	/** @persistent */
	public $filter = [];

	/** @persistent */
	public $min;

	/** @persistent */
	public $max;

	/** @persistent */
	public $sort;

	public function __construct(FeatureProducts $featureProducts, Products $products, FilterService $filterService, Categories $categories,
	                            ProductsFacade  $productsFacade, Exchange $exchange)
	{
		$this->featureProducts   = $featureProducts;
		$this->productsService   = $products;
		$this->filterService     = $filterService;
		$this->categoriesService = $categories;
		$this->productsFacade    = $productsFacade;
		$this->exchange          = $exchange;
		$this->sortValues        = $this->getSortValues();
	}

	protected function attached($presenter): void
	{
		parent::attached($presenter);
		$do = $presenter->getParameter('do');

		if (!$do || !(Strings::contains($do, 'cartAddForm') || Strings::contains($do, 'productPreview'))) {
			if ($presenter->isAjax()) {
				if ($do && Strings::contains($do, 'formSubmit'))
					$this->formSubmit();
				$this->eventDispatcher->dispatch(new FilterEvent($this->getFilterValues()), ProductsFilter::class . '::onFilter');
				$this->redrawControl('filters');
			}

			$this->eventDispatcher->addListener(ProductsFilter::class . '::onFilter', function() {
				$this->redrawControl('filters');
			});
		}
	}

	public function getSortValues()
	{
		return [
			'recommended',
			'p.price ASC',
			'p.price DESC',
			'p.created DESC',
		];
	}

	public function render()
	{
		$ac           = $this->getFilterValues();
		$features     = $this->getFeatures();
		$minMax       = $this->getMinMax();
		$manu         = $this->getManufacturers();
		$tags         = $this->getTags();
		$filterActive = false;

		$inActiveGroups = [];
		foreach ($features as $group) {
			$hasActive = false;
			foreach ($group->items as $itemK => $item) {
				if ($item->productsCountInit === 0) {
					unset($group->items[$itemK]);
					continue;
				}
				if (isset($ac['features'][$group->id]) && in_array($item->id, $ac['features'][$group->id])) {
					$item->isActive = true;
					$hasActive      = true;
					$filterActive   = true;
				}
			}

			if (!$hasActive)
				$inActiveGroups['feature'][] = $group->id;
		}

		// Manufacturers
		$hasActive = false;
		foreach ($manu->items as $itemK => $item) {
			if ($item->productsCountInit === 0 || !$item->value) {
				unset($manu->items[$itemK]);
				continue;
			}
			if (isset($ac['manu']) && in_array($item->id, $ac['manu'])) {
				$item->isActive = true;
				$hasActive      = true;
				$filterActive   = true;
			}
		}

		if (!$hasActive)
			$inActiveGroups['manu'] = $manu->id;

		// Tags
		$hasActive = false;
		foreach ($tags->items as $itemK => $item) {
			if ($item->productsCountInit === 0 || !$item->value) {
				unset($tags->items[$itemK]);
				continue;
			}
			if (isset($ac['tags']) && in_array($item->id, $ac['tags'])) {
				$item->isActive = true;
				$hasActive      = true;
				$filterActive   = true;
			}
		}

		if (!$hasActive)
			$inActiveGroups['tags'] = $manu->id;

		if (Config::load('productPreview.basePriceWithoutVat', false)) {
			if (isset($minMax['min']))
				$minMax['min'] = round($minMax['min'] / (1 + ((float) $minMax['minVat']) / 100), 2);
			if (isset($minMax['max']))
				$minMax['max'] = round($minMax['max'] / (1 + ((float) $minMax['maxVat']) / 100), 2);
		}

		$this->template->minMax         = $minMax;
		$this->template->currentMinMax  = [
			'min' => $this->min ?: $minMax['min'],
			'max' => $this->max ?: $minMax['max'],
		];
		$this->template->inActiveGroups = $inActiveGroups;
		$this->template->features       = $features;
		$this->template->manu           = $manu;
		$this->template->tags           = $tags;
		$this->template->sort           = $this->sortValues;

		$this->template->currentSort  = $this->sort;
		$this->template->filterActive = $filterActive || $this->min || $this->max;

		$presenter = $this->getPresenter();
		$args      = $presenter->getParameters();
		unset($args['activeNavigation']);
		unset($args['action']);
		unset($args['do']);
		$this->template->resetLink = $presenter->link(':' . $presenter->getName() . ':' . $presenter->getAction(), $args);

		$this->template->render($this->getTemplateFile());
	}

	/*******************************************************************************************************************
	 * ============================== Handle
	 */

	/**
	 * @param $t
	 * @param $k
	 * @param $v
	 *
	 * @crossOrigin
	 *
	 * @throws \Nette\Application\UI\InvalidLinkException
	 */
	public function handleSet($t, $k, $v)
	{
		$this->filter['f' . $t][$k][] = $v;
		if ($this->getPresenter()->isAjax()) {
			$this->eventDispatcher->dispatch(new FilterEvent($this->getFilterValues()), ProductsFilter::class . '::onFilter');
			$this->getPresenter()->payload->url = $this->link('this');
		}
	}

	public function handlePrice($min, $max)
	{
		$this->min = $this->getMinMax()['min'] == $min ? null : $min;
		$this->max = $this->getMinMax()['max'] == $max ? null : $max;

		if ($this->getPresenter()->isAjax()) {
			$this->eventDispatcher->dispatch(new FilterEvent($this->getFilterValues()), ProductsFilter::class . '::onFilter');
			$this->getPresenter()->payload->url = $this->link('this');
		}
	}

	public function handleSort($sort)
	{
		if ($sort == 'recommended')
			$sort = null;
		$this->sort = $sort;

		if ($this->getPresenter()->isAjax()) {
			$this->eventDispatcher->dispatch(new FilterEvent($this->getFilterValues()), ProductsFilter::class . '::onFilter');
			$this->getPresenter()->payload->url = $this->link('this');
		}
	}

	public function handleFormSubmit()
	{
		$this->formSubmit();
		if ($this->getPresenter()->isAjax()) {
			$this->eventDispatcher->dispatch(new FilterEvent($this->getFilterValues()), ProductsFilter::class . '::onFilter');
			$this->getPresenter()->payload->url = $this->link('this');
		}
	}

	protected function formSubmit()
	{
		$presenter   = $this->getPresenter();
		$httpRequest = $presenter->getHttpRequest();
		$this->resetFilter();

		$catIds = $presenter->getParameter('id');
		if ($catIds) {
			if (is_array($catIds))
				$this->filter['c'] = implode('|', $catIds);
		}

		$features = $httpRequest->getPost('feature');
		if ($features) {
			foreach ($features as $g => $vals) {
				if (is_string($vals) && $vals === 'on')
					continue;
				$this->filter['ff'][$g] = implode('|', $vals);
			}
		}

		$manufacturers = $httpRequest->getPost('manu');
		if ($manufacturers) {
			foreach ($manufacturers as $g => $vals) {
				if (is_string($vals) && $vals === 'on')
					continue;
				$this->filter['fm'][$g] = implode('|', $vals);
			}
		}

		$tags = $httpRequest->getPost('tags') ?: $httpRequest->getPost('tag');
		if ($tags) {
			foreach ($tags as $g => $vals) {
				if (is_string($vals) && $vals === 'on')
					continue;
				$this->filter['ft'][$g] = implode('|', $vals);
			}
		}

		$priceRange = $httpRequest->getPost('priceRange');
		if ($priceRange) {
			if (isset($priceRange['min']) && $this->getMinMax()['min'] != $priceRange['min'])
				$this->min = $priceRange['min'];
			if (isset($priceRange['max']) && $this->getMinMax()['max'] != $priceRange['max'])
				$this->max = $priceRange['max'];
		}

		$sort = $httpRequest->getPost('sort');
		if ($sort)
			$this->sort = $sort;
	}

	/*******************************************************************************************************************
	 * ============================== Components
	 */


	/*******************************************************************************************************************
	 * ============================== Get / Set
	 */

	public function setCategories(array $categories): self
	{
		$this->categories                     = array_map(fn($c) => $c ? $c->getId() : null, $categories);
		$this->categories                     = array_filter($this->categories, 'strlen');
		FilterService::$prepareDataCategories = $this->categories;
		$this->productIds                     = $this->productsService->getProductsIdInCategory($this->categories, 0, null, $this->sort);
		$this->loadFilters();

		return $this;
	}

	public function setSearchQuery(string $search): self
	{
		$this->searchQuery = $search;
		$this->productIds  = $this->productsService->getProductsIdBySearch($search);
		$this->loadFilters();

		return $this;
	}

	protected function filterPriceCheck(array $activeFilter, array $productData)
	{
		return $activeFilter['priceRange'] && ($activeFilter['priceRange']['min'] && $productData['price'] < $activeFilter['priceRange']['min']
			|| $activeFilter['priceRange']['max'] && $productData['price'] > $activeFilter['priceRange']['max'])
			? false : true;
	}

	protected array $cCheckTag = [];

	/**
	 * Kontrola produktu jestli obsahuje aktivni tag
	 *
	 * @param array $activeFilter
	 * @param array $productData
	 *
	 * @return bool|mixed
	 */
	protected function filterCheckHasTag(array $activeFilter, array $productData)
	{
		if (!$activeFilter['tags'])
			return true;

		if (!isset($this->cCheckTag[$productData['id']])) {
			$this->cCheckTag[$productData['id']] = false;

			if ($productData['t']) {
				foreach ($activeFilter['tags'] as $tag) {
					if (in_array($tag, $productData['t'])) {
						$this->cCheckTag[$productData['id']] = true;
						break;
					}
				}
			}
		}

		return $this->cCheckTag[$productData['id']];
	}

	protected function filterManufacturerCheck(array $activeFilter, array $productData)
	{
		return !$activeFilter['manu'] || $productData['m'] && in_array($productData['m'], $activeFilter['manu']);
	}

	protected function loadFilters(): void
	{
		$this->eventDispatcher->dispatch(new ControlEvent($this), ProductsFilter::class . '::beforeLoadFilters');

		$filter       = $this->getFilterValues();
		$filterActive = !empty($filter);

		$filterNoPrice = $filter;
		unset($filterNoPrice['priceRange']);
		if (!empty($this->categories))
			$this->productIdsNoPriceFilter = $this->productsService->getProductsIdInCategory($this->categories, null, null, null, $filterNoPrice);
		else if ($this->searchQuery)
			$this->productIdsNoPriceFilter = $this->productsService->getProductsIdBySearch($this->searchQuery, null, null, null, $filterNoPrice);

		$this->getMinMax();
		$features = $this->getFeatures();
		$manu     = $this->getManufacturers();
		$tags     = $this->getTags();

		if (!$filterActive) {
			$this->productIdsAfterFilter = $this->productIds;
		} else {
			$countsP = $this->getProductsCountInFilter()['p'];
			$countsT = $this->getProductsCountInFilter()['t'];

			foreach ($this->productIds as $prodId) {
				$prod = $countsP[$prodId];

				if (!$prod)
					continue;

				// Kontrola ceny
				if (!$this->filterPriceCheck($filter, $prod))
					continue;

				if ($this->filterManufacturerCheck($filter, $prod)) {
					if ($filter['tags']) {
						$hasTag = false;

						if ($prod['t']) {
							foreach ($filter['tags'] as $fTag) {
								if (in_array($fTag, $prod['t'])) {
									$hasTag = true;
								}
							}
						}

						if (!$hasTag)
							continue;
					}

					if ($filter['features']) {
						if (isset($prod['f'])) {
							$filterGroupCountTrue = 0;

							foreach ($filter['features'] as $group => $items) {
								$canAdd = false;
								foreach ($items as $item) {
									if (in_array($item, $prod['f'])) {
										$canAdd = true;
										break;
									}
								}

								if ($canAdd) {
									$filterGroupCountTrue++;
								}
							}

							if ($filterGroupCountTrue === count($filter['features'])) {
								$this->productIdsAfterFilter[] = $prodId;
							}
						}
					} else {
						$this->productIdsAfterFilter[] = $prodId;
					}
				}
			}
		}
	}

	/**
	 * @return FilterGroup[]|null
	 */
	public function getFeatures()
	{
		if ($this->cFeatures === null) {
			$done = [];
			$ac   = $this->getFilterValues();
			if (!empty($this->categories)) {
				$cats     = array_map(fn($c) => $this->categoriesService->get((int) $c), $this->categories);
				$features = [];
				foreach ($this->categories as $catId) {
					$cat = $this->categoriesService->get((int) $catId);

					foreach ($this->filterService->getCategoryFilters((int) $catId, $cat->filtersFromParent ? true : false) as $fea) {
						/** @var FilterGroup $fea */
						if (!isset($features[$fea->id]) && $fea->name)
							$features[$fea->id] = $fea;
					}
				}

				$this->cFeatures = $features;
			} else {
				$this->cFeatures = $this->filterService->getFeatures($this->productIds);
			}

			if ($this->allowedFeatures) {
				$arr = [];
				foreach ($this->allowedFeatures as $id)
					if (isset($this->cFeatures[$id]))
						$arr[$id] = $this->cFeatures[$id];
				$this->cFeatures = $arr;
			}

			$counts  = $this->getProductsCountInFilter()['f'];
			$countsP = $this->getProductsCountInFilter()['p'];

			foreach ($this->cFeatures as $kf => $feature) {
				foreach ($feature->items as $ki => $item) {
					if (isset($counts[$ki])) {
						// Nastavení init hodnoty. Jestli se má vůbec zobrazit na frontu
						$this->cFeatures[$kf]->items[$ki]->productsCountInit = count($counts[$ki]);

						$count = 0;
						// Projdutí všech produktů a zjištění spolčných hodnot
						foreach ($counts[$ki] as $prodId) {
							if (isset($done[$prodId . '-' . $ki]))
								continue;
							$done[$prodId . '-' . $ki] = true;

							$vals = $countsP[$prodId];

							// Kontrola ceny
							if (!$this->filterPriceCheck($ac, $vals))
								continue;

							// Pokud produkt neobsahuje výrobce tak pokračovat
							if ($ac['manu'] && (!$vals['m'] || ($ac['manu'] && !in_array($vals['m'], $ac['manu']))))
								continue;

							// Kontrola jestli obsahuje tag
							if (!$this->filterCheckHasTag($ac, $vals))
								continue;

							// Pokud produkt vlastnost obsahuje tak odstraníme z $tmp a pokud je prázdný tak přičteme
							$tmp = $ac['features'];
							foreach ($ac['features'] as $fk => $fv) {
								if ($fk == $kf)
									$count++;

								foreach ($fv as $v2) {
									if (in_array($v2, $vals['f'])) {
										unset($tmp[$fk]);
										break;
									}
								}
							}

							if (empty($tmp)) {
								$count++;
							}
						}

						$this->cFeatures[$kf]->items[$ki]->productsCount = $count;
					}
				}
			}
		}

		return $this->cFeatures;
	}

	public function getMinMax()
	{
		if (!$this->cMinMax && $this->productIdsNoPriceFilter) {
			$min    = 999999999;
			$max    = 0;
			$minVat = null;
			$maxVat = null;

			foreach ($this->getProductsCountInFilter()['p'] as $prod) {
				if ($prod['price'] < $min) {
					$min    = $prod['price'];
					$minVat = $prod['vatRate'];
				}

				if ($prod['price'] > $max) {
					$max    = $prod['price'];
					$maxVat = $prod['vatRate'];
				}
			}

			$this->cMinMax = [
				'min'    => $this->exchange->change((float) $min),
				'max'    => $this->exchange->change((float) $max),
				'minVat' => $minVat,
				'maxVat' => $maxVat,
			];
		}

		return $this->cMinMax;
	}

	public function getManufacturers()
	{
		if (!$this->cManu) {
			$done = [];
			// Načteme vlastnosti pro porovnání jestli hodnoty obsahují některé produkty
			$ac      = $this->getFilterValues();
			$items   = $this->filterService->getManufacturers($this->productIds);
			$counts  = $this->getProductsCountInFilter()['m'];
			$countsP = $this->getProductsCountInFilter()['p'];

			$items[0] = (new FilterItem());
			foreach ($items as $k => $v) {
				if (isset($counts[$k]))
					$items[$k]->productsCountInit = count($counts[$k]);

				$count = 0;
				// Projdutí všech produktů výrobce
				foreach ($counts[$k] as $prodId) {
					if (isset($done[$prodId . '-' . $k]))
						continue;
					$done[$prodId . '-' . $k] = true;

					$vals = $countsP[$prodId];

					// Kontrola jestli produkt odpovídá ceně
					if (!$this->filterPriceCheck($ac, $vals))
						continue;

					// Kontrola jestli obsahuje tag
					if (!$this->filterCheckHasTag($ac, $vals))
						continue;

					// Pokud není aktivní filtr vlastnoti tak přidat a pokračovat
					if (!$ac['features']) {
						$count++;
						continue;
					}

					// Pokud produkt nemá vlastnosti tak ignorovat
					if (!$vals['f'])
						continue;

					// Pokud produkt vlastnost obsahuje tak odstraníme z $tmp a pokud je prázdný tak přičteme
					$tmp = $ac['features'];
					foreach ($ac['features'] as $fk => $fv) {
						foreach ($fv as $v2) {
							if (in_array($v2, $vals['f'])) {
								unset($tmp[$fk]);
								break;
							}
						}
					}

					if (empty($tmp)) {
						$count++;
					}
				}
				$items[$k]->productsCount = $count;
			}

			$tmp        = new FilterGroup();
			$tmp->type  = 'manu';
			$tmp->id    = 0;
			$tmp->name  = $this->t('eshopCatalogFront.filter.manufacturer');
			$tmp->items = $items;

			$this->cManu = $tmp;
		}

		return $this->cManu;
	}

	public function getTags()
	{
		if (!$this->cTags) {
			$done = [];
			// Načteme vlastnosti pro porovnání jestli hodnoty obsahují některé produkty
			$ac      = $this->getFilterValues();
			$items   = $this->filterService->getTags($this->productIds);
			$counts  = $this->getProductsCountInFilter()['t'];
			$countsP = $this->getProductsCountInFilter()['p'];

			$items[0] = (new FilterItem());
			foreach ($items as $k => $v) {
				if (isset($counts[$k]))
					$items[$k]->productsCountInit = count($counts[$k]);

				$count = 0;
				// Projdutí všech produktů výrobce
				foreach ($counts[$k] as $prodId) {
					if (isset($done[$prodId . '-' . $k]))
						continue;
					$done[$prodId . '-' . $k] = true;

					$vals = $countsP[$prodId];

					// Kontrola jestli produkt odpovídá ceně
					if (!$this->filterPriceCheck($ac, $vals) || !$this->filterManufacturerCheck($ac, $vals))
						continue;

					if (!$this->filterCheckHasTag($ac, $vals))
						continue;

					// Pokud není aktivní filtr vlastnoti tak přidat a pokračovat
					if (!$ac['features']) {
						$count++;
						continue;
					}

					// Pokud produkt nemá vlastnosti tak ignorovat
					if (!$vals['f'])
						continue;

					// Pokud produkt vlastnost obsahuje tak odstraníme z $tmp a pokud je prázdný tak přičteme
					$tmp = $ac['features'];
					foreach ($ac['features'] as $fk => $fv) {
						foreach ($fv as $v2) {
							if (in_array($v2, $vals['f'])) {
								unset($tmp[$fk]);
								break;
							}
						}
					}

					if (empty($tmp)) {
						$count++;
					}
				}
				$items[$k]->productsCount = $count;
			}
			$tagsGroup        = new FilterGroup();
			$tagsGroup->id    = 0;
			$tagsGroup->type  = 'tag';
			$tagsGroup->name  = $this->translator->translate('eshopCatalogFront.filter.tags');
			$tagsGroup->items = $items;

			$this->cTags = $tagsGroup;
		}

		return $this->cTags;
	}

	/**
	 * Vrátí ID produktů které odpovídají filtru
	 *
	 * @return array
	 */
	public function getProductIdsAfterFilter(): array
	{
		if ($this->resultAfterFilter === null) {
			$products = $this->productIdsAfterFilter
				? array_intersect($this->productIds, $this->productIdsAfterFilter)
				: ($this->getFilterValues() ? [] : $this->productIds);

			if (Config::load('groupVariantsInList', false)) {
				$allVariants = [];

				foreach ($this->em->createQueryBuilder()->select('IDENTITY(pv.product) as prod, pv.variantId, pv.isDefault')
					         ->from(ProductVariant::class, 'pv')
					         ->getQuery()->getScalarResult() as $row) {
					$allVariants[$row['prod']] = $row;
				}

				$tmp          = [];
				$usedVariants = [];

				foreach ($products as $k => $id) {
					$v = $allVariants[$id] ?? null;

					if (!$v) {
						$tmp[$id] = $id;
					} else if (!isset($usedVariants[$v['variantId']])) {
						$tmp[$id]                      = $id;
						$usedVariants[$v['variantId']] = $id;
					} else if (isset($usedVariants[$v['variantId']]) && $v['isDefault'] === 1) {
						$tmp[$usedVariants[$v['variantId']]] = $id;
					}
				}

				$this->resultAfterFilter = array_values($tmp);
			} else {
				$this->resultAfterFilter = $products;
			}
		}

		return $this->resultAfterFilter;
	}

	public function getFilterValues()
	{
		$return = [];

		if (isset($this->filter['c']))
			$return['categories'] = explode('|', $this->filter['c']);
		if (isset($this->filter['ff']))
			foreach ($this->filter['ff'] as $group => $vals)
				$return['features'][$group] = explode('|', $this->filter['ff'][$group]);
		if (isset($this->filter['fm']))
			foreach ($this->filter['fm'] as $group => $vals)
				$return['manu'] = explode('|', $this->filter['fm'][$group]);
		if (isset($this->filter['ft']))
			foreach ($this->filter['ft'] as $group => $vals)
				$return['tags'] = explode('|', $this->filter['ft'][$group]);
		if ($this->min)
			$return['priceRange']['min'] = $this->min;
		if ($this->max)
			$return['priceRange']['max'] = $this->max;

		if ($return['priceRange']['min']) {
			if (Config::load('productPreview.basePriceWithoutVat', false))
				$return['priceRange']['min'] = ((float) $return['priceRange']['min'] / 100) * 121;

			$return['priceRange']['min'] = $this->exchange->change((float) $return['priceRange']['min'], 'default', 'current');
		}

		if ($return['priceRange']['max']) {
			if (Config::load('productPreview.basePriceWithoutVat', false))
				$return['priceRange']['max'] = ((float) $return['priceRange']['max'] / 100) * 121;

			$return['priceRange']['max'] = $this->exchange->change((float) $return['priceRange']['max'], 'default', 'current');
		}

		if ($this->sort)
			$return['sort'] = $this->sort;

		return $return;
	}

	protected function resetFilter()
	{
		$this->filter = null;
		$this->min    = null;
		$this->max    = null;
		$this->sort   = null;
	}

	protected function getProductsCountInFilter()
	{
		$key = $this->categories ? 'c' . serialize($this->categories) : $this->searchQuery;

		if (!$key)
			return [];

		return $this->filterService->getProductsData($this->productIds, $key);
	}
}
