<?php declare(strict_types = 1);

namespace EshopCatalog\AdminModule\Components\Products;

use Core\Model\Helpers\Arrays;
use Core\Model\UI\Form\BaseContainer;
use EshopCatalog\AdminModule\Model\CategoryProducts;
use EshopCatalog\Model\Entities\CategoryProduct;
use EshopCatalog\Model\Entities\Product;
use Kdyby\Doctrine\EntityManager;
use Nette\Localization\ITranslator;
use Nette\Utils\ArrayHash;
use EshopCatalog\AdminModule\Model\Categories;
use EshopCatalog\Model\Entities\Category;

/**
 * Class CategoryContainer
 * @package EshopCatalog\AdminModule\Components\Products
 */
class CategoryContainer
{
	/** @var EntityManager */
	protected $em;

	/** @var ITranslator */
	protected $translator;

	/** @var Categories */
	protected $categoryServices;

	/** @var CategoryProducts */
	protected $categoryProductService;

	public function __construct(EntityManager $em, Categories $categories, CategoryProducts $categoryProducts, ITranslator $translator)
	{
		$this->em                     = $em;
		$this->categoryServices       = $categories;
		$this->translator             = $translator;
		$this->categoryProductService = $categoryProducts;
	}

	public function getContainer()
	{
		$container = new BaseContainer();

		$trees = $this->categoryServices->getTreeForSelect();

		$container->addCustomData('trees', $trees);
		$container->addCustomData('template', __DIR__ . '/CategoryContainer.latte');

		$container->addRadioList('defaultCategory', 'eshopCatalog.productForm.defaultCategory', $this->categoryServices->getOptionsForSelect());
		$container->addCheckboxList('category', 'eshopCatalog.productForm.categories', $this->categoryServices->getOptionsForSelect());

		return $container;
	}

	/**
	 * @param array|ArrayHash $values
	 *
	 * @return array
	 */
	public function getFormData($values): array
	{
		return (array) $values;
	}

	/**
	 * @param BaseContainer $container
	 * @param array         $values
	 */
	public function setDefaults(&$container, $values)
	{
		foreach ($container->getControls() as $c) {
			if (isset($values[$c->getName()])) {
				$c->setDefaultValue($values[$c->getName()]);
			}
		}
	}

	public function saveData(array $productsId, array $formCategories)
	{
		$defaultCategory          = $formCategories['defaultCategory'] ? $this->em->getReference(Category::class, $formCategories['defaultCategory']) : null;
		$currentProductCategories = [];

		foreach ($this->em->getRepository(CategoryProduct::class)->createQueryBuilder('cp')
			         ->select('IDENTITY(cp.product) as product, IDENTITY(cp.category) as category')
			         ->where('cp.product IN (:ids)')
			         ->setParameter('ids', $productsId)
			         ->getQuery()->getScalarResult() as $row)
			$currentProductCategories[$row['product']][] = $row['category'];

		foreach ($productsId as $productId) {
			/** @var Product $product */
			$product           = $this->em->getReference(Product::class, $productId);
			$productCategories = $currentProductCategories[$productId] ?: [];

			if ($defaultCategory)
				$product->setDefaultCategory($defaultCategory);

			foreach ($formCategories['category'] as $k => $v) {
				if ($v == $formCategories['defaultCategory']) {
					unset($formCategories['category'][$k]);
					break;
				}
			}

			// Smazat
			$remove = array_diff($productCategories, $formCategories['category']);
			if (!empty($remove)) {
				$this->em->createQueryBuilder()->delete(CategoryProduct::class, 'cp')
					->where('cp.product = :product')
					->andWhere('cp.category IN (:categories)')
					->setParameters([
						'product'    => $product->getId(),
						'categories' => $remove,
					])->getQuery()->execute();
			}

			// Přidat
			foreach (array_diff($formCategories['category'], $productCategories) as $id) {
				$tmp = new CategoryProduct($product, $this->em->getReference(Category::class, $id));
				$this->em->persist($tmp);
			}


			$this->em->persist($product);
		}
	}
}
