<?php declare(strict_types = 1);

namespace EshopCatalog\AdminModule\Components\AdminTools\ExportNames;

use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\ORM\Query\Parameter;
use Core\Model\UI\BaseControl;
use Core\Model\UI\DataGrid\BaseDataGrid;
use Core\Model\UI\Form\BaseForm;
use EshopCatalog\FrontModule\Model\ProductsFacade;
use EshopCatalog\Model\Config;
use EshopProductsComparison\Model\Entities\ProductExport;
use Exception;
use Nette\Http\FileUpload;
use Nette\Utils\ArrayHash;
use Nette\Utils\FileSystem;
use Nette\Utils\Json;

class FromHeureka extends BaseControl
{
	protected string $dataFilePath = DATA_DIR . '/exportNamesFromHeureka.json';

	public function __construct(
		protected ProductsFacade $productsFacade,
	)
	{
	}

	public function render(): void
	{
		$this->template->data = file_exists($this->dataFilePath) ? Json::decode((string) file_get_contents($this->dataFilePath), Json::FORCE_ARRAY) : [];

		$this->template->render($this->getTemplateFile());
	}

	protected function createComponentForm(): BaseForm
	{
		$form = $this->createForm();
		$form->setShowLangSwitcher(false);

		$form->addUpload('file', 'eshopCatalog.adminTools.exportNames.fromHeureka.file')
			->setRequired();

		$form->addSubmit('submit', 'eshopCatalog.adminTools.exportNames.fromHeureka.process');

		$form->onSuccess[] = $this->formSuccess(...);

		return $form;
	}

	public function formSuccess(BaseForm $form, ArrayHash $values): void
	{
		set_time_limit(600);

		try {
			/** @var FileUpload $file */
			$file = $values->file;

			if (!$file->isOk()) {
				throw new Exception();
			}

			$fr = fopen($file->getTemporaryFile(), 'r');

			if (!$fr) {
				return;
			}

			$i    = 0;
			$data = [];
			while (!feof($fr) && $line = fgetcsv($fr, null, ',')) {
				$i++;

				if ($i === 1) {
					continue;
				}

				$data[(int) $line[6]] = [
					'id'      => (int) $line[6],
					'newName' => trim((string) $line[2]),
				];
			}

			fclose($fr);

			foreach (array_chunk(array_keys($data), 1000) as $ids) {
				$products = $this->productsFacade->getProducts($ids);
				$dbData   = [];
				foreach ($this->em->getRepository(ProductExport::class)->createQueryBuilder('pe')
					         ->select('IDENTITY(pe.id) as id, pe.productName')
					         ->where('pe.id IN (' . implode(',', $ids) . ')')
					         ->andWhere('pe.lang = :lang')
					         ->andWhere('pe.service = :service')
					         ->setParameters(new ArrayCollection([new Parameter('lang', 'cs'), new Parameter('service', 'heureka')]))->getQuery()->getArrayResult() as $row) {
					$dbData[(int) $row['id']] = $row;
				}

				foreach ($ids as $id) {
					$row     = $dbData[$id] ?? null;
					$product = $products[$id] ?? null;

					if ($product && (!$row || !$row['productName'])) {
						$productName = $product->getName();

						if (Config::load('exportGrader.addManufacturerToName') && $product->getManufacturer()) {
							$manufacturer = $product->getManufacturer()->name;

							if (!\str_starts_with($productName, $manufacturer)) {
								$productName = trim($manufacturer . ' ' . $productName);
							}
						}

						$data[$id]['oldName'] = trim($productName);
					} else {
						$data[$id]['oldName'] = $row['productName'];
					}

					if (isset($data[$id]['newName']) && $data[$id]['newName'] === $data[$id]['oldName']) {
						unset($data[$id]);
					}
				}
			}

			file_put_contents($this->dataFilePath, Json::encode($data));
			$this->template->data = $data;
			$this->redrawControl('gridWrap');
		} catch (Exception $e) {
			bdump($e);
			$this->flashMessage($e->getMessage(), 'danger');
		}
	}

	protected function createComponentGrid(): BaseDataGrid
	{
		$grid = $this->createGrid();

		$data = file_exists($this->dataFilePath) ? Json::decode((string) file_get_contents($this->dataFilePath), Json::FORCE_ARRAY) : [];

		file_put_contents($this->dataFilePath, Json::encode($data));

		$grid->setPrimaryKey('id');
		$grid->setDataSource($data);

		$grid->addColumnLink('id', '', 'Products:edit')
			->setOpenInNewTab();
		$grid->addColumnText('oldName', 'eshopCatalog.adminTools.exportNames.fromHeureka.oldName');
		$grid->addColumnText('newName', 'eshopCatalog.adminTools.exportNames.fromHeureka.newName');

		$grid->getColumn('id')->getElementPrototype('td')->addClass('w1nw');

		$grid->addGroupAction('eshopCatalog.adminTools.exportNames.fromHeureka.changeNames')
			->onSelect[] = $this->changeNames(...);

		return $grid;
	}

	public function changeNames(array $ids): void
	{
		try {
			$data = file_exists($this->dataFilePath) ? Json::decode((string) file_get_contents($this->dataFilePath), Json::FORCE_ARRAY) : [];

			foreach ($ids as $id) {
				$this->em->getRepository(ProductExport::class)->createQueryBuilder('pe')
					->update(ProductExport::class, 'pe')
					->set('pe.productName', ':newName')
					->set('pe.product', ':newName')
					->where('pe.id = :id')
					->andWhere('pe.lang = :lang')
					->setParameters(new ArrayCollection([new Parameter('newName', $data[(int) $id]['newName']), new Parameter('id', (int) $id), new Parameter('lang', 'cs')]))->getQuery()->execute();

				unset($data[(int) $id]);
			}

			FileSystem::delete($this->dataFilePath);
			file_put_contents($this->dataFilePath, Json::encode($data));
			$this['grid']->setDataSource($data);
			$this['grid']->reload();

			sleep(1);
			$this->presenter->flashMessageSuccess('default.saved');
		} catch (Exception) {
			$this->presenter->flashMessageDanger('default.error');
		}

		$this->presenter->redrawControl('flashes');
	}
}
