<?php declare(strict_types = 1);

namespace EshopCatalog\AdminModule\Components\Categories;

use Core\AdminModule\Model\Sites;
use Core\FrontModule\Model\SeoContainer;
use Core\Model\Event\ComponentTemplateEvent;
use Core\Model\Event\CreateFormEvent;
use Core\Model\Event\FormSuccessEvent;
use Core\Model\Event\FormValidateEvent;
use Core\Model\Event\SetFormDataEvent;
use Core\Model\Helpers\EntityHelpers;
use Core\Model\Helpers\Strings;
use Core\Model\UI\AdminPresenter;
use Core\Model\UI\BaseControl;
use Core\Model\UI\Form\BaseForm;
use Core\Model\UI\Form\BootstrapRenderer;
use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\ORM\Query\Expr\Join;
use Doctrine\ORM\Query\Parameter;
use DynamicModule\Model\DynamicModuleConfig;
use DynamicModule\Model\Entities\Group;
use DynamicModule\Model\Repository\Groups;
use EshopCatalog\AdminModule\Model\Categories;
use EshopCatalog\AdminModule\Model\Features;
use EshopCatalog\AdminModule\Model\Manufacturers;
use EshopCatalog\FrontModule\Model\CacheService;
use EshopCatalog\Model\Config;
use EshopCatalog\Model\Entities\Category;
use EshopCatalog\Model\Entities\CategoryFilter;
use EshopCatalog\Model\Entities\CategoryTexts;
use EshopCatalog\Model\Entities\DynamicFeatureCategory;
use EshopCatalog\Model\Entities\Feature;
use EshopCatalog\Model\Entities\FeatureCategory;
use EshopCatalog\Model\Entities\FeatureValue;
use EshopCatalog\Model\Entities\FeatureValueTexts;
use EshopOrders\AdminModule\Model\GroupsCustomers;
use Exception;
use Navigations\AdminModule\Model\Navigations;
use Nette\Application\Attributes\Persistent;
use Nette\Application\UI\Presenter;
use Nette\Caching\Cache;
use Nette\DI\Container;
use Nette\Forms\Controls\HiddenField;
use Nette\Forms\Controls\SelectBox;
use Nette\Http\Request;
use Nette\InvalidArgumentException;
use Nette\Utils\ArrayHash;
use Nette\Utils\Json;

class CategoryForm extends BaseControl
{
	#[Persistent]
	public ?int $categoryId = null;

	public ?Category $category = null;

	private ?array $cFeatures      = null;
	private ?array $cFeatureValues = null;

	public function __construct(
		protected Container       $container,
		protected Categories      $categoryServices,
		protected SeoContainer    $seoContainerService,
		protected Sites           $sitesService,
		protected CacheService    $cacheService,
		protected Request         $request,
		protected Features        $features,
		protected GroupsCustomers $groupsCustomers,
		protected Manufacturers   $manufacturers,
		protected Navigations     $navigations,
	)
	{
	}

	public function render(): void
	{
		$features                      = $this->getFeatures();
		$this->template->features      = $features;
		$this->template->featureValues = $this->getFeatureValues();

		$defaults = [];
		if ($this->category instanceof Category) {
			foreach ($this->category->features->toArray() as $row) {
				/** @var FeatureCategory $row */
				$defaults[] = [
					'feature' => $row->getFeature()->getId(),
					'value'   => $row->getFeatureValue()->getId(),
				];
			}

			foreach ($this->category->dynamicFeatures as $row) {
				$defaults[] = [
					'feature' => $row->feature->getId(),
					'value'   => $row->value,
				];
			}
		}
		$this->template->defaultFeatures = $defaults;

		$this->template->featuresJson = Json::encode([
			'features' => $features,
			'values'   => $this->getFeatureValues(),
		]);

		$this->eventDispatcher->dispatch(new ComponentTemplateEvent($this->template, $this), CategoryForm::class . '::render');

		$this->template->thisForm = $this['form'];
		$this->template->renderer = $this['form']->getRenderer();
		$this->template->render($this->getTemplateFile());
	}

	/*******************************************************************************************************************
	 * ==================  Handle
	 */

	public function handleLoadParents(string $site): void
	{
		$presenter = $this->presenter;

		try {
			$this->loadParentItems($site);
			$presenter->flashMessageSuccess('default.loaded');
		} catch (Exception) {
			$presenter->flashMessageDanger('default.error');
		}

		$this->redrawControl('parent');
		$presenter->redrawControl('flashes');
	}

	public function handleAddFeatureValue(string $id, ?string $val): void
	{
		if (!$val) {
			$this->presenter->sendJson([]);
		}

		$exist = $this->em->getRepository(FeatureValueTexts::class)->createQueryBuilder('fvt')
			->select('IDENTITY(fvt.id) as id, fvt.name as name')
			->innerJoin('fvt.id', 'fv', Join::WITH, 'fv.feature = :feature')
			->where('fvt.name = :v')
			->setParameters(new ArrayCollection([new Parameter('feature', $id), new Parameter('v', $val)]))
			->getQuery()->getResult()[0] ?? null;

		if (!$exist) {
			$featureValue = new FeatureValue();
			$featureValue->setFeature($this->features->getReference((int) $id));
			$featureValue->isPublished = 1;
			$this->em->persist($featureValue);

			foreach ($this->langsService->getLangs(false) as $lang) {
				$featureValueText = new FeatureValueTexts($featureValue, $lang->getTag());
				$featureValueText->setName($val);
				$this->em->persist($featureValueText);
			}

			$this->em->flush();
			$exist = [
				'id'      => $featureValue->getId(),
				'name'    => $val,
				'created' => true,
			];
		}

		$this->presenter->sendJson([
			'data' => $exist,
		]);
	}

	/*******************************************************************************************************************
	 * ==================  Components
	 */

	protected function createComponentForm(): BaseForm
	{
		$form = $this->createForm();
		$form->setAjax();
		$form->getElementPrototype()->addClass('eshop-catalog-categories-form');

		$filters = [];
		foreach ($this->em->getRepository(Feature::class)->createQueryBuilder('f')->select('f.id, ft.name')
			         ->join('f.featureTexts', 'ft', 'WITH', 'ft.lang = :lang')
			         ->andWhere('f.useAsFilter = 1')
			         ->setParameters(new ArrayCollection([new Parameter('lang', $this->translator->getLocale())]))->orderBy('f.position')
			         ->getQuery()->getArrayResult() as $row) {
			$filters[$row['id']] = $row['name'];
		}

		$form->addText('name', 'eshopCatalog.categoryForm.name')
			->setMaxLength(255)->setNullable()
			->setIsMultilanguage();
		$form->addText('alias', 'eshopCatalog.categoryForm.alias')
			->setMaxLength(255)
			->setIsMultilanguage();
		$form->addText('nameH1', 'eshopCatalog.categoryForm.nameH1')
			->setMaxLength(255)
			->setIsMultilanguage();
		$form->addBool('canProductsAddToCart', 'eshopCatalog.categoryForm.canProductsAddToCart')
			->setDefaultValue(1);
		$form->addBool('isPublished', 'eshopCatalog.categoryForm.isPublished')
			->setDefaultValue(1);
		$form->addBool('hideInMobileMenu', 'eshopCatalog.categoryForm.hideInMobileMenu')
			->setIsMultilanguage()->setDefaultValue(0);
		$form->addBool('hideInDesktopMenu', 'eshopCatalog.categoryForm.hideInDesktopMenu')
			->setIsMultilanguage()->setDefaultValue(0);
		$form->addBool('disableRegisterSale', 'eshopCatalog.category.disableRegisterSale')
			->setDefaultValue(0);

		if (Config::load('category.publishedByLang')) {
			$form->getComponent('isPublished')->setIsMultilanguage();
		}

		$sites = $this->sitesService->getOptionsForSelect();
		if (count($sites) > 1) {
			$form->addSelect('site', 'eshopCatalog.categoryForm.site', $sites);
		} else {
			$form->addHidden('site', array_values($sites)[0]);
		}
		$form->getComponent('site')->setDefaultValue(array_values($sites)[0]);

		$form->addSelect('parent', $this->t('default.parent'), [])->setTranslator(null)
			->setHtmlAttribute('data-threshold', '.6');
		$form->addFilesManager('image', 'eshopCatalog.category.defaultImage');

		if (count($this->langsService->getOptionsForSelect()) > 1) {
			$form->addFilesManager('langImage', 'eshopCatalog.category.langImage')
				->setIsMultilanguage(true);
		}

		$form->addEditor('descShort', 'eshopCatalog.categoryForm.descShort')->setHeight(200)
			->setDisableAutoP(false)
			->setIsMultilanguage();
		$form->addEditor('desc', 'eshopCatalog.categoryForm.desc')->setHeight(300)
			->setDisableAutoP(false)
			->setIsMultilanguage();

		if (Config::load('enableEmptyCategoryCustomText')) {
			$form->addEditor('emptyText', 'eshopCatalog.category.emptyText')
				->setHeight(200)
				->setDisableAutoP(false)
				->setIsMultilanguage();
		}

		if (Config::load('allowCategoryCustomerGroupRestriction')) {
			$form->addCheckboxList('allowedCustomerGroups', 'eshopCatalog.category.allowedCustomerGroupsRestriction', $this->groupsCustomers->getOptionsForSelect());
		}

		$form->addBool('filtersFromParent', 'eshopCatalog.categoryForm.filtersFromParent')->setDefaultValue(1);
		if (Config::load('productsList.allowAjaxFilterLoad')) {
			$form->addBool('ajaxFilterLoad', 'eshopCatalog.category.ajaxFilterLoad')->setDefaultValue(0);
		}
		$form->addSortableCheckboxList('filters', $this->t('eshopCatalog.categoryForm.availableFilters'), $filters)->setTranslator(null);
		$form->addSortableCheckboxList('hiddenFilters', $this->t('eshopCatalog.categoryForm.availableHiddenFilters'), $filters)->setTranslator(null);

		$form->addComponent($this->seoContainerService->getContainer(true), 'seo');

		$form->addBool('disablePickupPoints', 'eshopCatalog.category.disablePickupPoints')
			->setDefaultValue(0);

		$form->addFilesManager('safetyWarningImage', 'eshopCatalog.category.safetyWarningImage')
			->setIsMultilanguage();
		$form->addEditor('safetyWarningText', 'eshopCatalog.category.safetyWarningText')
			->setIsMultilanguage();

		// Dalsi obsah
		$content = $form->addContainer('moreContent');
		$content->addSortableCheckboxList('topBrands', $this->t('eshopCatalog.categoryForm.topBrands'), $this->manufacturers->getOptionsForSelect())
			->setTranslator(null);

		$navigations = [];
		if ($this->getCategory() instanceof Category) {
			$root = $this->getCategory()->getRoot();
			if ($root instanceof Category) {
				$ident = $root->getCategoryTexts()->first()->alias ?? null;

				if ($ident) {
					$navigations = $this->navigations->getOptionsForSelectParentsInSite($ident);
				}
			}
		}

		$content->addSortableCheckboxList('topNavigationsInCategory', 'eshopCatalog.navigation.topNavigationsInCategory', $navigations);
		$content->addSortableCheckboxList('bottomNavigationsInCategory', 'eshopCatalog.navigation.bottomNavigationsInCategory', $navigations);

		// Dalsi obsah z dynamickeho modulu
		$dynamicModuleCategoryDetail = (array) Config::load('category.dynamicModuleCategoryDetailLink') ?: [];
		if (isset($dynamicModuleCategoryDetail['module']) && class_exists(DynamicModuleConfig::class)) {
			/** @var Groups|null $dmGroups */
			$dmGroups = $this->container->hasService('dynamicmodule.admin.groups')
				? $this->container->getService('dynamicmodule.admin.groups')
				: null;

			if ($dmGroups) {
				$qb = $dmGroups->getQueryBuilderByModule($dynamicModuleCategoryDetail['module'])
					->addOrderBy('g.lft', 'ASC');

				if (!DynamicModuleConfig::load('multiLangPublication')) {
					$qb->andWhere('g.isPublished = 1');
				}

				$list = [];

				foreach ($qb->getQuery()->getResult() as $g) {
					/** @var Group $g */
					$prefix = [];
					$p      = $g->getParent();

					while ($p) {
						if ($p->lvl <= 0) {
							break;
						}

						$prefix[] = $p->title;
						$p        = $p->getParent();
					}

					$list[$g->getId()] = ($prefix !== [] ? implode(' > ', array_reverse($prefix)) . ' > ' : '') . $g->title;
				}

				$content->addSortableCheckboxList('topDynamicModuleInCategory', 'eshopCatalog.navigation.topDynamicModuleInCategory', $list);
				$content->addSortableCheckboxList('bottomDynamicModuleInCategory', 'eshopCatalog.navigation.bottomDynamicModuleInCategory', $list);

				$content->addSortableCheckboxList('beforeSubcategoriesDynamicModuleInCategory', 'eshopCatalog.navigation.beforeSubcategoriesDynamicModuleInCategory', $list);
				$content->addSortableCheckboxList('afterSubcategoriesDynamicModuleInCategory', 'eshopCatalog.navigation.afterSubcategoriesDynamicModuleInCategory', $list);
			}
		}

		// Rozšířené
		if (Config::load('allowCategoryFullUrlField', false)) {
			/** @var BootstrapRenderer $renderer */
			$renderer = $form->getRenderer();
			$form->addText('fullUrl', 'eshopCatalog.categoryForm.fullUrl')
				->setDescription('eshopCatalog.categoryForm.fullUrlDesc')
				->setIsMultilanguage();
			$renderer->addToBaseExtendedLayout('right', 'fullUrl');
		}

		$this->eventDispatcher->dispatch(new CreateFormEvent($form, $this->getPresenterIfExists() instanceof Presenter ? $this->template : null), CategoryForm::class . '::createForm');

		$form->addSaveCancelControl();

		$form->onValidate[] = $this->formValidate(...);
		$form->onSuccess[]  = $this->formSuccess(...);

		$this->monitor(AdminPresenter::class, function(AdminPresenter $presenter) use ($form): void {
			$httpRequest = $this->request;

			if ($this->category) {
				$this->categoryId = $this->category->getId();
			}

			$siteSelect = $form->getComponent('site');
			if ($siteSelect instanceof SelectBox) {
				$siteSelect->setHtmlAttribute('data-link-on-change', $this->link('loadParents!', '__val__'));

				if ($httpRequest->getQuery()['site-ident']) {
					$siteSelect->setValue($httpRequest->getQuery()['site-ident']);
				}
			}

			$site = $httpRequest->getPost('site') ?: $form->getComponent('site')->getValue();

			$this->loadParentItems($site);
		});

		return $form;
	}

	public function formValidate(BaseForm $form, ArrayHash $values): void
	{
		$presenter  = $this->presenter;
		$nameFilled = false;
		foreach ($values->name as $v) {
			if ($v) {
				$nameFilled = true;
			}
		}

		if (!$nameFilled) {
			$form->addError('eshopCatalog.categoryForm.categoryNameMissing');
		}

		// Duplikace textu
		if (Config::load('allowValidateDuplicateTextsCategory')) {
			$skipCheck = (array) Config::load('disableValidateDuplicateTextForLanguage') ?: [];
			foreach (['name', 'nameH1', 'alias'] as $col) {
				$data = $values->{$col};

				if (!$data) {
					continue;
				}

				$qb = $this->categoryServices->em->getRepository(CategoryTexts::class)->createQueryBuilder('ct')
					->select('IDENTITY(ct.id) as id, IDENTITY(c.root) as root, ct.lang, ct.' . $col)
					->innerJoin('ct.id', 'c');

				if (Config::load('category.publishedByLang')) {
					$qb->andWhere('ct.isPublished = 1');
				} else {
					$qb->andWhere('c.isPublished = 1');
				}

				$orWhere   = [];
				$tIterator = 0;
				foreach ($this->langsService->getLangs(false) as $langKey => $lang) {
					if (!isset($data[$langKey]) || in_array($langKey, $skipCheck, true)) {
						continue;
					}

					$orWhere[] = "(ct.lang = '{$langKey}' AND ct.{$col} = :tIte{$tIterator})";

					$qb->setParameter('tIte' . $tIterator, $col === 'alias' ? Strings::webalize($data[$langKey]) : $data[$langKey]);

					$tIterator++;
				}

				if ($orWhere !== []) {
					$qb->andWhere(implode(' OR ', $orWhere));
				}

				if ($form->values->site) {
					$rootId = $this->categoryServices->getSiteRootId($form->values->site);

					if ($rootId !== 0) {
						$qb->andWhere('c.root = :root')
							->setParameter('root', $rootId);
					}
				}

				if ($form->values->parent) {
					$parentCat = $this->categoryServices->get((int) $form->values->parent);

					if ($parentCat instanceof Category) {
						$qb->andWhere('c.lvl = :lvl')
							->setParameter('lvl', $parentCat->getLvl() + 1);
						$qb->andWhere('c.parent = :parent')
							->setParameter('parent', $form->values->parent);
					}
				}

				foreach ($qb->getQuery()->getArrayResult() as $row) {
					if (
						!$form->getComponent($col)->getErrors()
						&& $row[$col] && (
							!$this->categoryId
							|| ((int) $this->categoryId !== $row['id'])
						)
					) {
						$form->getComponent($col)->addError($this->t('eshopCatalog.categoryForm.duplicateField', [
							'lang'  => Strings::upper($row['lang']),
							'field' => $this->translator->translate('eshopCatalog.categoryForm.' . $col),
						]));
					}
				}
			}
		}

		$this->eventDispatcher->dispatch(
			new FormValidateEvent($form, $values, $presenter ? $this->template : null, $presenter ?: null),
			CategoryFilter::class . '::validateForm'
		);

		if ($form->hasErrors()) {
			$this->redrawControl('form');
		}
	}

	public function formSuccess(BaseForm $form, ArrayHash $values): bool
	{
		try {
			$langValues = $form->convertMultilangValuesToArray();
			/** @var CategoryTexts[] $categoryTexts */
			$categoryTexts = [];

			$httpPost = $this->request->getPost();

			if ($this->categoryId) {
				$this->category = $this->categoryServices->get($this->categoryId);
				$categoryTexts  = $this->category->getCategoryTexts()->toArray();
			}

			$category = $this->category ?: new Category();

			$category->isPublished          = Config::load('category.publishedByLang')
				? (int) array_values($values->isPublished)[0]
				: $values->isPublished;
			$category->image                = $values->image;
			$category->filtersFromParent    = $values->filtersFromParent;
			$category->canProductsAddToCart = $values->canProductsAddToCart;
			$category->disablePickupPoints  = (int) $values->disablePickupPoints;
			$category->disableRegisterSale  = (int) $values->disableRegisterSale;

			if (Config::load('productsList.allowAjaxFilterLoad')) {
				$category->ajaxFilterLoad = $values->ajaxFilterLoad;
			}

			if (Config::load('allowCategoryCustomerGroupRestriction')) {
				$category->setAllowedCustomerGroup($values->allowedCustomerGroups);
			}

			$seoData      = $this->seoContainerService->getFormData($values->seo, true);
			$activeByLang = Config::load('category.publishedByLang');

			foreach ($langValues as $l => $v) {
				if (!$v['name']) {
					if (isset($categoryTexts[$l])) {
						$this->em->remove($categoryTexts[$l]);
						unset($categoryTexts[$l]);
					}

					continue;
				}

				if (!isset($categoryTexts[$l])) {
					$categoryTexts[$l] = new CategoryTexts($category, $l);
				}

				$categoryTexts[$l]->setName($v['name'])
					->setAlias($v['alias'])
					->setNameH1($v['nameH1'])
					->setSeo($seoData[$l] ?? []);
				$categoryTexts[$l]->description        = $v['desc'];
				$categoryTexts[$l]->shortDescription   = $v['descShort'];
				$categoryTexts[$l]->image              = $v['langImage'] ?: null;
				$categoryTexts[$l]->safetyWarningImage = $v['safetyWarningImage'] ?: null;
				$categoryTexts[$l]->safetyWarningText  = $v['safetyWarningText'] ?: null;
				$categoryTexts[$l]->hideInMobileMenu   = $v['hideInMobileMenu'] ? 1 : 0;
				$categoryTexts[$l]->hideInDesktopMenu  = $v['hideInDesktopMenu'] ? 1 : 0;

				if (Config::load('enableEmptyCategoryCustomText')) {
					$categoryTexts[$l]->emptyText = $v['emptyText'];
				}

				if (!$activeByLang) {
					$categoryTexts[$l]->isPublished = $category->isPublished;
				} else {
					$categoryTexts[$l]->isPublished = $v['isPublished'] && (int) $v['isPublished'] === 1
						? 1
						: 0;
				}

				$this->em->persist($categoryTexts[$l]);
			}

			$category->setCategoryTexts($categoryTexts);

			if (!$values->parent) {
				$values->parent = $this->categoryServices->getSiteRootId($values->site);
			}

			if (!$values->parent) {
				$category->setParent(null);
			} else {
				$category->setParent($this->categoryServices->get($values->parent));
			}

			$this->em->persist($category);

			// FILTRY
			$formFilters = [];
			foreach ($values->filters as $v) {
				$formFilters[(int) $v] = [
					'id'     => (int) $v,
					'hidden' => 0,
				];
			}
			foreach ($values->hiddenFilters as $v) {
				$formFilters[(int) $v] = [
					'id'     => (int) $v,
					'hidden' => 1,
				];
			}

			$categoryFilters = $category->filters->toArray();

			// nové
			foreach (array_diff_key($formFilters, $categoryFilters) as $k => $v) {
				$categoryFilter           = new CategoryFilter($category, $this->features->getReference($v['id']), $k);
				$categoryFilter->isHidden = (int) $v['hidden'];
				$this->em->persist($categoryFilter);
				$category->filters->add($categoryFilter);
			}

			// odstranit
			foreach (array_diff_key($categoryFilters, $formFilters) as $v) {
				/** @var CategoryFilter $v */
				$category->filters->removeElement($v);
				$this->em->remove($v);
			}

			// seřadit
			foreach (array_values($formFilters) as $k => $v) {
				$categoryFilter = $category->filters->get($v['id']);
				if ($categoryFilter) {
					$categoryFilter->position = $k;
					$categoryFilter->isHidden = (int) $v['hidden'];
					$this->em->persist($categoryFilter);
				}
			}

			$this->em->flush();

			//Vlastnosti
			$currentFeatures = [];
			$formFeatures    = [];
			foreach ($category->features as $fp) {
				if (isset($currentFeatures[(int) $fp->getFeatureValue()->getId()])) {
					$this->category->features->removeElement($fp);
					$this->em->remove($fp);

					continue;
				}
				$currentFeatures[(int) $fp->getFeatureValue()->getId()] = $fp;
			}

			foreach ($form->getHttpData()['features'] as $vf) {
				$formFeatures[$vf] = $vf;
			}

			$newFeatures = array_diff_key($formFeatures, $currentFeatures);
			$oldFeatures = array_diff_key($currentFeatures, $formFeatures);

			foreach ($newFeatures as $value) {
				/** @var FeatureValue $fv */
				$fv = $this->em->getReference(FeatureValue::class, $value);

				$fp = new FeatureCategory(
					$category,
					$fv,
				);

				unset($formFeatures[$value]);
				$this->em->persist($fp);
			}

			foreach ($oldFeatures as $fp) {
				$this->category->features->removeElement($fp);
				$this->em->remove($fp);
			}

			/** @var DynamicFeatureCategory[][] $currentDynamicFeatures */
			$currentDynamicFeatures = [];
			$formDynamicFeatures    = $form->getHttpData()['dynamicFeatures'] ?? [];
			foreach ($category->dynamicFeatures as $df) {
				$currentDynamicFeatures[$df->feature->getId()][$df->value] = $df;
			}

			foreach ($formDynamicFeatures as $k => $v) {
				foreach ($v as $v2) {
					$dfp        = $currentDynamicFeatures[$k][(string) $v2] ?? new DynamicFeatureCategory($category, $this->features->getReference($k), (string) $v);
					$dfp->value = (string) $v2;
					$this->em->persist($dfp);
				}
			}

			$toRemove = [];
			foreach ($currentDynamicFeatures as $featureId => $vals) {
				foreach ($vals as $k => $v) {
					if (!in_array($k, $formDynamicFeatures[$featureId] ?? [])) {
						$toRemove[] = $v->getId();
					}
				}
			}

			if ($toRemove !== []) {
				$this->em->createQueryBuilder()->delete(DynamicFeatureCategory::class, 'dfp')
					->where('dfp.id IN (' . implode(',', $toRemove) . ')')
					->getQuery()->execute();
			}

			// Extra
			if (Config::load('allowCategoryFullUrlField', false)) {
				foreach ($values->fullUrl as $l => $v) {
					if ($v) {
						$efFullUrl = EntityHelpers::setExtraField($category, 'fullUrl', $v, $l);
						$this->em->persist($efFullUrl);
					} else {
						/** @phpstan-ignore-next-line */
						$efFullUrl = $category->getExtraFieldsByKey()['fullUrl'][$l] ?? null;

						if ($efFullUrl) {
							$this->em->remove($efFullUrl);
						}
					}
				}
			}

			// Dalsi obsah
			$category->setAttr('topBrands', $values->moreContent->topBrands ?: null);
			$category->setAttr('topNavigationsInCategory', $httpPost['moreContent']['topNavigationsInCategory'] ?: null);
			$category->setAttr('topDynamicModuleInCategory', $httpPost['moreContent']['topDynamicModuleInCategory'] ?: null);
			$category->setAttr('beforeSubcategoriesDynamicModuleInCategory', $httpPost['moreContent']['beforeSubcategoriesDynamicModuleInCategory'] ?: null);
			$category->setAttr('bottomNavigationsInCategory', $httpPost['moreContent']['bottomNavigationsInCategory'] ?: null);
			$category->setAttr('bottomDynamicModuleInCategory', $httpPost['moreContent']['bottomDynamicModuleInCategory'] ?: null);
			$category->setAttr('afterSubcategoriesDynamicModuleInCategory', $httpPost['moreContent']['afterSubcategoriesDynamicModuleInCategory'] ?: null);

			$event                   = new FormSuccessEvent($form, $values,
				$this->getPresenterIfExists() instanceof Presenter ? $this->template : null, $this->getPresenterIfExists() instanceof Presenter ? $this->presenter : null);
			$event->custom['entity'] = $category;
			$this->eventDispatcher->dispatch($event, CategoryForm::class . '::formSuccess');

			$this->em->flush();
			$form->addCustomData('categoryId', $category->getId());
			$form->addCustomData('siteIdent', $values->site);
			$this->presenter->flashMessageSuccess('eshopCatalog.categoryForm.categorySaved');

			$event                   = new FormSuccessEvent($form, $values, $this->template, $this->presenter);
			$event->custom['entity'] = $category;
			$event->presenter        = $this->presenter;
			$event->control          = $this;
			$this->eventDispatcher->dispatch($event, CategoryForm::class . '::formSuccessAfterSave');

			foreach (array_keys($langValues) as $l) {
				$this->cacheService->categoryCache->remove('fullUrl_' . $l);
				$this->cacheService->categoryCache->remove('categoriesQueryPath:' . $l);

				$this->cacheStorage->remove('eshopCatalog_eshopNav_default_' . $l . '_' . $category->getId() . '_' . $values->site . '_');

				if ($category->getRoot() instanceof Category) {
					$this->cacheService->categoryCache->remove('structured_' . $l . '-' . $category->getRoot()->getId());
				}
			}

			$this->cacheService->categoryCache->remove('catFeatures/' . $category->getId());

			$this->cacheStorage->clean([Cache::Tags => ['eshopNavigation', 'navigation']]);
			$this->cacheService->categoryCache->remove('categoryFilters_' . $category->getId());
			$this->cacheService->categoryCache->remove('categoryFilters_' . $category->getId() . '_deep');
			$this->cacheService->filterCache->remove('filtersPrepareData_' . md5((string) $category->getId()));

			$this->cacheService->categoryCache->clean([Cache::Tags => [
				\EshopCatalog\FrontModule\Model\Categories::CACHE_NAMESPACE,
				'categories',
				'category/' . $category->getId(),
			]]);
		} catch (Exception $e) {
			$form->addError($e->getMessage());
			$this->redrawControl('form');

			return false;
		}

		return true;
	}

	public function getCategory(): ?Category
	{
		if ($this->categoryId && !$this->category) {
			$this->category = $this->categoryServices->get($this->categoryId);
		}

		return $this->category;
	}

	public function setCategory(int $id): void
	{
		$this->categoryId = $id;
		$this->category   = $this->categoryServices->get($id);

		if ($this->category instanceof Category) {
			$c    = $this->category;
			$form = $this['form'];

			$moreContent = [
				'topBrands'                                  => $c->getAttr('topBrands') ?: [],
				'topNavigationsInCategory'                   => $c->getAttr('topNavigationsInCategory') ?: [],
				'topDynamicModuleInCategory'                 => $c->getAttr('topDynamicModuleInCategory') ?: [],
				'beforeSubcategoriesDynamicModuleInCategory' => $c->getAttr('beforeSubcategoriesDynamicModuleInCategory') ?: [],
				'bottomNavigationsInCategory'                => $c->getAttr('bottomNavigationsInCategory') ?: [],
				'bottomDynamicModuleInCategory'              => $c->getAttr('bottomDynamicModuleInCategory') ?: [],
				'afterSubcategoriesDynamicModuleInCategory'  => $c->getAttr('afterSubcategoriesDynamicModuleInCategory') ?: [],
			];

			$defaults = [
				'isPublished'          => $c->isPublished,
				'image'                => $c->image,
				'filtersFromParent'    => $c->filtersFromParent,
				'canProductsAddToCart' => $c->canProductsAddToCart !== 0 ? 1 : 0,
				'disablePickupPoints'  => $c->disablePickupPoints,
				'moreContent'          => $moreContent,
				'disableRegisterSale'  => $c->disableRegisterSale ? 1 : 0,
			];

			if (Config::load('productsList.allowAjaxFilterLoad')) {
				$defaults['ajaxFilterLoad'] = $c->ajaxFilterLoad;
			}

			$root = $c->getRoot();
			if ($root instanceof Category) {
				$alias = $root->getCategoryText()->alias;
				$this->loadParentItems($alias);

				$siteComponent = $form->getComponent('site');
				if (($siteComponent instanceof SelectBox && array_key_exists($alias, $siteComponent->getItems()))
					|| $siteComponent instanceof HiddenField) {
					$defaults['site'] = $alias;
				}
			}

			if ($c->parent instanceof Category && array_key_exists($c->parent->getId(), $form->getComponent('parent')->getItems())) {
				$defaults['parent'] = $c->parent->getId();
			}

			if (Config::load('allowCategoryCustomerGroupRestriction')) {
				$defaults['allowedCustomerGroups'] = $c->getAllowedCustomerGroup();
			}

			$form->setDefaults($defaults);

			// Texty
			$mTexts  = [];
			$seoData = [];
			foreach ($c->getCategoryTexts() as $lang => $texts) {
				foreach ([
					         'name',
					         'alias',
					         'description'      => 'desc',
					         'shortDescription' => 'descShort',
					         'nameH1',
					         'image'            => 'langImage',
					         'safetyWarningText',
					         'safetyWarningImage',
					         'hideInMobileMenu',
					         'hideInDesktopMenu',
				         ] as $k => $v) {
					$k = is_string($k) ? $k : $v;

					/** @phpstan-ignore-next-line */
					$mTexts[$v][$lang] = $texts->$k;
				}

				foreach ($texts->getSeo() as $k => $v) {
					$seoData[$k][$lang] = $v;
				}

				if (Config::load('enableEmptyCategoryCustomText')) {
					$mTexts['emptyText'][$lang] = $texts->emptyText;
				}

				if (Config::load('category.publishedByLang')) {
					$mTexts['isPublished'][$lang] = $texts->isPublished;
				}
			}
			$form->setDefaults($mTexts);
			$this->seoContainerService->setDefaults($form->getComponent('seo'), $seoData);

			// Filtry
			$defaults       = [];
			$defaultsHidden = [];
			foreach ($c->filters->toArray() as $k => $v) {
				/** @var CategoryFilter $v */
				if ($v->isHidden) {
					if (array_key_exists($k, $form->getComponent('hiddenFilters')->getItems())) {
						$defaultsHidden[] = $k;
					}
				} else if (array_key_exists($k, $form->getComponent('filters')->getItems())) {
					$defaults[] = $k;
				}
			}
			$form->getComponent('filters')->setDefaultValue($defaults);
			$form->getComponent('hiddenFilters')->setDefaultValue($defaultsHidden);

			// Extra
			$ef    = $c->getExtraFieldsValues();
			$extra = [];

			if (Config::load('allowCategoryFullUrlField', false)) {
				$extra['fullUrl'] = $ef['fullUrl'] ?? '';
			}
			$form->setDefaults($extra);

			$this->eventDispatcher->dispatch(new SetFormDataEvent($form, $this->category), CategoryForm::class . '::setCategory');
		} else {
			throw new InvalidArgumentException();
		}
	}

	protected function loadParentItems(string $site): void
	{
		$rootId = $this->categoryServices->getSiteRootId($site);

		$this['form']->getComponent('parent')->setItems([$rootId => ' '] + ($this->categoryServices->getOptionsForSelectNested($rootId, [$this->categoryId])[$site] ?? []));
	}

	protected function getFeatures(): array
	{
		if ($this->cFeatures === null) {

			$this->cFeatures = [];
			foreach ($this->features->getEr()
				         ->createQueryBuilder('f')
				         ->select('f.id, ft.name, f.type, f.unit, f.decimals')
				         ->join('f.featureTexts', 'ft', 'WITH', 'ft.lang = :lang')
				         ->setParameter('lang', $this->translator->getLocale())
				         ->orderBy('f.position')
				         ->getQuery()
				         ->getResult() as $v) {
				$v['name'] = trim((string) $v['name']);

				if ($v['type'] === Feature::TYPE_RANGE) {
					$v['step'] = Feature::getStep((int) $v['decimals']);
				}

				$this->cFeatures[$v['id']] = $v;
			}
		}

		return $this->cFeatures;
	}

	protected function getFeatureValues(?int $id = null): array
	{
		if ($this->cFeatureValues === null) {
			$this->cFeatureValues = [];

			foreach ($this->features->em->getRepository(FeatureValue::class)
				         ->createQueryBuilder('fv')
				         ->select('fv.id, fvt.name, IDENTITY(fv.feature) as featureId')
				         ->join('fv.featureValueTexts', 'fvt', 'WITH', 'fvt.lang = :lang')
				         ->setParameter('lang', $this->translator->getLocale())
				         ->andWhere('fv.isPublished = 1')
				         ->orderBy('fv.position')
				         ->getQuery()
				         ->getResult() as $v) {
				$this->cFeatureValues[$v['featureId']][] = [
					'id'   => $v['id'],
					'name' => trim((string) $v['name']),
				];
			}
		}

		return $id ? $this->cFeatureValues[$id] ?? [] : $this->cFeatureValues;
	}
}
