<?php declare(strict_types = 1);

namespace EshopCatalog\AdminModule\Components\Categories;

use Core\Model\Event\Event;
use Core\Model\UI\BaseControl;
use EshopCatalog\AdminModule\Model\Categories;
use EshopCatalog\AdminModule\Model\RelatedCategories;
use EshopCatalog\FrontModule\Model\CacheService;
use EshopCatalog\FrontModule\Model\Categories as FrontCategories;
use EshopCatalog\Model\Entities\CategoryRelated;
use Nette\Caching\Cache;

class RelatedForm extends BaseControl
{
	public function __construct(
		public int                  $categoryId,
		protected RelatedCategories $relatedCategories,
		protected Categories        $categories,
		protected FrontCategories   $frontCategories,
		protected CacheService      $cacheService,
	)
	{
	}

	public function render(): void
	{
		$inputs = ['relatedCategory' => [
			'title'            => 'eshopCatalog.relatedForm.searchInCategories',
			'targetKey'        => 'category',
			'autocompleteName' => 'relatedCategory',
			'url'              => $this->presenter->link(":EshopCatalog:Cron:Categories:categories"),
		]];

		$event = new Event([
			'inputs' => &$inputs,
		]);
		$this->eventDispatcher->dispatch($event, RelatedForm::class . '::getInputs');

		$this->template->related    = $this->getRelated();
		$this->template->inputs     = $inputs;
		$this->template->addLink    = $this->link('add!');
		$this->template->removeLink = $this->link('remove!');
		$this->template->render($this->getTemplateFile());
	}

	/********************
	 * ==== HANDLE
	 */

	public function handleAdd(): void
	{
		$target = (int) $this->getPost('target');
		$key    = (string) $this->getPost('key');

		$result = $this->relatedCategories->add($this->categoryId, $target, $key);
		if ($result instanceof CategoryRelated) {
			$this->presenter->flashMessageSuccess('default.saved');
		} else {
			$this->presenter->flashMessageDanger('default.error');
		}
		$this->presenter->payload->result = $result instanceof CategoryRelated ? 'ok' : 'error';
		$this->presenter->payload->id     = $result->getId();
		$this->presenter->payload->key    = $this->t('eshopCatalog.relatedForm.targetKeys.' . $result->targetKey);
		$this->presenter->redrawControl('flashes');
		$this->cacheStorage->clean([Cache::TAGS => ['eshopNavigation']]);
		$this->cacheService->categoryCache->clean([Cache::TAGS => [\EshopCatalog\FrontModule\Model\Categories::CACHE_NAMESPACE]]);
	}

	public function handleRemove(): void
	{
		$id = (int) $this->getPost('id');

		$result = $this->relatedCategories->remove($id);
		if ($result) {
			$this->presenter->flashMessageSuccess('default.removed');
		} else {
			$this->presenter->flashMessageDanger('default.removeFailed');
		}
		$this->presenter->payload->result = $result ? 'ok' : 'error';

		$this->cacheStorage->clean([Cache::TAGS => ['eshopNavigation']]);
		$this->cacheService->categoryCache->clean([Cache::TAGS => [FrontCategories::CACHE_NAMESPACE]]);

		$this->presenter->redrawControl('flashes');
		$this->cacheStorage->clean([Cache::TAGS => ['eshopNavigation']]);
		$this->cacheService->categoryCache->clean([Cache::TAGS => [\EshopCatalog\FrontModule\Model\Categories::CACHE_NAMESPACE]]);
	}

	public function handleChangePosition(): void
	{
		$presenter = $this->presenter;
		$request   = $presenter->getHttpRequest();

		$id       = (int) $request->getPost('id');
		$position = (int) $request->getPost('position');

		if ($id !== 0) {
			if ($this->relatedCategories->setPosition($id, $position)) {
				$presenter->flashMessageSuccess('default.positionChanged');
			} else {
				$presenter->flashMessageDanger('default.positionChangeFailed');
			}
		}

		$this->cacheStorage->clean([Cache::TAGS => ['eshopNavigation']]);
		$this->cacheService->categoryCache->clean([Cache::TAGS => [FrontCategories::CACHE_NAMESPACE]]);

		$presenter->redrawControl('flashes');
	}

	/********************
	 * ==== GET
	 */

	public function getRelated(): array
	{
		/** @var array[] $preloaded */
		$preloaded = [];
		$arr       = [];
		$roots     = array_flip($this->frontCategories->getAllRootIds());
		$rootsById = $this->frontCategories->getRootsForIds();
		$related   = $this->relatedCategories->getByCategory($this->categoryId);

		$event = new Event([
			'related' => $related,
			'arr'     => &$preloaded,
		]);
		$this->eventDispatcher->dispatch($event, RelatedForm::class . '::getRelated');

		foreach ($related as $id => $row) {
			if (isset($preloaded[$id])) {
				$arr[$id] = $preloaded[$id];
			} else {
				$name = '';
				if ($row['targetKey'] === 'category') {
					$cat  = $this->frontCategories->getCategories($rootsById[$row['target']])[$row['target']];
					$name = $roots[$cat->getRootId()] . ' > ' . ($cat->getParent() ? $cat->getParentPathStringFlipped() . ' > ' : '') . $cat->name;
				}

				$arr[$id] = $row + [
						'name' => $name,
					];
			}
		}

		return $arr;
	}

}
