<?php declare(strict_types = 1);

namespace EshopCatalog\AdminModule\Components\Features;

use Core\Model\Entities\QueryBuilder;
use Core\Model\UI\BaseControl;
use Core\Model\UI\DataGrid\BaseDataGrid;
use Core\Model\UI\DataGrid\DataSource\DoctrineDataSource;
use Doctrine\ORM\Query\Expr\Join;
use EshopCatalog\Model\Config;
use EshopCatalog\Model\Entities\Availability;
use EshopCatalog\Model\Entities\FeatureProduct;
use EshopCatalog\Model\Entities\Product;
use EshopCatalog\Model\Entities\ProductSupplier;
use Nette\Application\Attributes\Persistent;
use Nette\Utils\Html;

class ProductsGrid extends BaseControl
{
	#[Persistent]
	public ?int $featureValueId = null;

	public function __construct()
	{
	}

	public function render(): void
	{
		if (!$this->featureValueId) {
			return;
		}

		$this->template->render($this->getTemplateFile());
	}

	protected function createComponentGrid(): BaseDataGrid
	{
		$grid = $this->createGrid();
		$grid->setRememberState(false);
		$grid->setDefaultPerPage(20);

		$qb         = $this->em->getRepository(Product::class)->createQueryBuilder('p')
			->select('p, pt')
			->join(FeatureProduct::class, 'fp', Join::WITH, 'fp.product = p.id AND fp.featureValue = :featureValue')
			->join('p.productTexts', 'pt')
			->setParameter('featureValue', $this->featureValueId)
			->orderBy('p.id', 'DESC');
		$dataSource = new DoctrineDataSource($qb, 'p.id');

		$grid->setDataSource($dataSource);

		$availabilities       = [];
		$availabilitiesSelect = [];
		foreach ($this->em->getRepository(Availability::class)->createQueryBuilder('av')
			         ->addSelect('avt')
			         ->innerJoin('av.texts', 'avt', 'WITH', 'avt.lang = :lang')
			         ->setParameter('lang', $this->translator->getLocale())
			         ->getQuery()->getArrayResult() as $row) {
			$row['texts']                  = $row['texts'][$this->translator->getLocale()];
			$availabilities[$row['id']]    = $row;
			$availabilities[$row['ident']] = $row;

			$availabilitiesSelect[$row['id']] = $row['texts']['name'];
		}

		$quantities = [];

		$dataSource->onDataLoaded[] = function(array $data) use (&$quantities): void {
			/** @var Product[] $data */

			$ids = [];

			foreach ($data as $row) {
				$ids[] = $row->getId();
			}
			if ($ids !== []) {
				foreach ($this->em->getRepository(ProductSupplier::class)->createQueryBuilder('ps')
					         ->select('IDENTITY(ps.product) as id, ps.quantity')
					         ->where('ps.product IN (:ids)')
					         ->andWhere('ps.isActive = 1')
					         ->setParameter('ids', $ids)
					         ->getQuery()->getArrayResult() as $v) {
					if (!isset($this->quantities[$v['id']])) {
						$quantities[$v['id']] = 0;
					}

					$quantities[$v['id']] += (int) $v['quantity'];
				}
			}
		};

		// Columns
		$grid->addColumnText('name', 'eshopCatalog.defaultGrid.name')->setRenderer(fn(Product $row) => Html::el('a', [
			'href'   => $this->presenter->link('Products:edit', [$row->getId()]),
			'target' => '_blank',
		])->setText($row->getText()->name))
			->setFilterText()->setCondition(function(QueryBuilder $qb, string $value): void {
				foreach (explode(',', $value) as $k => $val) {
					$qb->andWhere("pt.name LIKE :likeName{$k}")
						->setParameter('likeName' . $k, '%' . trim($val) . '%');
				}
			});

		$grid->addColumnText('ean', 'eshopCatalog.defaultGrid.ean')
			->setFilterText();

		$grid->addColumnText('codes', 'eshopCatalog.defaultGrid.codes')->setRenderer(function(Product $row) {
			$html = Html::el();

			if ($row->code1) {
				$html->addHtml(Html::el('div')->setText($row->code1));
			}
			if ($row->code2) {
				$html->addHtml(Html::el('div')
					->setText($this->t('eshopCatalog.defaultGrid.code2') . ': ' . $row->code2));
			}

			return $html;
		})
			->setFilterText()->setCondition(function(QueryBuilder $qb, string $value): void {
				foreach (explode(',', $value) as $k => $val) {
					$qb->andWhere("p.code1 LIKE :likeCode{$k} OR p.code2 LIKE :likeCode{$k}")
						->setParameter('likeCode' . $k, '%' . trim($val) . '%');
				}
			});

		$grid->addColumnText('availability', 'eshopCatalog.productGrid.availability')->setRenderer(function(Product $row) use ($availabilities) {
			$av = $row->getAvailability() instanceof Availability ? $availabilities[$row->getAvailability()->getId()] : null;

			if ($row->unlimitedQuantity !== 0) {
				$av = $availabilities['inStock'];
			}

			return $av
				? Html::el('span', [
					'class' => 'btn',
					'style' => sprintf('cursor: default; color: %s; background-color: %s;', $av['color'], $av['bgColor']),
				])->setText($av['texts']['name'])
				: '';
		})->setAlign('center')
			->setFilterSelect(['' => ''] + $availabilitiesSelect);

		$grid->addColumnNumber('quantity', 'eshopCatalog.productGrid.quantity')->setRenderer(function(Product $row) use (&$quantities) {
			if (Config::load('pseudoWarehouse') && $row->unlimitedQuantity) {
				return '∞';
			}

			$html = Html::el('');
			$html->addHtml(Html::el('span', [
				'class' => 'btn btn-xs ' . ($row->quantity > 0 ? 'btn-success' : 'btn-danger'),
			])->setText($row->quantity));

			if (isset($quantities[$row->getId()]) || Config::load('productsGrid.suppliersCountAlways', false)) {
				$html
					->addHtml(Html::el('span class=quantity-separator')->setText('/ '))
					->addHtml(Html::el('span', [
						'class' => 'btn btn-xs ' . ($quantities[$row->getId()] > 0 ? 'btn-success' : 'btn-danger'),
					])->setText($quantities[$row->getId()]));
			}

			return $html;
		});

		$grid->getColumn('ean')->getElementPrototype('td')->addClass('w1nw');
		$grid->getColumn('codes')->getElementPrototype('td')->addClass('w1nw');
		$grid->getColumn('availability')->getElementPrototype('td')->addClass('w1nw');
		$grid->getColumn('quantity')->getElementPrototype('td')->addClass('w1nw');

		return $grid;
	}
}
