<?php declare(strict_types = 1);

namespace EshopCatalog\AdminModule\Components\ImportExport;

use Core\AdminModule\Model\Sites;
use Core\Model\UI\BaseControl;
use Core\Model\UI\Form\BaseForm;
use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\ORM\Query\Expr\Join;
use Doctrine\ORM\Query\Parameter;
use EshopCatalog\AdminModule\Model\Products;
use EshopCatalog\FrontModule\Model\CacheService;
use EshopCatalog\Model\Config;
use EshopCatalog\Model\Entities\Product;
use EshopCatalog\Model\Entities\ProductTexts;
use Exception;
use Nette\Http\FileUpload;
use Nette\Http\Request;
use Nette\Utils\ArrayHash;
use Nette\Utils\DateTime;
use Nette\Utils\FileSystem;
use PhpOffice\PhpSpreadsheet\IOFactory;

class MissingText extends BaseControl
{
	public function __construct(
		protected Request      $request,
		protected Sites        $sites,
		protected CacheService $cacheService,
		protected Products     $products,
	)
	{
	}

	public function render(): void
	{
		if (!Config::load('importExport.allowMissingText')) {
			return;
		}

		$this->template->render($this->getTemplateFile());
	}

	public function createComponentExport(): BaseForm
	{
		$form = $this->createForm();
		$form->setAjax(false);
		$form->setShowLangSwitcher(false);

		$form->addSelect('defaultLang', 'eshopCatalog.missingText.defaultLang', ['' => ''] + $this->langsService->getOptionsForSelect());
		$form->addSelect('missingLang', 'eshopCatalog.missingText.missingLang', $this->langsService->getOptionsForSelect());
		$form->addCheckboxList('fields', 'eshopCatalog.missingText.productFields', [
			'name'             => $this->t('eshopCatalog.missingText.fields.name'),
			'shortDescription' => $this->t('eshopCatalog.missingText.fields.shortDescription'),
			'description'      => $this->t('eshopCatalog.missingText.fields.description'),
		]);
		$form->addCheckboxList('sites', 'eshopCatalog.missingText.sites', $this->sites->getOptionsForSelect());

		if (Config::load('importExport.missingText.defaultLang')) {
			$form->getComponent('defaultLang')->setDefaultValue(Config::load('importExport.missingText.defaultLang'));
		}

		if (Config::load('importExport.missingText.missingLang')) {
			$form->getComponent('missingLang')->setDefaultValue(Config::load('importExport.missingText.missingLang'));
		}

		$form->addSubmit('export', 'eshopCatalog.missingText.export');

		$form->onSuccess[] = $this->onSuccessExport(...);

		return $form;
	}

	public function onSuccessExport(BaseForm $form, ArrayHash $values): void
	{
		$lang        = $values->defaultLang;
		$missingLang = $values->missingLang;
		$fields      = $values->fields;
		$sites       = $values->sites;

		if (!$sites) {
			$form->addError('eshopCatalog.missingText.missingSites');
		}
		if (!$fields) {
			$form->addError('eshopCatalog.missingText.missingFields');
		}

		if ($form->hasErrors()) {
			$this->redrawControl('export');

			return;
		}

		$data = [];

		$cleanText = static function($text) {
			$text = (string) $text;

			if (Config::load('importExport.missingText.showTags') === false) {
				$text = strip_tags($text);
			}

			$text = str_replace("&nbsp;", " ", $text);
			$text = preg_replace("/&#?[a-z0-9]+;/i", " ", $text);
			$text = html_entity_decode((string) $text);

			return trim($text);
		};

		if ($lang) {
			$originText = $this->em->getRepository(Product::class)->createQueryBuilder('p', 'p.id')
				->select('p.id, p.code1, pt.name, pt.shortDescription, pt.description')
				->leftJoin('p.productTexts', 'pt', Join::WITH, 'pt.lang = :lang')
				->innerJoin('p.sites', 's', Join::WITH, 's.site IN (:sites) AND s.isActive = 1')
				->setParameters(new ArrayCollection([new Parameter('lang', $lang), new Parameter('sites', implode(',', $sites))]))
				->groupBy('p.id')
				->getQuery()->getArrayResult();
		} else {
			$originText = [];
		}

		$where = [];
		foreach ($fields as $v) {
			$where[] = "pt.{$v} IS NULL";
			$where[] = "pt.{$v} = ''";
		}

		foreach ($this->em->getRepository(Product::class)->createQueryBuilder('p')
			         ->select('p.id, p.code1, pt.name, pt.shortDescription, pt.description')
			         ->leftJoin('p.productTexts', 'pt', Join::WITH, 'pt.lang = :lang')
			         ->innerJoin('p.sites', 's', Join::WITH, 's.site IN (:sites) AND s.isActive = 1')
			         ->setParameters(new ArrayCollection([new Parameter('lang', $missingLang), new Parameter('sites', implode(',', $sites))]))
			         ->andWhere(implode(' OR ', $where))
			         ->groupBy('p.id')
			         ->getQuery()->getArrayResult() as $row) {

			$data[] = [
				$row['id'],
				$row['code1'],
				$originText[$row['id']]['name'] ?? '',
				$row['name'],
				$originText[$row['id']]['shortDescription']
					? $cleanText((string) $originText[$row['id']]['shortDescription'])
					: '',
				$cleanText((string) $row['shortDescription']),
				$originText[$row['id']]['description']
					? $cleanText((string) $originText[$row['id']]['description'])
					: '',
				$cleanText((string) $row['description']),
			];
		}

		header('Content-Encoding: UTF-8');
		header("content-type:application/csv;charset=UTF-8");
		header("Content-Disposition:attachment;filename=\"texty-" . (new DateTime())->format('Y-m-d-H-i') . ".csv\"");
		header('Content-Transfer-Encoding: binary');

		$fp = fopen('php://output', 'wb');
		if (!$fp) {
			throw new Exception('Failed open stream');
		}

		fwrite($fp, "\xEF\xBB\xBF");
		fputcsv($fp, [
			$this->t('eshopCatalog.missingText.headers.id'),
			$this->t('eshopCatalog.missingText.headers.code'),
			$this->t('eshopCatalog.missingText.headers.name', ['lang' => $lang]),
			$this->t('eshopCatalog.missingText.headers.name', ['lang' => $missingLang]),
			$this->t('eshopCatalog.missingText.headers.shortDescription', ['lang' => $lang]),
			$this->t('eshopCatalog.missingText.headers.shortDescription', ['lang' => $missingLang]),
			$this->t('eshopCatalog.missingText.headers.description', ['lang' => $lang]),
			$this->t('eshopCatalog.missingText.headers.description', ['lang' => $missingLang]),
		], ';');
		foreach ($data as $line) {
			fputcsv($fp, $line, ';');
		}

		fclose($fp);

		exit;
	}

	protected function createComponentImport(): BaseForm
	{
		$form = $this->createForm();
		$form->setShowLangSwitcher(false);
		$form->setAjax();

		$form->addSelect('missingLang', 'eshopCatalog.missingText.missingLang', $this->langsService->getOptionsForSelect());
		$form->addUpload('file', 'eshopCatalog.missingText.importFile');

		$form->addSubmit('import', 'eshopCatalog.missingText.import');

		if (Config::load('importExport.missingText.missingLang')) {
			$form->getComponent('missingLang')->setDefaultValue(Config::load('importExport.missingText.missingLang'));
		}

		$form->onSuccess[] = $this->onSuccessImport(...);

		return $form;
	}

	public function onSuccessImport(BaseForm $form, ArrayHash $values): bool
	{
		set_time_limit(900);
		$conn = $this->em->getConnection();
		$lang = $values->missingLang;

		/** @var array<int, array{name: string, shortDescription: string, description: string}> $dataForUpdate */
		$dataForUpdate = [];

		/** @var FileUpload $file */
		$file = $values->file;

		$tmp       = explode('.', $file->getUntrustedName());
		$extension = array_pop($tmp);

		try {
			if ($extension === 'csv') {
				$fr = fopen($file->getTemporaryFile(), 'r');
				if (!$fr) {
					throw new Exception('Failed open stream');
				}
				while (!feof($fr) && $line = fgetcsv($fr, null, ';')) {
					if (!is_numeric($line[0]) || empty($line[2])) {
						continue;
					}

					$id = (int) $line[0];

					$dataForUpdate[$id] = [
						'name'             => (string) $line[3],
						'shortDescription' => (string) $line[5],
						'description'      => (string) $line[7],
					];
				}
				fclose($fr);
				FileSystem::delete($file->getTemporaryFile());
			} else if (class_exists(IOFactory::class) && $extension === 'xlsx') {
				$spreadsheet = IOFactory::load($file->getTemporaryFile());
				$spreadsheet->setActiveSheetIndex(0);
				$list    = $spreadsheet->getActiveSheet();
				$lastRow = $list->getHighestRow();
				for ($r = 2; $r <= $lastRow; $r++) {
					$id = (int) $list->getCell([1, $r])->getValue();

					if ($id === 0) {
						continue;
					}

					$dataForUpdate[$id] = [
						'name'             => (string) $list->getCell([4, $r])->getValue(),
						'shortDescription' => (string) $list->getCell([6, $r])->getValue(),
						'description'      => (string) $list->getCell([8, $r])->getValue(),
					];
				}
				FileSystem::delete($file->getTemporaryFile());
			} else {
				$form->addError('eshopCatalog.missingText.importBadExtension');
				$this->redrawControl('import');

				return false;
			}

			if (!empty($dataForUpdate)) {
				foreach (array_chunk($dataForUpdate, 200, true) as $chunk) {
					/** @var ProductTexts[] $existTexts */
					$existTexts = [];

					foreach ($this->em->getRepository(ProductTexts::class)->createQueryBuilder('pt')
						         ->where('pt.lang = :lang')
						         ->setParameter('lang', $lang)
						         ->andWhere('pt.id IN (' . implode(',', array_keys($chunk)) . ')')
						         ->getQuery()->getResult() as $row) {
						/** @var ProductTexts $row */
						$existTexts[$row->getProduct()->getId()] = $row;
					}

					foreach ($chunk as $id => $data) {

						// Short description
						$shortDescription = '<p>' . $data['shortDescription'] . '</p>';

						// Description
						if ($data['description'] == strip_tags($data['description'])) {
							$tmp = explode("\n", $data['description']);

							foreach ($tmp as $k => $v) {
								$tmp[$k] = trim($v);
							}

							$description = '';
							$listStart   = false;
							foreach ($tmp as $k => $v) {
								$isListRow = strspn($v, "\t") === 1;
								$v         = trim((string) preg_replace('/\t+/', '', $v));
								if ($v === '' || $v === '0') {
									continue;
								}

								if ($isListRow) {
									if (!$listStart) {
										$listStart   = true;
										$description .= '<ul>';
									}

									$description .= '<li>' . $v . '</li>';
								} else {
									if ($listStart) {
										$description .= '</ul>';
									}

									$description .= '<p>' . $v . '</p>';
								}


								if ($listStart && !isset($tmp[$k + 1])) {
									$description .= '</ul>';
								}
							}
						} else {
							$description = $data['description'];
						}

						$text = $existTexts[$id] ?? new ProductTexts($this->products->getReference($id), $lang);
						$text->setName($data['name']);
						$text->shortDescription = $shortDescription;
						$text->description      = $description;

						$this->em->persist($text);

						$conn->update('eshop_catalog__product_variant', [
							'use_name'              => 1,
							'use_short_description' => 1,
							'use_description'       => 1,
						], [
							'product_id' => $id,
						]);
					}

					$this->em->flush();
					$this->em->clear();
				}

				$this->template->messageOk = $this->t('eshopCatalog.missingText.importDone');
				$this->redrawControl('import');

				return true;
			} else {
				$form->addError('eshopCatalog.missingText.importDataNotFound');
				$this->redrawControl('import');

				return false;
			}
		} catch (Exception $e) {
			bdump($e->getMessage());
		}

		$form->addError('eshopCatalog.missingText.importError');
		$this->redrawControl('import');

		return false;
	}
}
