<?php declare(strict_types = 1);

namespace EshopCatalog\AdminModule\Components\Manufacturers;

use Core\Model\Countries;
use Core\Model\Event\ComponentTemplateEvent;
use Core\Model\Event\CreateFormEvent;
use Core\Model\Event\FormSuccessEvent;
use Core\Model\Event\FormValidateEvent;
use Core\Model\Event\SetFormDataEvent;
use Core\Model\UI\BaseControl;
use Core\Model\UI\Form\BaseForm;
use EshopCatalog\AdminModule\Model\Manufacturers;
use EshopCatalog\Model\Entities\Manufacturer;
use EshopCatalog\Model\Entities\ManufacturerTexts;
use Exception;
use Nette\Application\BadRequestException;
use Nette\Application\UI\Presenter;
use Nette\Caching\Cache;
use Nette\Utils\ArrayHash;

class ManufacturerForm extends BaseControl
{
	public ?Manufacturer $manufacturer = null;

	public function __construct(
		protected Manufacturers $manufacturerServices,
		protected Countries     $countries,
	)
	{
	}

	public function render(): void
	{
		$this->eventDispatcher->dispatch(new ComponentTemplateEvent($this->template, $this), ManufacturerForm::class . '::render');

		$this->template->thisForm = $this['form'];
		$this->template->render($this->getTemplateFile());
	}

	protected function createComponentForm(): BaseForm
	{
		$form = $this->createForm();

		$form->addText('name', 'eshopCatalog.manufacturerForm.name')
			->setRequired()
			->setMaxLength(255);
		$form->addBool('isPublished', 'eshopCatalog.manufacturerForm.isPublished')
			->setDefaultValue(0);
		$form->addText('website', 'eshopCatalog.manufacturerForm.website')
			->setNullable()
			->setMaxLength(255);
		$form->addInteger('leadTimeDays', 'eshopCatalog.manufacturerForm.leadTimeDays')
			 ->setNullable()
			 ->addCondition($form::FILLED)
			 	->addRule($form::MIN, null, 1);
		$form->addFilesManager('logo', 'eshopCatalog.manufacturerForm.logo');
		$form->addEditor('description', 'eshopCatalog.manufacturerForm.desc')
			->setHeight(300)
			->setIsMultilanguage();
		$form->addSelect('country', 'eshopCatalog.manufacturer.country', ['' => ''] + $this->countries->getAllNameColumn());
		$form->addText('street', 'eshopCatalog.manufacturer.street');
		$form->addText('city', 'eshopCatalog.manufacturer.city');
		$form->addText('zipCode', 'eshopCatalog.manufacturer.zipCode');
		$form->addText('email', 'eshopCatalog.manufacturer.email')
			->addCondition(BaseForm::FILLED)
			->addRule(BaseForm::EMAIL);
		$form->addText('contactLink', 'eshopCatalog.manufacturer.contactLink');
		$form->addText('phone', 'eshopCatalog.manufacturer.phone');

		$this->eventDispatcher->dispatch(new CreateFormEvent($form, $this->getPresenterIfExists() instanceof Presenter ? $this->template : null), ManufacturerForm::class . '::createForm');

		$form->addSaveCancelControl();

		$form->onValidate[] = $this->formValidate(...);
		$form->onSuccess[]  = $this->formSuccess(...);

		return $form;
	}

	public function formValidate(BaseForm $form, ArrayHash $values): void
	{
		$this->eventDispatcher->dispatch(
			new FormValidateEvent($form, $values, $this->presenter ? $this->template : null, $this->presenter ?: null),
			ManufacturerForm::class . '::validateForm'
		);

		if ($form->hasErrors()) {
			$this->redrawControl('form');
		}
	}

	public function formSuccess(BaseForm $form, ArrayHash $values): bool
	{
		try {
			$langValues = $form->convertMultilangValuesToArray();
			/** @var ManufacturerTexts[] $texts */
			$texts = [];

			if ($this->manufacturer instanceof Manufacturer) {
				$manufacturer = $this->manufacturer;
				$texts        = $manufacturer->getTexts()->toArray();
			} else {
				$manufacturer = new Manufacturer;
			}

			$manufacturer->leadTimeDays = $values->leadTimeDays ? ((int) $values->leadTimeDays) : null;
			$manufacturer->name         = $values->name;
			$manufacturer->isPublished  = $values->isPublished;
			$manufacturer->website      = $values->website;
			$manufacturer->logo         = $values->logo;
			$manufacturer->country      = $values->country ? $this->countries->get($values->country) : null;
			$manufacturer->street       = $values->street;
			$manufacturer->city         = $values->city;
			$manufacturer->zipCode      = $values->zipCode;
			$manufacturer->email        = $values->email;
			$manufacturer->contactLink  = $values->contactLink;
			$manufacturer->phone        = $values->phone;

			foreach ($langValues as $l => $v) {
				if (!isset($texts[$l])) {
					$texts[$l] = new ManufacturerTexts($manufacturer, $l);
					$manufacturer->texts->set($l, $texts[$l]);
				}

				$texts[$l]->description = $v['description'];

				$this->em->persist($texts[$l]);
			}

			$this->em->persist($manufacturer);

			$event                   = new FormSuccessEvent($form, $values,
				$this->getPresenterIfExists() instanceof Presenter ? $this->template : null, $this->presenter ?: null);
			$event->custom['entity'] = $manufacturer;
			$this->eventDispatcher->dispatch($event, ManufacturerForm::class . '::formSuccess');

			$this->em->flush();

			$cache = new Cache($this->cacheStorage, 'manufacturers');
			foreach ($this->langsService->getLangs(false) as $lang) {
				$cache->remove('manufacturers/' . $lang->getTag());
			}

			$form->addCustomData('manufacturerId', $manufacturer->getId());
			$this->presenter->flashMessageSuccess('eshopCatalog.manufacturerForm.manufacturerSaved');
		} catch (Exception $e) {
			$form->addError($e->getMessage());

			return false;
		}

		return true;
	}

	/**
	 * @param int|string $id
	 *
	 * @throws BadRequestException
	 */
	public function setManufacturer($id): void
	{
		$this->manufacturer = $this->em->getRepository(Manufacturer::class)->find($id);

		if ($this->manufacturer) {
			$form = $this['form'];
			$d = [
				'name'         => $this->manufacturer->name,
				'isPublished'  => $this->manufacturer->isPublished,
				'website'      => $this->manufacturer->website,
				'logo'         => $this->manufacturer->logo,
				'country'      => $this->manufacturer->country ? $this->manufacturer->country->getId() : null,
				'street'       => $this->manufacturer->street,
				'city'         => $this->manufacturer->city,
				'zipCode'      => $this->manufacturer->zipCode,
				'email'        => $this->manufacturer->email,
				'contactLink'  => $this->manufacturer->contactLink,
				'phone'        => $this->manufacturer->phone,
				'leadTimeDays' => $this->manufacturer->leadTimeDays,
			];

			foreach ($this->manufacturer->texts->toArray() as $l => $v) {
				$d['description'][$l] = $v->description;
			}

			$form->setDefaults($d);

			$this->eventDispatcher->dispatch(new SetFormDataEvent($form, $this->manufacturer), ManufacturerForm::class . '::setManufacturer');
		} else {
			$this->presenter->error();
		}
	}

}
