<?php declare(strict_types = 1);

namespace EshopCatalog\AdminModule\Components\ProductAssign;

use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\ORM\Query\Parameter;
use Core\Model\UI\BaseControl;
use Core\Model\UI\DataGrid\BaseDataGrid;
use Doctrine\ORM\Query\Expr\Join;
use EshopCatalog\AdminModule\Model\ProductsAssignedToCustomers;
use EshopCatalog\Model\Entities\ProductAssignCustomer;

class ProductsAssignedToCustomersGrid extends BaseControl
{
	public function __construct(
		protected ProductsAssignedToCustomers     $productsAssignedToCustomers,
		protected IProductsToCustomersFormFactory $formFactory,
	)
	{
	}

	public function render(): void
	{
		$this->template->render($this->getTemplateFile());
	}

	public function handleAdd(): void
	{
		$this->template->modalTitle = $this->t('eshopCatalog.title.newProductsAssignedToCustomers');
		$this->template->modal      = 'form';
		$this->redrawControl('modal');
	}

	public function handleDelete(int $id): void
	{
		if ($this->productsAssignedToCustomers->remove($id)) {
			$this->presenter->flashMessageSuccess('default.removed');
		} else {
			$this->presenter->flashMessageDanger('default.removeFailed');
		}

		$this['grid']->reload();
		$this->presenter->redrawControl('flashes');
	}

	protected function createComponentGrid(): BaseDataGrid
	{
		$grid = $this->createGrid();

		$grid->setDataSource($this->productsAssignedToCustomers->getEr()->createQueryBuilder('pc')
			->addSelect('p, pt, c, user')
			->innerJoin('pc.product', 'p')
			->innerJoin('p.productTexts', 'pt', Join::WITH, 'pt.lang = :lang')
			->innerJoin('pc.customer', 'c')
			->innerJoin('c.user', 'user')
			->setParameters(new ArrayCollection([new Parameter('lang', $this->translator->getLocale())]))
			->orderBy('c.id')
		);

		$grid->addColumnText('customer', 'eshopCatalog.productAssignCustomer.customer')
			->setRenderer(static fn(ProductAssignCustomer $row): string => $row->getCustomer()->getUser()->getFullName());
		$grid->addColumnText('customerEmail', 'eshopCatalog.productAssignCustomer.customerEmail')
			->setRenderer(static fn(ProductAssignCustomer $row): string => $row->getCustomer()->getUser()->email);
		$grid->addColumnText('product', 'eshopCatalog.productAssignCustomer.product')
			->setRenderer(static fn(ProductAssignCustomer $row): string => $row->getProduct()->getId() . ' | ' . $row->getProduct()->getText()->name);

		$grid->addAction('delete', '', 'delete!')
			->setIcon('times')
			->setBsType('danger')
			->addClass('ajax')
			->setConfirm('default.reallyDelete');

		return $grid;
	}

	protected function createComponentForm(): ProductsToCustomersForm
	{
		$control = $this->formFactory->create();

		$control->onAnchor[] = function() use ($control): void {
			$control['form']->onSuccessSaveAndClose[] = function(): void {
				$this->presenter->payload->hideModal = true;
				$this->presenter->redrawControl('flashes');
				$this['grid']->reload();
			};

			$control['form']->getComponent('saveControl')->closeModalOnCancel();
		};

		return $control;
	}

}
