<?php declare(strict_types = 1);

namespace EshopCatalog\AdminModule\Components\ProductReplenishPlanner;

use Core\Model\Entities\QueryBuilder;
use Core\Model\UI\BaseControl;
use Core\Model\UI\DataGrid\BaseDataGrid;
use Core\Model\UI\DataGrid\DataSource\DoctrineDataSource;
use Doctrine\ORM\Query\Expr\Join;
use EshopCatalog\AdminModule\Model\ProductReplenishmentPlanner;
use EshopCatalog\Model\Config;
use EshopCatalog\Model\Entities\ProductReplenishmentPlan;
use Nette\Utils\DateTime;
use Nette\Utils\Html;

class ProductReplenishmentPlanGrid extends BaseControl
{
	protected ProductReplenishmentPlanner $productReplenishmentPlanner;

	public function __construct(ProductReplenishmentPlanner $productReplenishmentPlanner)
	{
		$this->productReplenishmentPlanner = $productReplenishmentPlanner;
	}

	public function render(): void
	{
		$this->template->render($this->getTemplateFile());
	}

	public function createComponentGrid(): BaseDataGrid
	{
		$qb = $this->productReplenishmentPlanner->getEr()->createQueryBuilder('pr');
		$qb->addSelect('m.name manufacturer , pr.lastUpdate')
			->addSelect('p.id, p.code1, p.code2, p.ean, p.quantity, pt.name productName')
			->join('pr.product', 'p')
			->join('p.productTexts', 'pt', Join::WITH, 'pt.lang = :lang')
			->join('p.manufacturer', 'm')
			->setParameter('lang', $this->translator->getLocale());

		$dataSource = new DoctrineDataSource($qb, 'p.id');

		$grid = $this->createGrid();
		$grid->setDataSource($dataSource);
		$grid->setDefaultSort(['manufacturer' => 'ASC', 'productName' => 'ASC']);
		$grid->addExportCsv('CSV Export', 'navrh_zasob.csv', 'windows-1250', ';', false, true);

		$grid->addColumnText('productName', 'eshopCatalog.defaultGrid.name', 'pt.name')
			->setTemplateEscaping(false)
			->setRenderer(fn(array $data) => Html::el('a', ['href' => $this->presenter->link(':EshopCatalog:Admin:Products:edit', [$data['id']])])->setText($data['productName']))
			->setSortable()
			->setFilterText();
		$grid->addColumnText('manufacturer', 'eshopCatalog.productReplenishmentPlanGrid.manufacturer', 'm.name')
			->setRenderer(static fn(array $data) => $data['manufacturer'])
			->setSortable()
			->setFilterText();
		$grid->addColumnText('code1', 'eshopCatalog.productReplenishmentPlanGrid.code1', 'p.code1')
			->setRenderer(static fn(array $data) => $data['code1'])
			->setFitContent()
			->setFilterText();
		$grid->addColumnText('code2', 'eshopCatalog.productReplenishmentPlanGrid.code2', 'p.code2')
			->setRenderer(static fn(array $data) => $data['code2'])
			->setFitContent()
			->setFilterText();
		$grid->addColumnText('ean', 'eshopCatalog.productReplenishmentPlanGrid.ean', 'p.ean')
			->setRenderer(static fn(array $data) => $data['ean'])
			->setFitContent()
			->setFilterText();
		$grid->addColumnText('recommendedQuantity', 'eshopCatalog.productReplenishmentPlanGrid.recommendedQuantity')
			->setRenderer(static fn(array $data) => $data[0]->recommendedQuantity . ' ks')
			->setFitContent()
			->setFilterSelect([
				''                        => '',
				'greaterThanZeroQuantity' => $this->t('eshopCatalog.productReplenishmentPlanGrid.recommendedQuantityFilter.options.greaterThanZeroQuantity'),
				'zeroQuantity'            => $this->t('eshopCatalog.productReplenishmentPlanGrid.recommendedQuantityFilter.options.zeroQuantity'),
			])
			->setCondition(function(QueryBuilder $qb, ?string $value) {
				if ($value) {
					if ($value === 'zeroQuantity') {
						$qb->andWhere('pr.recommendedQuantity <= 0');
					} else {
						if ($value === 'greaterThanZeroQuantity') {
							$qb->andWhere('pr.recommendedQuantity > 0');
						}
					}
				}
			});
		$grid->addColumnText('quantity', 'eshopCatalog.productReplenishmentPlanGrid.quantity')
			->setRenderer(static fn(array $data) => $data['quantity'] . ' ks')
			->setFitContent()
			->setFilterSelect([
				''           => '',
				'inStock'    => $this->t('eshopCatalog.productReplenishmentPlanGrid.quantityFilter.options.inStock'),
				'outOfStock' => $this->t('eshopCatalog.productReplenishmentPlanGrid.quantityFilter.options.outOfStock'),
			])
			->setCondition(function(QueryBuilder $qb, ?string $value) {
				if ($value) {
					if ($value === 'inStock') {
						$qb->andWhere('p.quantity > 0');
					} else {
						if ($value === 'outOfStock') {
							$qb->andWhere('p.quantity <= 0');
						}
					}
				}
			});

		foreach (Config::load('productReplenishmentPlanner.statistics', []) as $k => $col) {
			$grid->addColumnText($col, 'admin.productReplenishmentPlanGrid.' . $col)
				->setTemplateEscaping(false)
				->setRenderer(static function(array $data) use ($col) {
					/** @var ProductReplenishmentPlan $entity */
					$entity = $data[0];

					if (!$entity->salesStatistics || !array_key_exists($col, $entity->salesStatistics)) {
						return '';
					}

					$pieces   = $entity->salesStatistics[$col]['pieces'] . ' ks';
					$dateFrom = DateTime::createFromFormat('Y-m-d', $entity->salesStatistics[$col]['dateFrom']);
					$dateTo   = DateTime::createFromFormat('Y-m-d', $entity->salesStatistics[$col]['dateTo']);

					return sprintf('%s <br /><span style="white-space: nowrap">%s - %s</span>', $pieces, $dateFrom->format($dateFrom->format('Y') === $dateTo->format('Y') ? 'j.n.' : 'j.n.Y'), $dateTo->format('j.n.Y'));
				});
		}

		$grid->addColumnDateTime('lastUpdate', 'eshopCatalog.productReplenishmentPlanGrid.lastUpdate')
			->setFormat('d.m.Y H:i:s');

		return $grid;
	}

}
