<?php declare(strict_types = 1);

namespace EshopCatalog\AdminModule\Components\Products;

use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\ORM\Query\Parameter;
use Core\Model\UI\BaseControl;
use Core\Model\UI\Form\BaseForm;
use Doctrine\ORM\Query\Expr\Join;
use EshopCatalog\AdminModule\Model\Products;
use EshopCatalog\FrontModule\Model\CacheService;
use Exception;
use Nette\Application\IPresenter;
use Nette\Caching\Cache;
use Nette\Utils\ArrayHash;

class SetAsVariantsFor extends BaseControl
{
	public function __construct(
		protected Products     $products,
		protected CacheService $cacheService,
	)
	{
	}

	public function render(): void
	{
		$this->template->thisForm = $this['form'];
		$this->template->render($this->getTemplateFile());
	}

	public function handleLoadProducts(): void
	{
		$output = [];

		foreach ($this->products->getEr()->createQueryBuilder('p')
			         ->select('p.id, p.code1, p.ean, pt.name')
			         ->innerJoin('p.productTexts', 'pt', Join::WITH, 'pt.lang = :lang')
			         ->leftJoin('p.isVariant', 'isVar')
			         ->where('isVar IS NULL OR isVar.isDefault = 1')
			         ->andWhere('p.isDeleted = 0')
			         ->setParameters(new ArrayCollection([new Parameter('lang', $this->translator->getLocale())]))->getQuery()->getArrayResult() as $row) {
			$output[$row['id']] = [
				'id'    => (string) $row['id'],
				'name'  => (string) $row['name'],
				'code1' => (string) $row['code1'],
				'ean'   => (string) $row['ean'],
			];
		}

		$this->presenter->sendJson($output);
	}

	public function createComponentForm(): BaseForm
	{
		$form = $this->createForm();

		$form->addHidden('productId');
		$form->addText('product', 'eshopCatalog.product.searchProduct')
			->setHtmlAttribute('data-autocomplete-name', 'productsSetAsVariantsFor')
			->setHtmlAttribute('data-autocomplete-target', $form->getComponent('productId')->getHtmlId())
			->setHtmlAttribute('data-autocomplete-keys', 'id,code1,ean,name');

		$this->monitor(IPresenter::class, function() use ($form): void {
			$form->getComponent('product')->setHtmlAttribute('data-autocomplete-url', $this->link('loadProducts!'));
		});

		$form->addSaveCancelControl();

		$form->onSuccess[] = $this->formSuccess(...);

		return $form;
	}

	public function formSuccess(BaseForm $form, ArrayHash $values): bool
	{
		$productsIds = $this->presenter->getParameter('ids');
		$form->addCustomData('productIds', $productsIds);
		$presenter = $this->presenter;

		if ($productsIds) {
			foreach (explode('-', (string) $productsIds) as $prodId) {
				try {
					$this->products->setAsVariantFor((int) $prodId, (int) $values->productId);
					$this->em->flush();
					$this->em->clear();
				} catch (Exception $e) {
					bdump($e->getMessage());
					$presenter->flashMessageDanger($e->getMessage());
				}
			}

			$presenter->flashMessageSuccess('default.saved');

			$this->cacheService->productCache->clean([
				Cache::Tags => ['variants'],
			]);
		}

		return true;
	}

}
