<?php declare(strict_types = 1);

namespace EshopCatalog\AdminModule\Components\Search;

use Core\AdminModule\Model\Sites;
use Core\Model\UI\BaseControl;
use Core\Model\UI\DataGrid\BaseDataGrid;
use Core\Model\UI\Form\BaseForm;
use DateTime;
use Nette\Application\Attributes\Persistent;
use Nette\Utils\ArrayHash;

class HistoryGrid extends BaseControl
{
	#[Persistent]
	public ?string $from = null;

	#[Persistent]
	public ?string $to = null;

	public function __construct(
		protected Sites $sites
	)
	{
		$this->from = (new DateTime)->modify('-6 months')->format('Y-m-d');
		$this->to   = (new DateTime)->format('Y-m-d');
	}

	public function render(): void
	{
		$this->template->render($this->getTemplateFile());
	}

	protected function createComponentFromToForm(): BaseForm
	{
		$form = $this->createForm();
		$form->setAjax();

		$form->addDatePicker('from', '')
			->setHtmlAttribute('onChange', '$(this.form).closest(\'form\').first().submit();')
			->setDefaultValue($this->from);
		$form->addDatePicker('to', '')
			->setHtmlAttribute('onChange', '$(this.form).closest(\'form\').first().submit();')
			->setDefaultValue($this->to);

		$form->addSubmit('update', 'default.update');

		$form->onSuccess[] = function(BaseForm $form, ArrayHash $values): void {
			$this->from = $values->from->format('Y-m-d');
			$this->to   = $values->to->format('Y-m-d');

			$this['grid']->reload();
			$this->redrawControl('wrap');
		};

		return $form;
	}

	protected function createComponentGrid(): BaseDataGrid
	{
		$grid = $this->createGrid();
		$grid->setRememberState(true);

		$data = [];
		foreach ($this->em->getConnection()->fetchAllAssociative("SELECT * FROM eshop_catalog__search_history WHERE created_at >= :from AND created_at <= :to", [
			'from' => $this->from,
			'to'   => $this->to,
		]) as $row) {
			$key = $row['site_id'] . '|' . $row['text'];
			if (!isset($data[$key])) {
				$data[$key] = [
					'text'  => $row['text'],
					'views' => 0,
					'site'  => $row['site_id'],
				];
			}

			$data[$key]['views'] += (int) $row['views'];
		}

		$grid->setDataSource($data);
		$grid->setDefaultSort(['views' => 'DESC']);

		$grid->addColumnText('site', 'eshopCatalog.searchHistory.site')
			->setFilterSelect(['' => ''] + $this->sites->getOptionsForSelect(true));
		$grid->addColumnText('text', 'eshopCatalog.searchHistory.text')
			->setSortable()
			->setFilterText();
		$grid->addColumnNumber('views', 'eshopCatalog.searchHistory.views')
			->setSortable();

		$grid->getColumn('site')->getElementPrototype('th')->addClass('w1nw');
		$grid->getColumn('views')->getElementPrototype('th')->addClass('w1nw');
		$grid->getColumn('views')->getElementPrototype('td')->addClass('w1nw');

		return $grid;
	}
}
