<?php declare(strict_types = 1);

namespace EshopCatalog\AdminModule\Components\Sorting\Products;

use Core\Model\Sites;
use Core\Model\UI\BaseControl;
use Core\Model\UI\DataGrid\BaseDataGrid;
use Core\Model\UI\DataGrid\DataSource\DoctrineDataSource;
use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\ORM\Query\Expr\Join;
use Doctrine\ORM\Query\Parameter;
use EshopCatalog\AdminModule\Model\AvailabilityService;
use EshopCatalog\AdminModule\Model\Categories;
use EshopCatalog\Model\Config;
use EshopCatalog\Model\Entities\Product;
use Nette\Application\LinkGenerator;
use Nette\Http\Session;
use Nette\Utils\Html;

class ProductsGrid extends BaseControl
{
	protected ?array $cCategories = null;

	public function __construct(
		protected LinkGenerator       $linkGenerator,
		protected Categories          $categories,
		protected Sites               $sites,
		protected AvailabilityService $availabilityService,
		protected Session             $session,
	)
	{
	}

	public function render(): void
	{
		$this->template->render($this->getTemplateFile());
	}

	protected function createComponentGrid(): BaseDataGrid
	{
		$grid = $this->createGrid();
		$grid->setDefaultPerPage(50);
		$grid->setItemsPerPageList([20, 50, 100, 200], false);
		$grid->setOuterFilterRendering(true);

		$grid->setSortable();
		$grid->setSortableHandler($this->getName() . ':gridSortableRow!');

		$categories = $this->getCategories();
		$data       = $this->getGridData();

		$grid->setPrimaryKey('id');
		$grid->setDataSource($data);

		$grid->addColumnText('isPublished', 'eshopCatalog.productGrid.isPublished')->setRenderer(function(array $row): Html {
			$wrap = Html::el('div class=grid-langs');
			bdump($row);
			$wrap->setAttribute('style', 'justify-content: center;');
			$wrap->addHtml(Html::el('span')
				->setAttribute('class', 'btn btn-xs ' . ($row['isPublished'] ? 'btn-success' : 'btn-danger'))
				->addHtml(Html::el('i')
					->setAttribute('class', 'fa ' . ($row['isPublished'] ? 'fa-check' : 'fa-times'))
				)
			);

			return $wrap;
		});
		$grid->addColumnText('name', 'eshopCatalog.product.productName')
			->setRenderer(fn(array $row): Html => Html::el('a', ['target' => '_blank'])
				->href($this->linkGenerator->link('EshopCatalog:Admin:Products:edit', ['id' => $row['id']]))
				->setText($row['name']));
		$grid->addColumnText('categoriesNames', 'eshopCatalog.category.categoryName')
			->setRenderer(fn(array $row): Html => Html::fromHtml(implode('<br>', $row['categoriesNames'])))
			->setFilterSelect(['' => ''] + $categories)
			->setCondition(function($rows, $catId): array {
				foreach ($rows as $k => $v) {
					if (!in_array($catId, $v['catsIds'], true)) {
						unset($rows[$k]);
					}
				}

				return $rows;
			});
		$grid->addColumnText('position', 'eshopCatalog.product.position')
			->setRenderer(function(array $row): Html {
				$value = $row['position'] === 999999 ? '' : $row['position'];

				return Html::fromHtml('<input data-sorting-products data-product="' . $row['id'] . '" type="number" name="product-' . $row['id'] . '" value="' . $value . '" style="width: 60px; text-align: right; height: 20px; padding: 0;">');
			})->setAlign('right');

		$grid->getColumn('isPublished')->getElementPrototype('th')->addClass('w1nw');
		$grid->getColumn('position')->getElementPrototype('th')->addClass('w1nw');

		$grid->addFilterSelect('availability', 'eshopCatalog.product.availability', ['' => ''] + $this->availabilityService->getOptionsForSelect())
			->setCondition(function($rows, $value): array {
				foreach ($rows as $k => $v) {
					if ($v['availability'] !== $value) {
						unset($rows[$k]);
					}
				}

				return $rows;
			});
		$grid->addFilterSelect('isPublished', 'eshopCatalog.product.isPublished', [
			''  => '',
			'1' => $this->t('default.yes'),
			'0' => $this->t('default.no'),
		])
			->setCondition(function($rows, $value): array {
				foreach ($rows as $k => $v) {
					if ($v['isPublished'] !== (int) $value) {
						unset($rows[$k]);
					}
				}

				return $rows;
			});

		return $grid;
	}

	public function handleGridSortableRow(?int $id, ?int $position): void
	{
		$presenter = $this->presenter;
		$request   = $presenter->getHttpRequest();
		$id        = $id ?: (int) $request->getPost('id');

		$filters = [];

		$sessionName    = $this['grid']->getSessionSectionName();
		$sessionSection = $this->session->getSection($sessionName);
		foreach ($this['grid']->assembleFilters() as $k => $v) {
			$value = $sessionSection->get($k);

			$v->setValue($value);

			$filters[$k] = $v;
		}

		/** @var DoctrineDataSource $dataSource */
		$dataSource = $this['grid']->getDataSource();

		$dataSource->filter($filters);

		if ($request->getPost('position') !== null) {
			$position = (int) $request->getPost('position');
			$position++;
		}

		$last = null;
		foreach ($dataSource->getData() as $v) {
			if ($v['position'] && $v['position'] !== 999999 && $v['position'] >= $position) {
				if ($v['position'] > $last && $last !== null) {
					break;
				}

				$pos = $v['position'] + 1;

				$this->changePosition($v['id'], $pos);

				$last = $pos;
			}
		}

		$this->changePosition($id, $position);
		$this->em->flush();

		$this->presenter->flashMessageSuccess('default.saved');
		$this->presenter->redrawControl('flashes');
		$this->reloadGrid();
	}

	protected function changePosition(int $productId, ?int $position): void
	{
		$entity = $this->em->find(Product::class, $productId);
		if ($entity) {
			$entity->position = $position;
			$this->em->persist($entity);
		}
	}

	protected function getGridData(): array
	{
		$categories = $this->getCategories();
		$oneSite    = count($this->sites->getSites()) === 1;

		$data = [];

		$qb = $this->em->getRepository(Product::class)->createQueryBuilder('p')
			->select('p.id, pt.name, IDENTITY(p.availability) as availability, IDENTITY(s.site) as site, IDENTITY(s.category) as categoryId, GROUP_CONCAT(IDENTITY(c.category)) as cats, CASE WHEN p.position IS NULL THEN 999999 ELSE p.position END AS position')
			->innerJoin('p.productTexts', 'pt', Join::WITH, 'pt.lang = :lang')
			->leftJoin('p.sites', 's')
			->leftJoin('p.categoryProducts', 'c')
			->setParameters(new ArrayCollection([new Parameter('lang', $this->translator->getLocale())]))
			->orderBy('position', 'ASC')
			->groupBy('p.id');

		if (Config::load('product.publishedByLang')) {
			$qb->addSelect('pt.isPublished');
		} else {
			$qb->addSelect('p.isPublished');
		}

		foreach ($qb->getQuery()->getArrayResult() as $row) {
			$cats            = $row['cats'] ? explode(',', (string) $row['cats']) : [];
			$catsIds         = array_merge([$row['categoryId']], $cats);
			$catsIds         = array_map('intval', $catsIds);
			$categoriesNames = [];

			if ($oneSite) {
				$category = ($categories[$row['categoryId']] ?? '');

				foreach ($catsIds as $catId) {
					$categoriesNames[] = ($categories[$catId] ?? '');
				}
			} else {
				$category = ($categories[$row['site']][$row['categoryId']] ?? '');

				foreach ($catsIds as $catId) {
					$categoriesNames[] = ($categories[$row['site']][$row['categoryId']] ?? '');
				}
			}

			$data[$row['id']] = [
				'id'              => $row['id'],
				'name'            => $row['name'],
				'site'            => $row['site'],
				'catsIds'         => $catsIds,
				'categoriesNames' => $categoriesNames,
				'category'        => $category,
				'categoryId'      => $row['categoryId'],
				'position'        => $row['position'],
				'availability'    => $row['availability'] ?? '',
				'isPublished'     => $row['isPublished'],
			];
		}

		return $data;
	}

	protected function getCategories(): array
	{
		if ($this->cCategories === null) {
			$this->cCategories = $this->categories->getOptionsForSelectNested();
			$oneSite           = count($this->sites->getSites()) === 1;

			if ($oneSite) {
				$this->cCategories = array_values($this->cCategories)[0];
			}
		}

		return $this->cCategories;
	}

	public function reloadGrid(): void
	{
		$this['grid']->setDataSource($this->getGridData());
		$this['grid']->reload();
	}
}
