<?php declare(strict_types = 1);

namespace EshopCatalog\AdminModule\Components\Tags;

use Core\Model\Helpers\Strings;
use Core\Model\UI\BaseControl;
use Core\Model\UI\Form\BaseForm;
use EshopCatalog\AdminModule\Model\Tags;
use EshopCatalog\FrontModule\Model\CacheService;
use EshopCatalog\Model\Entities\Tag;
use EshopCatalog\Model\Entities\TagText;
use Exception;
use Nette\Application\Attributes\Persistent;
use Nette\Application\BadRequestException;
use Nette\Utils\ArrayHash;

class TagForm extends BaseControl
{
	#[Persistent]
	public ?int $tagId = null;

	public ?Tag $tag = null;

	public function __construct(
		protected Tags         $tagServices,
		protected CacheService $cacheService,
	)
	{
	}

	public function render(): void
	{
		$this->template->render($this->getTemplateFile());
	}

	protected function createComponentForm(): BaseForm
	{
		$form = $this->createForm();
		$form->setAjax();

		$form->addText('type', 'eshopCatalog.tagForm.type')
			->setMaxLength(255);
		$form->addText('name', 'eshopCatalog.tagForm.name')
			->setIsMultilanguage()
			->setMaxLength(255);
		$form->addBool('hideProductTagName', 'eshopCatalog.tagForm.hideProductTagName');
		$form->addText('description', 'eshopCatalog.tagForm.description')
			->setIsMultilanguage()
			->setMaxLength(255);
		$form->addFilesManager('image', 'default.image');
		$form->addColorInput('color', 'default.textColor');
		$form->addColorInput('bgColor', 'default.bgColor');
		$form->addInteger('ppcPriority', 'eshopCatalog.tag.ppcPriority');

		$form->addSaveCancelControl();

		$form->onValidate[] = $this->formValidate(...);
		$form->onSuccess[]  = $this->formSuccess(...);

		return $form;
	}

	public function formValidate(BaseForm $form, ArrayHash $values): void
	{
		$nameFilled = false;
		foreach ($values->name as $v) {
			if ($v) {
				$nameFilled = true;
			}
		}

		if (!$nameFilled) {
			$form->addError('eshopCatalog.tagForm.nameMissing');
		}

		$existTagTypes = [];
		foreach ($this->em->getRepository(Tag::class)->createQueryBuilder('t')
			         ->select('t.id, t.type')
			         ->getQuery()->getScalarResult() as $tag) {
			if ($tag['id'] !== $this->tagId) {
				$existTagTypes[$tag['type']] = $tag['type'];
			}
		}

		if (isset($existTagTypes[$values->type])) {
			$form->getComponent('type')->addError('eshopCatalog.tagForm.typeExists');
		}

		if ($form->hasErrors()) {
			$this->redrawControl('form');
		}
	}

	public function formSuccess(BaseForm $form, ArrayHash $values): bool
	{
		try {
			$langValues = $form->convertMultilangValuesToArray();
			/** @var TagText[] $tagTexts */
			$tagTexts = [];

			if ($this->tagId) {
				$tag      = $this->tagServices->get($this->tagId);
				$tagTexts = $tag->getTexts()->toArray();
			} else {
				$tag = new Tag;
			}

			$type                    = Strings::toAscii($values->type);
			$type                    = str_replace(' ', '_', $type);
			$tag->type               = $type;
			$tag->image              = $values->image;
			$tag->hideProductTagName = (int) $values->hideProductTagName;
			$tag->color              = $values->color;
			$tag->bgColor            = $values->bgColor;
			$tag->ppcPriority        = $values->ppcPriority ?: null;
			$this->em->persist($tag);

			foreach ($langValues as $l => $v) {
				if ($v['name'] === '') {
					continue;
				}

				if (!isset($tagTexts[$l])) {
					$tagTexts[$l] = new TagText($tag, $l);
				}

				$tagTexts[$l]->name        = $v['name'];
				$tagTexts[$l]->description = $v['description'];
				$this->em->persist($tagTexts[$l]);
			}
			$tag->setTexts($tagTexts);

			$this->em->persist($tag);
			$this->em->flush();
			$form->addCustomData('tagId', $tag->getId());
			$form->addCustomData('tagType', $tag->type);
			$this->presenter->flashMessageSuccess('eshopCatalog.tagForm.saved');

			$this->cacheService->clearTagsCache();
		} catch (Exception $e) {
			$form->addError($e->getMessage());

			return false;
		}

		return true;
	}

	/**
	 * @param int|string $type
	 *
	 * @throws BadRequestException
	 */
	public function setTag($type): void
	{
		$this->tag = is_int($type) ? $this->tagServices->get($type) : $this->tagServices->getByType($type);

		if ($this->tag) {
			$this->tagId = $this->tag->getId();
			$form        = $this['form'];
			$defaults    = [
				'type'               => $this->tag->type,
				'image'              => $this->tag->image,
				'color'              => $this->tag->color,
				'bgColor'            => $this->tag->bgColor,
				'hideProductTagName' => $this->tag->hideProductTagName,
				'ppcPriority'        => $this->tag->ppcPriority,
			];

			foreach ($this->langsService->getLangs(false) as $lang) {
				$langTag = $lang->getTag();
				$texts   = $this->tag->getText($langTag);

				if ($texts) {
					$defaults['name'][$langTag]        = $texts->name;
					$defaults['description'][$langTag] = $texts->description;
				}
			}

			$form->setDefaults($defaults);
		} else {
			$this->presenter->error();
		}
	}

}
