<?php declare(strict_types = 1);

namespace EshopCatalog\AdminModule\Components\VatRates;

use Core\Model\Exceptions\EntityHasRelations;
use Core\Model\UI\BaseControl;
use Core\Model\UI\DataGrid\BaseDataGrid;
use Core\Model\UI\DataGrid\DataSource\DoctrineDataSource;
use EshopCatalog\AdminModule\Model\VatRates;
use EshopCatalog\Model\Entities\Product;
use EshopCatalog\Model\Entities\ProductPrice;
use EshopCatalog\Model\Entities\VatRate;

class VatRatesGrid extends BaseControl
{
	public function __construct(
		protected VatRates            $vatRateServices,
		protected IVatRateFormFactory $vatRateFormFactory,
	)
	{
	}

	public function render(): void
	{
		$this->template->render($this->getTemplateFile());
	}

	/*******************************************************************************************************************
	 * ==================  Handle
	 */

	public function handleAdd(): void
	{
		$this->template->modalTitle = $this->t('eshopCatalog.title.newVatRate');
		$this->template->modal      = 'form';
		$this->redrawControl('modal');
	}

	public function handleEdit(int $id): void
	{
		$this['form']->id = $id;
		$this['form']->setVatRate($id);
		$this->template->modalTitle = $this->t('eshopCatalog.title.editVatRate',
			['title' => $this['form']->vatRate->name]);
		$this->template->modal      = 'form';
		$this->redrawControl('modal');
	}

	public function handleDelete(int $id): void
	{
		$presenter = $this->presenter;
		try {
			if ($this->vatRateServices->remove($id)) {
				$presenter->flashMessageSuccess('eshopCatalog.defaultGrid.removed');
			} else {
				$presenter->flashMessageDanger('eshopCatalog.defaultGrid.removeFailed');
			}
		} catch (EntityHasRelations $e) {
			$presenter->flashMessageDanger($e->getMessage());
		}

		if ($presenter->isAjax()) {
			$this['grid']->reload();
			$presenter->redrawControl('flashes');
		} else {
			$presenter->redirect('this');
		}
	}

	/*******************************************************************************************************************
	 * ==================  Components
	 */

	protected function createComponentGrid(): BaseDataGrid
	{
		$grid = $this->createGrid();
		$grid->hideTableFooter();

		$assignedProducts = [];
		$qb               = $this->vatRateServices->getEr()->createQueryBuilder('vr')
			->orderBy('vr.position');

		$dataSource = new DoctrineDataSource($qb, 'vr.id');

		$grid->setSortable();
		$grid->setSortableHandler('vatRatesGrid:gridSortableRow!');

		$dataSource->onDataLoaded[] = function($items) use (&$assignedProducts): void {
			foreach ($this->em->getRepository(Product::class)->createQueryBuilder('p')
				         ->select('COUNT(p.id) as count, IDENTITY(p.vatRate) as vatRate')
				         ->where('p.vatRate IS NOT NULL')
				         ->groupBy('p.vatRate')
				         ->getQuery()->getArrayResult() as $item) {
				$assignedProducts[$item['vatRate']] = (int) $item['count'];
			}

			foreach ($this->em->getRepository(ProductPrice::class)->createQueryBuilder('pp')
				         ->select('COUNT(pp.product) as count, IDENTITY(pp.vatRate) as vatRate')
				         ->groupBy('pp.vatRate')
				         ->where('pp.vatRate IS NOT NULL')
				         ->getQuery()->getArrayResult() as $item) {
				$assignedProducts[$item['vatRate']] += (int) $item['count'];
			}
		};

		$grid->setDataSource($dataSource);

		// Columns
		$grid->addColumnText('name', 'eshopCatalog.defaultGrid.name');
		$grid->addColumnNumber('rate', 'eshopCatalog.defaultGrid.value');
		$grid->addColumnNumber('assignedProducts', 'eshopCatalog.defaultGrid.assignedProducts')
			->setRenderer(function(VatRate $vatRate) use (&$assignedProducts) {
				return number_format((float) ($assignedProducts[$vatRate->getId()] ?? 0), 0, '', ' ');
			})->setAlign('right');

		// Actions
		$grid->addAction('edit', '', 'edit!')->setIcon('edit')->setBsType('primary')
			->addClass('ajax');
		$grid->addAction('delete', '', 'delete!')->setIcon('times')->setBsType('danger')
			->addClass('ajax')
			->setConfirm('default.reallyDelete');

		// Columns prototype

		return $grid;
	}

	protected function createComponentForm(): VatRateForm
	{
		$control = $this->vatRateFormFactory->create();
		if ($this->getParameter('id')) {
			$control->setVatRate((int) $this->getParameter('id'));
		}

		$control['form']->onSuccessSave[]         = function(): void {
			$this->presenter->redrawControl('flashes');
			$this['grid']->reload();
		};
		$control['form']->onSuccessSaveAndClose[] = function(): void {
			$this->presenter->payload->hideModal = true;
			$this->presenter->redrawControl('flashes');
			$this['grid']->reload();
		};
		$control['form']->getComponent('saveControl')->closeModalOnCancel();

		return $control;
	}

	public function handleGridSortableRow(): void
	{
		$presenter = $this->presenter;
		$request   = $presenter->getHttpRequest();
		$id        = $request->getPost('id');
		$position  = $request->getPost('position');

		if ($id != null && $position != null && $this->vatRateServices->setPosition((int) $id, (int) $position)) {
			$presenter->flashMessageSuccess('default.positionChanged');
		} else {
			$presenter->flashMessageDanger('default.positionChangeFailed');
		}

		$presenter->redrawControl('flashes');
		$this['grid']->reload();
	}
}
