<?php declare(strict_types = 1);

namespace EshopCatalog\AdminModule\Model;

use Core\Model\Entities\EntityManagerDecorator;
use Core\Model\Translation\TextTranslator;
use EshopCatalog\Model\Entities\Product;
use EshopCatalog\Model\Entities\ProductTexts;

class CatalogTranslator
{
	public function __construct(
		protected TextTranslator         $translator,
		protected EntityManagerDecorator $em,
		protected Products               $productServices,
	)
	{
	}

	public function translateProducts(
		array  $products,
		string $targetLang,
		string $sourceLang = 'cs',
		bool   $translateOnlyEmpty = false
	): void
	{
		foreach (array_chunk($products, 200) as $productsChunk) {
			$data = [];

			foreach ($this->em->getRepository(ProductTexts::class)->findBy([
				'id'   => $productsChunk,
				'lang' => [$sourceLang, $targetLang],
			]) as $row) {
				/** @var ProductTexts $row */
				if (!$row->getProduct()) {
					continue;
				}

				$data[$row->getProduct()->getId()][$row->getLang()] = $row;
			}

			foreach ($data as $productId => $texts) {
				/** @var ?ProductTexts $sourceTexts */
				$sourceTexts = $texts[$sourceLang] ?? null;
				if (!$sourceTexts) {
					continue;
				}

				/** @var ?ProductTexts $targetTexts */
				$targetTexts = $texts[$targetLang] ?? null;
				if (!$targetTexts) {
					/** @var Product $product */
					$product = $this->em->getReference(Product::class, $productId);

					$targetTexts = new ProductTexts($product, $targetLang);
				}

				if ($sourceTexts->name && (!$translateOnlyEmpty || !$targetTexts->name)) {
					$targetTexts->setName($this->translator->translateText($sourceTexts->name, $sourceLang, $targetLang));
				}

				if ($sourceTexts->name2 && (!$translateOnlyEmpty || !$targetTexts->name2)) {
					$targetTexts->setName2($this->translator->translateText($sourceTexts->name2, $sourceLang, $targetLang));
				}

				if ($sourceTexts->shortDescription && (!$translateOnlyEmpty || !$targetTexts->shortDescription)) {
					$targetTexts->shortDescription = $this->translator->translateHtml($sourceTexts->shortDescription, $sourceLang, $targetLang);
				}

				if ($sourceTexts->description && (!$translateOnlyEmpty || !$targetTexts->description)) {
					$targetTexts->description = $this->translator->translateHtml($sourceTexts->description, $sourceLang, $targetLang);
				}

				$this->em->persist($targetTexts);
			}

			$this->em->flush();

			foreach (array_keys($data) as $prodId) {
				$this->productServices->clearProductCache((int) $prodId);
			}
		}
	}
}
