<?php declare(strict_types = 1);

namespace EshopCatalog\AdminModule\Model;

use Contributte\Translation\Translator;
use Core\Model\Application\AppState;
use Core\Model\Dao\Site;
use Core\Model\Entities\Redirect;
use Core\Model\Entities\Repository\NestedTreeRepository;
use Core\Model\Event\Event;
use Core\Model\Event\EventDispatcher;
use Core\Model\Helpers\Arrays;
use Core\Model\Helpers\BaseEntityService;
use Core\Model\Helpers\Traits\TPublish;
use Core\Model\Http\Session;
use Core\Model\Lang\Langs;
use Core\Model\Sites;
use Core\Model\TemplateReader\TemplateTextTypesCollection;
use Core\Model\TemplateTextType\Link;
use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\ORM\Query\Expr\Join;
use Doctrine\ORM\Query\Parameter;
use DynamicModule\FrontModule\Model\Repository\Groups;
use DynamicModule\FrontModule\Model\Repository\IGroupsFactory;
use DynamicModule\Model\Repository\Members;
use EshopCatalog\FrontModule\Model\CacheService;
use EshopCatalog\FrontModule\Model\ProductsFacade;
use EshopCatalog\Model\Config;
use EshopCatalog\Model\Entities;
use EshopCatalog\Model\Entities\Category;
use EshopCatalog\Model\Entities\CategoryTexts;
use EshopCatalog\Model\Entities\Product;
use Exception;
use Navigations\Model\Entities\Navigation;
use Nette\Application\LinkGenerator;
use Nette\Caching\Cache;
use Nette\DI\Container;
use Nette\Http\SessionSection;
use Nette\Http\Url;
use Nette\Utils\Json;
use Override;

/**
 * @method Category|null getReference($id)
 * @method Category[] getAll()
 * @method NestedTreeRepository getEr()
 */
class Categories extends BaseEntityService
{
	use TPublish;

	public static bool $hidePrefixAndSuffix = false;

	/** @var Session @inject */
	public $session;

	protected $entityClass = Category::class;

	protected ?array $cForTree     = null;
	protected ?array $cRootIds     = null;
	protected ?array $cSiteRootIds = null;

	public function __construct(
		protected CacheService    $cacheService,
		protected Translator      $translator,
		protected Langs           $langs,
		protected LinkGenerator   $linkGenerator,
		protected Sites           $sites,
		protected Container       $container,
		protected EventDispatcher $eventDispatcher,
	)
	{
	}

	public function getSessionSection(): SessionSection
	{
		return $this->session->getSection('EshopCatalog/Admin/Categories');
	}

	#[Override]
	public function get($id): ?Category
	{
		return $this->getEr()->createQueryBuilder('c')->addSelect('ct')
			->leftJoin('c.categoryTexts', 'ct')
			->where('c.id = :id')->setParameter('id', $id)
			->getQuery()->getOneOrNullResult();
	}

	public function getRootIds(): array
	{
		if ($this->cRootIds === null) {
			$this->cRootIds = [];

			foreach ($this->em->getRepository(Category::class)->createQueryBuilder('c')
				         ->select('c.id, ct.alias')
				         ->where('c.lvl = 0 AND c.id = c.root')
				         ->innerJoin('c.categoryTexts', 'ct')
				         ->getQuery()->getArrayResult() as $row) {
				if ($row['alias']) {
					$this->cRootIds[$row['id']] = $row['alias'];
				}
			}
		}

		return $this->cRootIds;
	}

	public function getSiteRootId(?string $siteIdent = null): int
	{
		if ($siteIdent) {
			if ($this->cSiteRootIds === null) {
				$this->cSiteRootIds = array_flip($this->getRootIds());
			}

			$rootId = $this->cSiteRootIds[$siteIdent] ?? null;
		} else {
			$rootId = $this->em->getRepository(Category::class)->createQueryBuilder('c')
				->select('c.id')
				->where('c.lvl = 0 AND c.id = c.root')
				->getQuery()->setMaxResults(1)->getArrayResult()[0]['id'] ?? null;
		}

		if (!$rootId) {
			$category = new Category;
			$this->em->persist($category);

			foreach ($this->langs->getLangs(false) as $lang) {
				$categoryText = new CategoryTexts($category, $lang->getTag());
				$categoryText->setAlias($siteIdent);
				$this->em->persist($categoryText);
			}

			$this->em->flush();
			$rootId = $category->getId();

			if ($this->cSiteRootIds !== null) {
				$this->cSiteRootIds[$siteIdent] = $rootId;
			}
		}

		return (int) $rootId;
	}

	/**
	 * @param int|string $id
	 *
	 * @throws Exception
	 */
	public function removeCategory($id): bool
	{
		$this->em->beginTransaction();
		try {
			$category = $this->get($id);

			if (!$category instanceof Category) {
				throw new Exception("category '$id' not found");
			}

			if ($category->getParent() instanceof Category) {
				$siteIdent                   = $this->getRootIds()[$category->getRoot()->getId()];
				$site                        = $this->sites->getSites()[$siteIdent];
				Sites::$currentIdentOverride = $siteIdent;

				$existRedirects = [];
				foreach ($this->em->createQueryBuilder()
					         ->select('r.id, r.relationValue, r.siteIdent, r.lang')
					         ->from(Redirect::class, 'r')
					         ->where('r.package = :package')
					         ->andWhere('r.relationKey = :relationKey')
					         ->andWhere('r.relationValue = :relationValue')
					         ->setParameters(new ArrayCollection([new Parameter('package', 'EshopCatalog'), new Parameter('relationKey', 'Category'), new Parameter('relationValue', $id)]))
					         ->getQuery()
					         ->getArrayResult() as $row) {
					$existRedirects[$row['lang']] = $row['id'];
				}

				foreach ($site->getDomains() as $domain) {
					$lang = $domain->getLang();

					if (isset($existRedirects[$lang])) {
						continue;
					}

					Sites::$currentLangOverride = $lang;

					$oldUrl = $this->linkGenerator->link('EshopCatalog:Front:Default:category', [
						'id'     => $id,
						'locale' => $lang,
					]);
					$oldUrl = new Url($oldUrl);
					$oldUrl = ltrim($oldUrl->getPath(), '/');

					$newUrl = $this->linkGenerator->link('EshopCatalog:Front:Default:category', [
						'id'     => $category->getParent()->getId(),
						'locale' => $lang,
					]);
					$newUrl = new Url($newUrl);
					$newUrl = ltrim($newUrl->getPath(), '/');

					if ($oldUrl !== $newUrl) {
						$redirect                = new Redirect($category->getCategoryText()->name, $oldUrl, $newUrl);
						$redirect->package       = 'EshopCatalog';
						$redirect->relationKey   = 'Category';
						$redirect->relationValue = (string) $category->getId();
						$redirect->siteIdent     = $siteIdent;
						$redirect->lang          = $lang;

						$this->em->persist($redirect);
					}
				}

				$this->em->flush();

				Sites::$currentIdentOverride = null;
				Sites::$currentLangOverride  = null;
			}

			foreach ($this->em->getRepository(CategoryTexts::class)->findBy(['id' => $id]) as $ct) {
				$this->em->remove($ct);
			}
			$this->em->remove($category);
			$this->em->flush();

			$this->em->commit();

			$cache = new Cache($this->cacheStorage, Redirect::CACHE_NAMESPACE);
			$cache->clean([Cache::All => true]);

			return true;
		} catch (Exception) {
			if ($this->em->getConnection()->isTransactionActive()) {
				$this->em->rollback();
			}
		}

		return false;
	}

	public function cleanCacheDeep(int $categoryId): void
	{
		$cat = $this->get($categoryId);

		if (!$cat instanceof Category) {
			return;
		}

		$tags = ['category/' . $cat->getId(), 'hierarchy'];
		$loop = static function(array $childs) use (&$loop): void {
			foreach ($childs as $child) {
				$tags[] = 'category/' . $child->getId();

				if ($child->children->toArray()) {
					$loop($child->children->toArray());
				}
			}
		};

		$loop($cat->children->toArray());

		$this->cacheService->categoryCache->clean([Cache::TAGS => $tags]);
	}

	protected function prepareDataForTree(?string $siteIdent = null): array
	{
		$rootIds = $this->getRootIds();
		$cKey    = $siteIdent ?: array_values($rootIds)[0];

		if ($this->cForTree === null) {
			$this->cForTree = [];

			$byRootId = [];
			$qb       = $this->getEr()->createQueryBuilder('c')
				->select('c.id, ct.name, c.lvl, IDENTITY(c.parent) as parent, IDENTITY(c.root) as rootId')
				->leftJoin('c.categoryTexts', 'ct', 'WITH', 'ct.lang = :lang')
				->setParameter('lang', $this->translator->getLocale())
				->groupBy('c.id')
				->addOrderBy('c.root')->addOrderBy('c.lft');

			if (Config::load('category.publishedByLang')) {
				$qb->addSelect('ct.isPublished as isPublished');
			} else {
				$qb->addSelect('c.isPublished as isPublished');
			}

			foreach ($qb->getQuery()->getArrayResult() as $row) {
				$byRootId[$row['rootId']][] = $row;
			}

			foreach ($byRootId as $rootId => $rows) {
				$roots      = [];
				$categories = [];
				$flat       = [];

				foreach ($rows as $c) {
					if ($c['lvl'] == 0) {
						$roots[$c['id']] = [
							'id'   => $c['id'],
							'name' => $c['name'],
						];
						continue;
					}

					$name = !$c['isPublished'] && !self::$hidePrefixAndSuffix ? '[x ' . $c['name'] . ' x]' : $c['name'];

					$categories[$c['id']] = $name;
					$flat[]               = [
						'id'     => $c['id'],
						'parent' => $c['parent'],
						'name'   => $name,
					];
				}

				$this->cForTree[$rootIds[$rootId]] = [
					'roots'      => $roots,
					'categories' => $categories,
					'flat'       => $flat,
				];
			}

			$byRootId = null;
		}

		return $this->cForTree[$cKey] ?? [
			'roots'      => [],
			'categories' => [],
			'flat'       => [],
		];
	}

	public function getOptionsForSelect(?string $siteIdent = null): array { return $this->prepareDataForTree($siteIdent)['categories']; }

	public function getOptionsForSelectNested(?int $rootId = null, array $skip = [], bool $onlyPublished = false): array
	{
		$roots = $this->getRootIds();

		$categories = [];
		$qb         = $this->getEr()->createQueryBuilder('c')->select('c.id, c.lvl, ct.name, IDENTITY(c.root) as root')
			->andWhere('c.root IN (:root)')
			->andWhere('c.lvl > 0')
			->join('c.categoryTexts', 'ct', Join::WITH, 'ct.lang = :lang')
			->setParameters(new ArrayCollection([new Parameter('lang', $this->translator->getLocale()), new Parameter('root', $rootId ? [$rootId] : array_keys($roots))]))
			->groupBy('c.id')
			->addOrderBy('c.lft');

		if ($onlyPublished) {
			if (Config::load('category.publishedByLang')) {
				$qb->andWhere('ct.isPublished = 1');
			} else {
				$qb->andWhere('c.isPublished = 1');
			}
		}

		$titles  = [];
		$lastLvl = 0;
		foreach ($qb->getQuery()->getArrayResult() as $category) {
			if ($category['lvl'] === 1) {
				$titles = [];
			}

			if ($category['lvl'] <= $lastLvl) {
				$titles = array_slice($titles, 0, $category['lvl'], true);
			}

			$titles[$category['lvl']] = $category['name'];
			$lastLvl                  = $category['lvl'];

			if (Arrays::contains($skip, $category['id'])) {
				continue;
			}

			$categories[$roots[$category['root']]][$category['id']] = $category['id'] . ' - ' . implode(' > ', $titles);
		}

		return $categories;
	}

	public function getFlatTree(?string $siteIdent = null): array { return $this->prepareDataForTree($siteIdent)['flat']; }

	public function getTreeForSelect(?string $siteIdent = null): array
	{
		$data = $this->prepareDataForTree($siteIdent);

		$trees = [];
		foreach ($data['roots'] as $id => $root) {
			$trees[$id] = [
				'name'     => $root['name'],
				'children' => Arrays::buildTree($data['flat'], 'parent', 'id', $root['id']),
			];
		}

		return $trees;
	}

	public function invertPublish(int $id, ?string $lang = null): bool
	{
		$category = $this->get($id);

		if (!$category || ($lang && !$category->getCategoryText($lang))) {
			return false;
		}

		if ($lang) {
			$category->getCategoryText($lang)->isPublished = (int) !$category->getCategoryText($lang)->isPublished;
			$this->em->persist($category);
		} else {
			$category->isPublished = (int) !$category->isPublished;
			$this->em->persist($category);

			foreach ($category->getCategoryTexts() as $text) {
				$text->isPublished = $category->isPublished;
				$this->em->persist($text);
			}
		}

		$this->em->flush();

		foreach ($this->langs->getLangs(false) as $v) {
			$this->cacheStorage->remove('eshopCatalog_eshopNav_default_' . $v->getTag() . '_' . $category->getId() . '_' . $this->getRootIds()[$category->getRoot()->getId()] . '_');
		}

		$this->cacheStorage->clean([Cache::TAGS => ['eshopNavigation', 'navigation']]);
		$this->cacheService->categoryCache->clean([Cache::TAGS => [
			\EshopCatalog\FrontModule\Model\Categories::CACHE_NAMESPACE,
			'categories',
		]]);

		return true;
	}

	public function clearProductsNotDefaultCategoryNested(string $siteIdent, array $ids): bool
	{
		if (!Config::load('category.allowClearProductsNotDefault')) {
			return false;
		}

		$trees           = [];
		$categories      = [];
		$updatedProducts = [];

		try {
			foreach (array_chunk($ids, 200) as $chunk) {
				foreach ($this->getEr()->createQueryBuilder('c')
					         ->select('c.id, c.lft, c.gt, IDENTITY(c.root) as root')
					         ->where('c.id IN (' . implode(',', $chunk) . ')')
					         ->getQuery()->getArrayResult() as $row) {
					$trees[] = [
						'root' => $row['root'],
						'lft'  => $row['lft'],
						'rgt'  => $row['gt'],
					];
				}
			}

			foreach (array_chunk($trees, 100) as $chunk) {
				$arr = [];

				foreach ($chunk as $v) {
					$arr[] = "(c.root = {$v['root']} AND c.lft >= {$v['lft']} AND c.gt <= {$v['rgt']})";
				}

				foreach ($this->getEr()->createQueryBuilder('c')
					         ->select('c.id')
					         ->where(implode(' OR ', $arr))
					         ->getQuery()->getArrayResult() as $row) {
					$categories[] = $row['id'];
				}
			}

			if ($categories !== []) {
				foreach ($this->em->getRepository(Product::class)->createQueryBuilder('p')
					         ->innerJoin('p.categoryProducts', 'cp', Join::WITH, 'cp.category IN (' . implode(',', $categories) . ')')
					         ->getQuery()->getResult() as $product) {
					/** @var Entities\Product $product */
					/** @var Entities\ProductInSite|null $site */
					$site = $product->sites[$siteIdent] ?? null;
					if ($site && $site->category && Arrays::contains($categories, $site->category->getId())) {
						continue;
					}

					foreach ($categories as $catId) {
						$cat = $product->categoryProducts->get($catId);
						if ($cat) {
							$product->categoryProducts->remove($catId);
							$this->em->remove($cat);
						}
					}

					$this->em->persist($product);
					$this->em->flush();

					$updatedProducts[] = $product->getId();
				}
			}

			if ($updatedProducts !== []) {
				foreach ($ids as $id) {
					$this->cacheService->clearProduct((int) $id);
				}
			}
		} catch (Exception) {
			return false;
		}

		return true;
	}

	public function checkActiveProducts(array $ids, ProductsFacade $productsFacade, bool $useCleaner = false): array
	{
		$conn   = $this->em->getConnection();
		$result = [];
		$i      = 0;

		foreach (array_chunk($ids, 250) as $chunk) {
			$catsProducts = [];

			foreach ($conn->executeQuery("SELECT id_product, id_category 
							FROM eshop_catalog__category_product
							WHERE id_category IN (" . implode(',', $chunk) . ")
							ORDER BY id_product DESC")
				         ->iterateAssociative() as $row) {
				/** @var array $row */
				$catsProducts[(int) $row['id_category']][] = (int) $row['id_product'];
			}

			foreach ($conn->executeQuery("SELECT product_id, category_id
							FROM eshop_catalog__product_in_site
							WHERE category_id IN (" . implode(',', $chunk) . ")
							ORDER BY product_id DESC")->iterateAssociative() as $row) {
				/** @var array $row */
				$catsProducts[(int) $row['category_id']][] = (int) $row['product_id'];
			}

			foreach ($catsProducts as $categoryId => $products) {
				$result[$categoryId] = 0;

				foreach (array_chunk(array_unique($products), 25) as $prodsChunk) {
					foreach ($productsFacade->getProducts($prodsChunk) as $product) {
						if ($product->canAddToCart) {
							$result[$categoryId] = 1;
							break 2;
						}
					}
				}

				$i++;
				if ($useCleaner && $i % 5 === 0) {
					$productsFacade->clearTemp();
					$this->em->clear();
					gc_collect_cycles();
				}
			}
		}

		return $result;
	}

	public function checkActiveProductsVirtual(array $virtualIds, ProductsFacade $productsFacade, bool $useCleaner = false): array
	{
		$result = [];
		$conn   = $this->em->getConnection();
		$i      = 0;

		if (class_exists('\EshopAdvancedFeature\DI\EshopAdvancedFeatureExtension')) {
			foreach (array_chunk(array_unique($virtualIds), 50, true) as $chunk) {
				foreach ($conn->executeQuery("SELECT vct.url, vc.id, GROUP_CONCAT(vcc.category_id) as cats, GROUP_CONCAT(vcf.feature_value_id) as fea, GROUP_CONCAT(vcm.manufacturer_id) as manu
					FROM eshop_advanced_feature__virtual_category_text vct
					INNER JOIN eshop_advanced_feature__virtual_category vc ON vct.id = vc.id
                      LEFT JOIN eshop_advanced_feature__virtual_category_categories vcc ON vcc.virtual_category_id = vc.id
                      LEFT JOIN eshop_advanced_feature__virtual_category_features vcf on vcf.virtual_category_id = vc.id
                      LEFT JOIN eshop_advanced_feature__virtual_category_manufacturers vcm on vcm.virtual_category_id = vc.id
					WHERE vct.locale = :lang AND vct.url IN ('" . implode("','", $chunk) . "') GROUP BY vct.id", [
					'lang' => $this->translator->getLocale(),
				])->iterateAssociative() as $row) {
					/** @var array $row */
					$result[$row['url']] = 0;

					$cats = $row['cats'] ? array_unique(explode(',', (string) $row['cats'])) : [];
					$fea  = $row['fea'] ? array_unique(explode(',', (string) $row['fea'])) : [];
					$manu = $row['manu'] ? array_unique(explode(',', (string) $row['manu'])) : [];

					if (empty($cats) && empty($fea) && empty($manu)) {
						continue;
					}

					$prodsIds = [];
					foreach ($conn->executeQuery("SELECT p.id FROM eshop_catalog__product p
						INNER JOIN eshop_catalog__product_in_site pis ON pis.product_id = p.id AND pis.is_active = 1" .
						($fea ? " INNER JOIN eshop_catalog__feature_product fp ON fp.id_product = p.id AND fp.id_feature_value IN (" . implode(',', $fea) . ")" : '') .
						($cats ? " LEFT JOIN eshop_catalog__category_product cp ON cp.id_product = p.id" : '') .
						" WHERE p.is_published = 1" .
						($manu ? " AND p.id_manufacturer IN (" . implode(',', $manu) . ")" : '') .
						($cats ? " AND (cp.id_category IN (" . implode(',', $cats) . ") OR pis.category_id IN (" . implode(',', $cats) . "))" : '')
					)->iterateAssociative() as $row2) {
						/** @var array $row2 */
						$prodsIds[] = (int) $row2['id'];
					}

					if (!empty($prodsIds)) {
						foreach (array_chunk(array_unique($prodsIds), 25) as $prodsIdsChunk) {
							foreach ($productsFacade->getProducts($prodsIdsChunk) as $product) {
								if ($product->canAddToCart) {
									$result[$row['url']] = 1;

									break 2;
								}
							}
						}
					}

					$i++;
					if ($useCleaner && $i % 5 === 0) {
						$productsFacade->clearTemp();
						$this->em->clear();
						gc_collect_cycles();
					}
				}
			}
		}

		return $result;
	}

	public function checkProductsCountInCategories(array $categories): void
	{
		$conn                  = $this->em->getConnection();
		$categories            = array_unique($categories);
		$langValues            = $this->translator->getLocalesWhitelist() ?: [];
		$currentCategoryStatus = [];
		$catsForUpdate         = [];
		$site                  = $this->sites->getCurrentSite();
		$catRootId             = $this->getSiteRootId($site->getIdent());

		/** @var ProductsFacade $productsFacade */
		$productsFacade = $this->container->getService('eshopCatalog.front.productsFacade');

		foreach ($conn->executeQuery('SELECT id, tree_root, has_active_products 
				FROM eshop_catalog__category 
				WHERE tree_root = ? AND id IN (' . implode(',', $categories) . ')', [$catRootId])->iterateAssociative() as $row) {
			/** @var array $row */
			$currentCategoryStatus[$row['id']] = [
				'rootId'            => (int) $row['tree_root'],
				'hasActiveProducts' => (int) $row['has_active_products'],
			];
		}

		$categories = array_intersect($categories, array_keys($currentCategoryStatus));

		foreach ($this->checkActiveProducts($categories, $productsFacade) as $catId => $v) {
			if (isset($currentCategoryStatus[$catId]) && $currentCategoryStatus[$catId]['hasActiveProducts'] !== $v) {
				$conn->update('eshop_catalog__category', ['has_active_products' => $v], ['id' => $catId]);
				$currentCategoryStatus[$catId]['hasActiveProducts'] = (int) $v;

				foreach ($langValues as $l) {
					$this->cacheService->categoryCache->remove('structured_' . $l . '-' . $currentCategoryStatus[$catId]['rootId']);
				}

				$catsForUpdate[] = $catId;
				AppState::addUpdatedEntityId('eshopCatalog.category', $catId);
			}
		}

		if (!empty($catsForUpdate) && php_sapi_name() === 'cli') {
			$this->eventDispatcher->dispatch(new Event(['categories' => $catsForUpdate]), 'cli.eshopCatalog.category.updated');
		}

		$dynamicModuleGroupIds = [];
		foreach (array_chunk($categories, 250) as $chunk) {
			foreach ($conn->executeQuery('SELECT id, attrs FROM eshop_catalog__category WHERE id IN (' . implode(',', $chunk) . ') AND attrs IS NOT NULL')->iterateAssociative() as $row) {
				/** @var array $row */
				try {
					$attrs = str_starts_with($row['attrs'], '{')
						? Json::decode($row['attrs'], forceArrays: true)
						: unserialize($row['attrs'], ['allowed_classes' => true]);

					if (!empty($attrs)) {
						foreach ($attrs['topDynamicModuleInCategory'] ?? [] as $v) {
							$dynamicModuleGroupIds[] = (int) $v;
						}
						foreach ($attrs['bottomDynamicModuleInCategory'] ?? [] as $v) {
							$dynamicModuleGroupIds[] = (int) $v;
						}
					}
				} catch (\Exception $e) {
				}
			}
		}

		$this->checkProductsCountInDynamicModuleGroups($site, $dynamicModuleGroupIds);
	}

	public function checkVirtualGroup(Site $site): void
	{
		if (!class_exists('\EshopAdvancedFeature\DI\EshopAdvancedFeatureExtension')) {
			return;
		}

		$conn                  = $this->em->getConnection();
		$dynamicModuleGroupIds = [];

		// TODO php 8.3 predelat na SQL
		foreach ($this->em->getRepository(Navigation::class)->findBy([
			'componentType' => 'eshopAdvancedFeature.navigation.virtualCategoryGroup',
			'site'          => $site->getIdent(),
		]) as $nav) {
			try {
				/** @var Navigation $nav */
				$group = $nav->componentParams['group'] ?? null;

				if ($group) {
					foreach ($conn->fetchAllAssociative("SELECT vc.id, vc.params FROM eshop_advanced_feature__virtual_category vc
 							INNER JOIN eshop_advanced_feature__virtual_category_in_group vcg ON vcg.virtual_category_id = vc.id AND vcg.group_id = :group", ['group' => $group]) as $row) {
						/** @var array $row */
						try {
							if ($row['params']) {
								$params = Json::decode($row['params'], Json::FORCE_ARRAY);

								foreach ($params['eshopCatalog.topDynamicModuleInCategory'] ?? [] as $v) {
									$dynamicModuleGroupIds[] = (int) $v;
								}
								foreach ($params['eshopCatalog.bottomDynamicModuleInCategory'] ?? [] as $v) {
									$dynamicModuleGroupIds[] = (int) $v;
								}
							}
						} catch (Exception $e) {

						}
					}
				}
			} catch (\Exception $e) {
			}
		}

		$this->checkProductsCountInDynamicModuleGroups($site, $dynamicModuleGroupIds);
	}

	protected function checkProductsCountInDynamicModuleGroups(Site $site, array $dynamicModuleGroupIds): void
	{
		if (empty($dynamicModuleGroupIds) || !$this->container->hasService('dynamicmodule.front.groupsFactory')) {
			return;
		}

		$conn       = $this->em->getConnection();
		$extra      = [];
		$langValues = $this->translator->getLocalesWhitelist() ?: [];

		/** @var ProductsFacade $productsFacade */
		$productsFacade = $this->container->getService('eshopCatalog.front.productsFacade');

		/** @var IGroupsFactory $groupsFactory */
		$groupsFactory = $this->container->getService('dynamicmodule.front.groupsFactory');

		/** @var TemplateTextTypesCollection $templateTextTypeCollection */
		$templateTextTypeCollection = $this->container->getService('templateTextTypesCollection');

		$membersForUpdate = [];
		$groupsCache      = new Cache($this->cacheStorage, Groups::CACHE_NAMESPACE);
		$memberCache      = new Cache($this->cacheStorage, Members::CACHE_NAMESPACE);

		$dynamicModuleGroups = $groupsFactory->create('dynamic');

		/** @var Link $ttLink */
		$ttLink = $templateTextTypeCollection->getItemByType('link');
		$lang   = $this->translator->getLocale();

		foreach ($dynamicModuleGroupIds as $groupId) {
			foreach ($dynamicModuleGroups->getByIds([$groupId]) as $dynamicModuleGroup) {
				foreach ($dynamicModuleGroup->getMembers() as $member) {
					$link = $member->getFieldValue('link') ? Json::decode($member->getFieldValue('link'), Json::FORCE_ARRAY) : null;
					if ($ttLink && $link) {
						$ttLink->setDefault($link);
						$parsedLink = $ttLink->render(['multiLang' => $lang]);

						if (isset($parsedLink['link'])) {
							$url = new Url($parsedLink['link']);

							$extra[$member->getId()] = '/' . ltrim($url->getPath(), '/');
						}
					}
				}
			}
		}

		if (!empty($extra)) {
			$result = [];

			foreach ($this->checkActiveProductsVirtual($extra, $productsFacade) as $url => $urlV) {
				foreach ($extra as $memberId => $memberUrl) {
					if ($url === $memberUrl) {
						unset($extra[$memberId]);

						$result[$memberId] = $urlV;
					}
				}
			}

			if (!empty($extra)) {
				/** @var \EshopCatalog\FrontModule\Model\Categories|null $frontCategories */
				$frontCategories = $this->container->hasService('eshopCatalog.front.categories')
					? $this->container->getService('eshopCatalog.front.categories')
					: null;

				$missingFlipped = array_flip($extra);
				$baseUrl        = $site->getCurrentDomain() ? $site->getCurrentDomain()->getDomain() : null;

				if ($frontCategories) {
					$rootId = $frontCategories->getRootIdForSite($site->getIdent());
					foreach ($frontCategories->getCategories($rootId, $lang) as $cat) {
						$link = $cat->link;
						if ($baseUrl) {
							$link = str_replace('https://' . $baseUrl, '', $link);
						}

						if (isset($missingFlipped[$link])) {
							$newV = $cat->activeProducts;

							foreach ($extra as $k => $v) {
								if ($v === $link) {
									$result[$k] = $newV;

									unset($extra[$k]);
								}
							}
						}
					}
				}
			}

			foreach (array_chunk(array_keys($result), 100) as $chunk) {
				foreach ($conn->executeQuery("SELECT id, params FROM dynamicmodule__member WHERE id IN (" . implode(',', $chunk) . ")")->iterateAssociative() as $row) {
					/** @var array $row */
					if ($row['params']) {
						$row['params'] = Json::decode($row['params'], Json::FORCE_ARRAY);
					} else {
						$row['params'] = [];
					}

					if (!isset($row['params']['activeProducts']) || $row['params']['activeProducts'] !== $result[$row['id']]) {
						$row['params']['activeProducts'] = $result[$row['id']];

						$conn->update('dynamicmodule__member', [
							'params' => Json::encode($row['params']),
						], [
							'id' => $row['id'],
						]);

						foreach ($langValues as $l) {
							$memberCache->remove('dynamicModuleMember/' . $l . '/' . $row['id']);
						}

						$groupsCache->clean([Cache::Tags => [Groups::CACHE_NAMESPACE]]);
						$memberCache->clean([Cache::Tags => [Members::CACHE_NAMESPACE]]);

						$membersForUpdate[] = $row['id'];
						AppState::addUpdatedEntityId('dynamicModule.member', $row['id']);
					}
				}
			}
		}

		if (!empty($membersForUpdate) && php_sapi_name() === 'cli') {
			$this->eventDispatcher->dispatch(new Event(['members' => $membersForUpdate]), 'cli.dynamicModule.member.updated');
		}
	}
}
