<?php declare(strict_types = 1);

namespace EshopCatalog\AdminModule\Model\FormContainers;

use Core\Model\Entities\EntityManagerDecorator;
use Core\Model\Helpers\Strings;
use Core\Model\UI\Form\BaseContainer;
use DateTimeInterface;
use EshopCatalog\AdminModule\Model\Products;
use EshopCatalog\AdminModule\Model\ProductsChangePlan;
use EshopCatalog\AdminModule\Model\VatRates;
use EshopCatalog\Model\Config;
use EshopCatalog\Model\Entities\Product;
use EshopCatalog\Model\Entities\ProductChangePlan;
use Nette\Application\LinkGenerator;
use Nette\Localization\Translator;
use Nette\Utils\ArrayHash;
use Nette\Utils\Html;
use Nette\Utils\Json;

class ProductChangePlanContainer
{
	public function __construct(
		protected EntityManagerDecorator $em,
		protected Products               $products,
		protected ProductsChangePlan     $productsChangePlan,
		protected VatRates               $vatRates,
		protected LinkGenerator          $linkGenerator,
		protected Translator             $translator,
	)
	{
	}

	public function getContainer(array $products): BaseContainer
	{
		$container = new BaseContainer;

		if (!Config::load('product.allowPlan', false)) {
			return $container;
		}

		$container->addSelect('column', 'eshopCatalog.productForm.column', [
			ProductChangePlan::columnVatRate => 'eshopCatalog.productForm.columnVatRate',
		]);

		$container->addSelect('vatRate', 'eshopCatalog.productForm.vatRate', $this->vatRates->getOptionsForSelect());
		$container->addDateTimePicker('changeAt', 'eshopCatalog.productForm.changeAt');
		$container->addHidden('products', Json::encode($products));

		$container->addButton('setChange', 'eshopCatalog.productForm.setChange');

		$container->addCustomData('template', __DIR__ . '/ProductChangePlanContainer.latte')
			->addCustomData('addLink', $this->linkGenerator->link('EshopCatalog:Admin:Products:setChangePlan'));

		if (count($products) === 1) {
			$container->addCustomData('dataTable', $this->getProductTable((int) $products[0]));
		}

		return $container;
	}

	public function changeData(
		array             $productsIds,
		DateTimeInterface $changeAt,
		string            $column,
		string            $value
	): void
	{
		if (!Config::load('product.allowPlan', false)) {
			return;
		}

		foreach ($productsIds as $id) {
			$changePlan = new ProductChangePlan(
				$this->products->getReference((int) $id),
				$column,
				$value,
				$changeAt
			);

			$this->em->persist($changePlan);
		}

		$this->em->flush();
	}

	public function removeData(int $id): void
	{
		if (!Config::load('product.allowPlan', false)) {
			return;
		}

		$this->em->remove($this->productsChangePlan->getReference($id));
		$this->em->flush();
	}

	public function getProductTable(int $productId): Html
	{
		if (!Config::load('product.allowPlan', false)) {
			return Html::el();
		}

		$table = Html::el('table', ['class' => 'productChangePlanTable'])
			->addHtml(Html::el('thead')
				->addHtml(Html::el('tr')
					->addHtml(Html::el('th')->setText($this->translator->translate('eshopCatalog.productForm.column')))
					->addHtml(Html::el('th')->setText($this->translator->translate('eshopCatalog.productForm.value')))
					->addHtml(Html::el('th')->setText($this->translator->translate('eshopCatalog.productForm.changeAt')))
					->addHtml(Html::el('th')->setText($this->translator->translate('eshopCatalog.productForm.changed')))
					->addHtml(Html::el('th'))
				)
			);

		$tbody = Html::el('tbody');

		foreach ($this->productsChangePlan->getPlanForProduct($productId) as $data) {
			$value = '';
			if ($data->getColumn() === ProductChangePlan::columnVatRate) {
				$value = $this->vatRates->getOptionsForSelect()[((int) $data->getValue())];
			}

			$tbody->addHtml(Html::el('tr', [
				'data-id'      => $data->getId(),
				'data-product' => $productId,
			])
				->addHtml(Html::el('td')->setText($this->translator->translate('eshopCatalog.productForm.column' . Strings::firstUpper($data->getColumn()))))
				->addHtml(Html::el('td')->setText($value))
				->addHtml(Html::el('td')->setText($data->getChangeAt()->format('j. n. Y H:i')))
				->addHtml(Html::el('td')->setText($data->getChanged() ? $data->getChanged()->format('j. n. Y H:i') : ''))
				->addHtml(Html::el('td')->addHtml(Html::el('a')
					->addAttributes([
						'href'         => '#',
						'class'        => 'btn btn-xs ajax btn-danger removeChangePlan',
						'data-link'    => $this->linkGenerator->link('EshopCatalog:Admin:Products:removeChangePlan'),
						'data-confirm' => $this->translator->translate('default.reallyDelete'),
					])
					->addHtml(Html::el('i', ['class' => 'fa fa-times']))
				))
			);
		}

		return $table->addHtml($tbody);
	}

	public function saveData(ArrayHash $values): void
	{
	}

	public function setDefaults(BaseContainer $container, array $speditions, Product $product): void
	{
	}

}
