<?php declare(strict_types = 1);

namespace EshopCatalog\AdminModule\Model\FormContainers;

use Core\Model\Entities\EntityManagerDecorator;
use Core\Model\Helpers\Arrays;
use Core\Model\UI\Form\BaseContainer;
use EshopCatalog\AdminModule\Model\Products;
use EshopCatalog\Model\Config;
use EshopCatalog\Model\Entities\Product;
use EshopCatalog\Model\Entities\ProductPayment;
use EshopOrders\AdminModule\Model\Payments;
use EshopOrders\Model\Entities\Payment;
use Nette\Utils\ArrayHash;

class ProductPaymentContainer
{
	public function __construct(
		protected EntityManagerDecorator $em,
		protected Payments               $payments,
		protected Products               $products,
	)
	{
	}

	public function getContainer(bool $isMultiple = false): BaseContainer
	{
		$container = new BaseContainer;

		if (!Config::load('product.allowModifyPayment', false)) {
			return $container;
		}

		if ($isMultiple) {
			$container->addSelect('disableApply', 'eshopCatalog.productPaymentContainer.disableApply', [
				'override'      => 'eshopCatalog.productPaymentContainer.override',
				'applyDisabled' => 'eshopCatalog.productPaymentContainer.applyDisabled',
			]);
		}

		$containerForPayments = new BaseContainer;

		foreach ($this->payments->getForSelectOption() as $id => $name) {
			$container2 = new BaseContainer;
			$container2->addCustomData('name', $name);
			$container2->addBool('disabled', $name);
			$container2->addHidden('id', (string) $id);

			$containerForPayments->addComponent($container2, (string) $id);
		}
		$container->addComponent($containerForPayments, 'payments');

		$container->addCustomData('template', __DIR__ . '/ProductPaymentContainer.latte');

		return $container;
	}

	public function saveData(ArrayHash $values, array $products): void
	{
		$payments     = (array) $values->payments;
		$disableApply = $values->disableApply ?? 'override';

		$canRemoveDisable = Arrays::contains(['override'], $disableApply);
		$canAddDisable    = Arrays::contains(['override', 'applyDisabled'], $disableApply);

		foreach ($products as $product) {
			if ($product instanceof Product === false) {
				$product = $this->products->getReference((int) $product);
			}

			$currentPayments = $product->payments->toArray();

			if ($canRemoveDisable) {
				foreach (array_diff_key($currentPayments, $payments) as $entity) {
					$this->em->remove($entity);
				}
			}

			foreach ($payments as $id => $data) {
				/** @var ProductPayment|null $entity */
				$entity = $product->payments->get($id);

				if ($data->disabled === 0) {
					if ($entity && $canRemoveDisable) {
						$this->em->remove($entity);
					}

					continue;
				}

				if ($canAddDisable) {
					if (!$entity) {
						/** @var Payment $payment */
						$payment = $this->payments->getReference($id);

						$entity = new ProductPayment($product, $payment);
					}

					$entity->paymentDisabled = (int) $data->disabled;
				}

				$this->em->persist($entity);
			}
		}
	}

	public function setDefaults(BaseContainer $container, array $payments, Product $product): void
	{
		$d = [
			'payments' => [],
		];

		foreach ($payments as $id => $entity) {
			/** @var ProductPayment $entity */
			$d['payments'][$id] = [
				'disabled' => $entity->paymentDisabled,
			];
		}

		$container->setDefaults($d);
	}

}
