<?php declare(strict_types = 1);

namespace EshopCatalog\AdminModule\Model\Import;

use Contributte\Translation\Translator;
use Core\AdminModule\Model\Sites;
use Core\Model\Entities\EntityManagerDecorator;
use Core\Model\Event\Event;
use Core\Model\Event\EventDispatcher;
use Core\Model\Helpers\Arrays;
use Core\Model\Notifiers\MailNotifiers\LogNotifier;
use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\ORM\Query\Expr\Join;
use Doctrine\ORM\Query\Parameter;
use EshopCatalog\AdminModule\Model\Products;
use EshopCatalog\Model\AvailabilityService;
use EshopCatalog\Model\Config;
use EshopCatalog\Model\Entities\Availability;
use EshopCatalog\Model\Entities\Category;
use EshopCatalog\Model\Entities\Product;
use EshopCatalog\Model\Entities\ProductSupplier;
use EshopCatalog\Model\Entities\ProductTexts;
use EshopCatalog\Model\Entities\Supplier;
use Exception;
use Gallery\Model\Entities\Album;
use Gallery\Model\Entities\AlbumText;
use Import\AdminModule\Model\ProcessingType\ProcessingType;
use Import\Model\Entities\Import;
use Import\Model\ImportDebugger;
use Nette\Utils\Strings;
use Override;
use Tracy\Debugger;

class ProcessPairing extends ProcessingType
{
	public string $title           = 'eshopCatalog.import.processingType.pairing';
	public string $name            = 'eshopCatalog.pairing';
	public array  $lastProductsRun = [];

	protected array $cSkipCodes = [];

	public function __construct(
		protected EntityManagerDecorator $em,
		protected Translator             $translator,
		protected ImportHelper           $importHelper,
		protected Sites                  $sites,
		protected AvailabilityService    $avService,
		protected Products               $products,
		protected EventDispatcher        $eventDispatcher,
	)
	{
	}

	/**
	 * @inheritDoc
	 */
	#[Override]
	public function run(Import $import, array $data): array
	{
		Debugger::$showBar = false;

		$pairing = (array) $import->data;
		$result  = [
			'created'  => 0,
			'updated'  => 0,
			'notFound' => 0,
			'message'  => '',
		];

		AvailabilityService::$allowCheckCountInCategories = false;

		$supplierOpt = (string) $import->getSyncOptString('supplier');

		$supplier = $this->importHelper->getSupplier($supplierOpt);
		if (!$supplier instanceof Supplier) {
			return $result;
		}

		$syncBy              = $import->getSyncOptString('syncBy');
		$soldOutAvailability = (int) $this->em->getConnection()->fetchOne("SELECT id FROM eshop_catalog__availability WHERE ident = ?", [Availability::SOLD_OUT]);
		$syncBySupplier      = $syncBy === 'supplierCode';

		$lang               = $this->translator->getDefaultLocale();
		$exists             = [];
		$existsWithSupplier = [];
		// Zjisteni existujicich produktu prirazenych dodavateli
		if ($syncBySupplier) {
			if (php_sapi_name() === 'cli') {
				echo "Sync by supplier\n";
			}
			foreach ($this->em->getConnection()->fetchAllAssociative("SELECT p.id, ps.id_supplier, ps.code FROM eshop_catalog__product_supplier ps
				INNER JOIN eshop_catalog__product p on ps.id_product = p.id AND p.is_deleted = 0
				WHERE ps.id_supplier = ?", [$supplier->getId()]) as $v) {
				$exists[$v['code']]             = $v['id'];
				$existsWithSupplier[$v['code']] = $v['id'];
			}
		} else {
			if (php_sapi_name() === 'cli') {
				echo "Sync by all\n";
			}
			$tmp1 = [];
			$tmp2 = [];
			foreach ($this->em->getRepository(Product::class)->createQueryBuilder('p')
				         ->select('p.id, IDENTITY(ps.supplier) as supplier, ps.code, pt.name, p.code1, p.' . $syncBy)
				         ->leftJoin('p.suppliers', 'ps', 'WITH', 'ps.supplier = :supplier')
				         ->innerJoin('p.productTexts', 'pt', Join::WITH, 'pt.lang = :lang')
				         ->setParameters(new ArrayCollection([new Parameter('supplier', $supplier->getId()), new Parameter('lang', $this->translator->getLocale())]))
				         ->groupBy('p.id')
				         ->getQuery()->getResult() as $v) {
				$syncByV = $v[$syncBy] ?: $v['name'];

				if (isset($tmp1[$syncByV]) || isset($tmp2[$syncByV])) {
					$syncByV = $syncByV . '_' . $v['code1'];
				}

				if ($v['supplier']) {
					$tmp1[$syncByV]                 = $v['id'];
					$existsWithSupplier[$v['code']] = $v['id'];
				} else {
					$tmp2[$syncByV] = $v['id'];
				}
			}

			$exists = $tmp1 + $tmp2;
		}
		if (php_sapi_name() === 'cli') {
			echo "Exist " . count($exists) . "\n";
		}

		// Srovnani podle synchronizacniho pole a vyfiltrovani zvolenych vyrobcu
		$tmp                  = [];
		$tmpForNotFound       = [];
		$allowedManufacturers = $pairing['allowedManufacturers']['value'] ?? null;
		if (count($allowedManufacturers) === 0) {
			$allowedManufacturers = null;
		}

		if (empty($data['data'])) {
			return $result;
		}

		foreach ($data['data'] as $row) {
			$skipThis = false;

			foreach (['ean', 'code1', 'code2'] as $col) {
				$colVal = $import->getSyncOptString('only' . ucfirst($col) . 'Start');
				if (!$colVal) {
					continue;
				}

				$skip = true;
				foreach (explode(',', (string) $colVal) as $ms) {
					if (\str_starts_with((string) $row[$pairing[$col]['key']], $ms)) {
						$skip = false;
						break;
					}
				}

				if ($skip) {
					$skipThis = true;
				}
			}

			if ($skipThis) {
				continue;
			}

			if (!$allowedManufacturers || in_array($row[$pairing['allowedManufacturers']['key']], $allowedManufacturers)) {
				$tmp[]                                                                                      = $row;
				$tmpForNotFound[(string) $row[$pairing[$pairing['supplierCode']['key'] ?? 'code1']['key']]] = $row;
			}
		}

		$data['data'] = $tmp;

		if (php_sapi_name() === 'cli') {
			echo "Find not found\n";
		}

		// Nenalezení v importu nastavit počet ks na 0
		$notFound = array_diff_key($existsWithSupplier, $tmpForNotFound);
		if (php_sapi_name() === 'cli') {
			echo "Not found " . count($notFound) . "\n";
		}
		if ($notFound !== []) {
			foreach (array_chunk($notFound, 100) as $nfs) {
				$notFoundProds = [];
				foreach ($this->em->getRepository(Product::class)->createQueryBuilder('p')
					         ->join('p.suppliers', 'ps')
					         ->where('p.id IN (:ids)')
					         ->setParameter('ids', $nfs)
					         ->getQuery()->getResult() as $row) {
					$notFoundProds[$row->getId()] = $row;
					$result['notFound']++;
				}

				$inValues = implode(',', $nfs);
				$this->em->getConnection()->executeStatement('UPDATE eshop_catalog__product_supplier SET quantity = 0 WHERE id_product IN (' . $inValues . ') AND id_supplier = ' . $supplier->getId());

				foreach ($notFoundProds as $prod) {
					$this->avService->updateAvailabilityByQuantity($prod);
					$this->em->persist($prod);
				}
				$this->em->flush();
			}

			$notFoundProds = null;
			$notFound      = null;
		}

		$totalCount       = count($data['data']);
		$totalCountLength = strlen((string) $totalCount);
		$currentI         = 0;
		$chunks           = array_chunk($data['data'], 25, true);

		$total           = 0;
		$totalLimit      = (int) Config::load('importTotalLimit', 20000);
		$galleries       = [];
		$updatedProducts = [];
		$codesForSkip    = $this->getCodesForSkip($supplier->getId());
		$newIds          = [];

		if (php_sapi_name() === 'cli') {
			echo "Start import\n";
		}
		foreach ($chunks as $chunk) {
			$this->em->beginTransaction();
			try {
				$i        = 0;
				$supplier = $this->importHelper->getSupplier($supplierOpt);
				$allIds   = [];

				foreach ($chunk as $row) {
					$currentI++;

					if (php_sapi_name() === 'cli') {
						echo "Start row - " . str_pad((string) $currentI, $totalCountLength, '0', STR_PAD_LEFT) . "/ {$totalCount}" . "\n";
					}


					if (
						isset($row[$pairing['supplierCode']['key']])
						&& array_key_exists($this->fixString((string) $row[$pairing['supplierCode']['key']]), $codesForSkip)
					) {
						if (php_sapi_name() === 'cli') {
							echo "Skip row supplierCode missing\n";
						}

						continue;
					}

					$exist = false;

					if (isset($pairing['priceVat']['key'])) {
						$price = (float) $this->fixString($row[$pairing['priceVat']['key']]);

						if ($price < (float) Config::loadScalar('import.minimumImportPrice')) {
							if (php_sapi_name() === 'cli') {
								echo "Low price " . Config::loadScalar('import.minimumImportPrice') . "\n";
							}

							continue;
						}
					}

					// Pokud neexistuje parovaci klic, tak se pouzije ean
					if (!$row[$pairing[$syncBy]['key']]) {
						$row[$pairing[$syncBy]['key']] = $row[$pairing['ean']['key']];
					}

					$syncByKey   = Strings::trim((string) $row[$pairing[$syncBy]['key']]);
					$pairingName = array_values($pairing['name']['key'])[0] ?? '';
					$productId   = null;

					if (php_sapi_name() === 'cli') {
						echo "Sync by - " . $syncBy . " - " . $syncByKey . "\n";
					}

					if (isset($existsWithSupplier[$row[$pairing['supplierCode']['key']]])) {
						$productId = $existsWithSupplier[$row[$pairing['supplierCode']['key']]];
					} else if (isset($exists[$syncByKey . '_' . $row[$pairing['code1']['key']]])) {
						$productId = $exists[$syncByKey . '_' . $row[$pairing['code1']['key']]];
					} else if (isset($exists[$syncByKey])) {
						$productId = $exists[$syncByKey];
					}

					if (!$productId && $pairingName && $row[$pairingName]) {
						if (isset($exists[$row[$pairingName]])) {
							$productId = $exists[$row[$pairingName]];
						} else if (isset($exists[$row[$pairingName] . '_' . $row[$pairing['code1']['key']]])) {
							$productId = $exists[$row[$pairingName] . '_' . $row[$pairing['code1']['key']]];
						}
					}

					$productTexts = [];
					if ($productId) {
						/** @var Product $product */
						$product      = $this->em->getRepository(Product::class)->find($productId);
						$productTexts = $product->getTexts()->toArray();
						$exist        = true;

						if (isset($updatedProducts[$product->getId()])) {
							continue;
						}
						$updatedProducts[$product->getId()] = true;
					} else {
						$product = null;
					}

					if (!$exist) {
						/** @phpstan-ignore-next-line */
						if ($import->getSyncOpt('onlyUpdate') == 1) {
							continue;
						}

						$product      = new Product;
						$productTexts = $product->getTexts()->toArray();
					}

					/** @var ProductSupplier|null $productSupplier */
					$productSupplier = $exist
						? ($this->em->getRepository(ProductSupplier::class)->createQueryBuilder('ps')
							->where('ps.product = :prod')
							->andWhere('ps.supplier = :supp')
							->setParameters(new ArrayCollection([new Parameter('prod', $product->getId()), new Parameter('supp', $supplier->getId())]))->getQuery()->setMaxResults(1)->getOneOrNullResult())
						: null;

					if (!$productSupplier) {
						$productSupplier       = new ProductSupplier($product, $supplier);
						$productSupplier->code = $this->fixString((string) $row[$pairing['supplierCode']['key']]);
					}

					if ($pairing['supplierCode']['key'] && (!$exist || $pairing['supplierCode']['applyOn'] != 'new')) {
						$productSupplier->code = $this->fixString((string) $row[$pairing['supplierCode']['key']]);
					}

					if (!$productSupplier->isActive) {
						continue;
					}

					$this->em->persist($product);
					$this->em->flush();

					foreach (['width', 'height', 'depth', 'weight'] as $v) {
						if (!isset($pairing[$v]) || ($exist && $pairing[$v]['applyOn'] === 'new')) {
							continue;
						}

						$modify = $pairing[$v]['modifyInt'];
						$value  = (float) str_replace(',', '.', $this->fixString($row[$pairing[$v]['key']]));

						if ($modify) {
							$value = round($value * (float) $modify);
						}

						$product->$v = (int) $value;
					}

					$pVals = ['ean', 'code1', 'code2', 'vatRate', 'price' => 'priceVat', 'purchasePrice',
						'retailPrice', 'recyclingFee'];
					foreach ($pVals as $k => $v) {
						$k = is_string($k) ? $k : $v;

						if (!isset($pairing[$v]) || ($exist && $pairing[$v]['applyOn'] === 'new')) {
							continue;
						}

						if ($v === 'vatRate') {
							/** @phpstan-ignore-next-line */
							$product->$k = $this->importHelper->getVatRate((int) $this->fixString($row[$pairing[$v]['key']]) ?: $pairing[$v]['fixedValue']);
						} else if ($v === 'ean') {
							$product->ean = $this->importHelper->validateEAN((string) ($this->fixString($row[$pairing[$v]['key']]) ?: $pairing[$v]['fixedValue']));
						} else if (in_array($k, ['price', 'retailPrice'])) {
							if ($k === 'retailPrice' && !Config::load('product.allowRetailPrice')) {
								continue;
							}
							$modify          = $pairing[$v]['modifyInt'];
							$priceWithoutVat = (bool) $pairing[$v]['withoutVat'];
							$price           = (float) $this->fixString($row[$pairing[$v]['key']]);
							if ($modify) {
								$price = round($price * (float) $modify, 0);
							}
							if ($priceWithoutVat) {
								$price = round($price * (1 + $product->getVateRate()->getModifier()));
							}
							$this->importHelper->setProductPrice($product, $price, $k);
						} else if (isset($pairing[$v]['key'])) {
							$fixedString = $this->fixString($row[$pairing[$v]['key']]);
							if (Arrays::contains(['ean', 'code1', 'code2'], $k)) {
								$fixedString = (string) $fixedString;
							}
							/** @phpstan-ignore-next-line */
							$product->$k = $fixedString;
						} else if (isset($pairing[$v]['fixedValue'])) {
							/** @phpstan-ignore-next-line */
							$product->$k = $this->fixString($pairing[$v]['fixedValue']);
						}
					}

					$pValsBool = ['isPublished', 'isAssort'];
					foreach ($pValsBool as $k => $v) {
						/** @phpstan-ignore-next-line */
						$k = is_string($k) ? $k : $v;

						if (!isset($pairing[$v]) || ($exist && $pairing[$v]['applyOn'] === 'new')) {
							continue;
						}

						if (isset($row[$pairing[$v]['key']])) {
							if (!isset($pairing[$v]['key'])) {
								$product->$k = 0;
							} else {
								$val         = (float) $this->fixString($row[$pairing[$v]['key']]);
								$compare     = (float) $pairing[$v]['value'];
								$product->$k = $this->importHelper->checkValueIs((string) $pairing[$v]['valueIs'], $val, $compare) ? 1 : 0;
							}
						} else if (isset($pairing[$v]['fixedValue'])) {
							$product->$k = (int) $this->fixString($pairing[$v]['fixedValue']);
						}
					}

					$ptVals       = ['name', 'description', 'shortDescription'];
					$textsUpdated = false;
					foreach ($ptVals as $k => $v) {
						/** @phpstan-ignore-next-line */
						$k = is_string($k) ? $k : $v;

						if (!isset($pairing[$v]) || ($exist && $pairing[$v]['applyOn'] === 'new')) {
							continue;
						}

						$lastText = null;
						foreach ($pairing[$v]['key'] as $kLang => $vKey) {
							/** @var ProductTexts|null $text */
							$text = $productTexts[$kLang];

							if (!$text) {
								$text                 = new ProductTexts($product, $kLang);
								$productTexts[$kLang] = $text;
							}

							$textVal = '';
							if ($vKey) {
								$textVal = $this->fixString($row[$vKey]);
							} else if (isset($pairing[$v]['fixedValue'])) {
								$textVal = $this->fixString($pairing[$v]['fixedValue']);
							}

							if (!$textVal && $lastText && $lastText->$k) {
								$textVal = $lastText->$k;
							}

							if ($v == 'shortDescription') {
								$text->setShortDescription((string) $textVal);
							}

							if ($v === 'name') {
								$text->setName((string) $textVal);
							}

							if ($v === 'name' && $text->$k) {
								$text->setAlias((string) $text->$k);
							}

							if ($v === 'description') {
								$text->setDescription((string) $textVal);

								if (!$text->shortDescription) {
									$text->setShortDescription((string) $text->description);
								}
							}
							$lastText = $text;
						}

						$textsUpdated = true;
					}

					if ($textsUpdated) {
						$this->em->persist($product);
						foreach ($productTexts as $pt) {
							$this->em->persist($pt);
						}
					}

					// TODO dodělat nefixní hodnotu
					//				if (isset($pairing['category']['key']) && $row[$pairing['category']['key']] && false)
					//					$productSupplier->quantity = $this->importHelper->getManufacturer($this->fixString($row[$pairing['quantity']['key']]));

					if (isset($pairing['manufacturer']) && (!$exist || $pairing['manufacturer']['applyOn'] != 'new')) {
						if (isset($row[$pairing['manufacturer']['key']])) {
							$man = $this->fixString($row[$pairing['manufacturer']['key']]);
							if ($man && $man != '') {
								$product->setManufacturer($this->importHelper->getManufacturer($man));
							} else {
								$product->setManufacturer(null);
							}
						} else if (isset($pairing['manufacturer']['fixedValue'])) {
							$product->setManufacturer($this->importHelper->getManufacturer($pairing['manufacturer']['fixedValue']));
						}
					}

					if (!$exist || ($pairing['category'] && $pairing['category']['applyOn'] !== 'new')) {
						$baseCat        = $this->importHelper->findCategory((int) $import->getSyncOpt('baseCategory'));
						$deepC          = null;
						$productBaseCat = $baseCat->getId();

						if ($baseCat instanceof Category) {
							$this->em->getConnection()->executeQuery("INSERT IGNORE INTO eshop_catalog__product_in_site (product_id, site, is_active, category_id) VALUES (?, ?, ?, ?)", [
								$product->getId(),
								$this->sites->get($baseCat->getRoot()->getCategoryText()->alias)->getIdent(),
								0,
								$baseCat->getId(),
							]);
						}

						if (isset($pairing['category']['fixedValue'])) {
							$cat = $this->importHelper->findCategory((int) $pairing['category']['fixedValue']);
							if ($cat instanceof Category) {
								$product->addCategoryProduct($this->importHelper->getCategoryProduct($product, $cat));
							}
						} else {
							$pCat       = $pairing['category'];
							$importCats = null;

							if (isset($row[$pCat['key']])) {
								if ($pCat['separator']) {
									$importCats = explode($pCat['separator'], (string) $row[$pCat['key']]);
								} else if (is_array($row[$pCat['key']])) {
									$importCats = $row[$pCat['key']];
								} else {
									$importCats = [$row[$pCat['key']]];
								}

								if ($pCat['deepSeparator']) {
									foreach ($importCats as $icK => $icV) {
										if (is_array($icV)) {
											foreach ($icV as $icVV) {
												$importCats[] = explode($pCat['deepSeparator'], (string) $icVV);
											}
										} else {
											$importCats[$icK] = explode($pCat['deepSeparator'], (string) $icV);
										}
									}
								}
							}

							if ($baseCat && $importCats) {
								foreach ($importCats as $ic) {
									$deepC = $this->importHelper->getDeepCategory($baseCat, $ic, $lang);
									if ($deepC instanceof Category) {
										$product->addCategoryProduct($this->importHelper->getCategoryProduct($product, $deepC));
										$productBaseCat = $deepC->getId();
									}
								}
							}
						}

						if ($productBaseCat) {
							$this->em->getConnection()->executeQuery("INSERT IGNORE INTO eshop_catalog__product_in_site (product_id, site, is_active, category_id) VALUES (?, ?, ?, ?)", [
								$product->getId(),
								$this->sites->get($baseCat->getRoot()->getCategoryText()->alias)->getIdent(),
								0,
								$productBaseCat,
							]);
						}
					}

					// Parametry
					if (isset($pairing['params']['key']) && (!$exist || $pairing['params']['applyOn'] != 'new')) {
						$params = [];
						foreach ($pairing['params']['key'] as $ik) {
							if (is_array($row[$ik])) {
								foreach ($row[$ik] as $ikk => $ikv) {
									$params[$ikk] = $ikv;
								}
							} else if ($row[$ik]) {
								$params[$ik] = $row[$ik];
							}
						}

						if ($params !== []) {
							$this->importHelper->parametersProccess($product, $params, $lang);
						}
					}

					// Dostupnost
					$quantity = $pairing['quantity']['key'] ? 0 : null;

					if (isset($row[$pairing['quantity']['key']])) {
						$quantity = $this->fixString($row[$pairing['quantity']['key']]);
					} else if (isset($pairing['quantity']['fixedValue'])) {
						$quantity = $this->fixString($pairing['quantity']['fixedValue']);
					}

					if ($quantity && !is_numeric($quantity)) {
						$quantity = filter_var($quantity, FILTER_SANITIZE_NUMBER_INT);
					}

					// Pokud není nastaven počet skladem, zkontroluje se hodnota
					if ($pairing['inStock']['key'] && (!$exist || $pairing['inStock']['applyOn'] != 'new')) {
						$iv       = $row[$pairing['inStock']['key']];
						$ic       = $pairing['inStock']['value'] !== '' ? $pairing['inStock']['value'] : $pairing['inStock']['custom'];
						$inStock  = $this->importHelper->checkValueIs($pairing['inStock']['valueIs'], $iv, $ic);
						$quantity = 0;
						if ($inStock) {
							$quantity = $pairing['quantity']['fixedValue'] ?? 1;
						}
					} else if ($pairing['idAvailability']['key'] && (!$exist || $pairing['idAvailability']['applyOn'] != 'new')) {
						$iv       = $row[$pairing['idAvailability']['key']];
						$ic       = $pairing['idAvailability']['value'] !== '' ? $pairing['idAvailability']['value'] : $pairing['idAvailability']['custom'];
						$inStock  = $this->importHelper->checkValueIs($pairing['idAvailability']['valueIs'], $iv, $ic);
						$quantity = 0;
						if ($inStock) {
							$quantity = $row[$pairing['idAvailability']['key']] ?? 1;
						}
					}

					if ($quantity !== null) {
						$quantity                  = $this->importHelper->extractInt($quantity);
						$productSupplier->quantity = $quantity;
					}

					if (isset($pairing['idAvailability']['inStock']) && $pairing['idAvailability']['inStock']) {
						$productSupplier->availabilityAfterSoldOut = $this->avService->getReference((int) $pairing['idAvailability']['inStock']);
					}

					if (isset($pairing['variant']) && isset($row[$pairing['variant']['key']]) && $row[$pairing['variant']['key']] !== '' && (($exist && $pairing['variant']['applyOn'] !== 'new') || !$exist)) {
						$this->importHelper->addToVariant(
							($pairing['variantPrefix']['key'] ?? '') . $row[$pairing['variant']['key']],
							$product->getId(),
							$row[$pairing['variantName']['key']] ?? null
						);
					}

					// Video
					if (isset($pairing['videos']) && (!$exist || $pairing['videos']['applyOn'] != 'new')) {
						$videos        = [];
						$currentVideos = [];

						foreach ($pairing['videos']['key'] as $k) {
							foreach ($row[$k] ?? [] as $v) {
								$videos[] = $v;
							}
						}

						if ($exist) {
							foreach ($this->em->getConnection()->fetchAllAssociative("SELECT id, url FROM eshop_catalog__product_video WHERE product_id = ?", [$product->getId()]) as $v) {
								$currentVideos[$v['id']] = $v['url'];
							}
						}

						$toDelete = array_diff($currentVideos, $videos);
						if (!empty($toDelete)) {
							$this->em->getConnection()->executeQuery("DELETE FROM eshop_catalog__product_video WHERE id IN (" . implode(',', array_keys($toDelete)) . ")");
						}

						foreach (array_diff($videos, $currentVideos) as $video) {
							try {
								$this->em->getConnection()->insert('eshop_catalog__product_video', [
									'product_id' => $product->getId(),
									'lang'       => null,
									'url'        => $video,
									'title'      => null,
									'thumbnail'  => null,
								]);
							} catch (Exception $e) {
								var_dump($e->getMessage());
								die();
							}
						}
					}


					$this->downloadGallery($row, $lang, $pairing, $exist, $product, Arrays::first($productTexts), $galleries);
					$productSupplier->setSupplier($supplier);

					$this->em->persist($productSupplier);

					if (!$product->getAvailability()) {
						$product->setAvailability($this->avService->getReference($soldOutAvailability));
					}

					$product->validateCreated();
					$this->avService->updateAvailabilityByQuantity($product);

					$this->em->persist($productSupplier);
					$this->em->persist($product);

					if ($productTexts) {
						foreach ($productTexts as $pt) {
							$this->em->persist($pt);
						}
					}
					if ($exist) {
						$result['updated']++;
					} else {
						$result['created']++;
					}
					$i++;
					$total++;

					if (!$exist) {
						$updatedProducts[$product->getId()] = true;
					}

					if ($i == 20) {
						$i = 0;
					}

					if ($total >= $totalLimit) {
						break;
					}

					$this->em->flush();
					$this->importHelper->moveGalleryImage($galleries);
					$galleries = [];
					$allIds[]  = $product->getId();

					if (!$exist) {
						$newIds[] = $product->getId();
					}
				}

				$this->em->flush();
				if ($this->em->getConnection()->isTransactionActive()) {
					$this->em->commit();
				}
				$this->em->clear();
				$supplier = $this->importHelper->getSupplier($supplierOpt);
				$this->importHelper->moveGalleryImage($galleries);
				$galleries                   = [];
				$this->importHelper->onError = [];

				foreach ($allIds as $id) {
					$this->products->clearProductCache((int) $id);
				}

				if ($total >= $totalLimit) {
					break;
				}
			} catch (Exception $e) {
				if ($this->em->getConnection()->isTransactionActive()) {
					$this->em->rollback();
				}
				Debugger::log($e, 'import');
				$result['message'] = $e->getMessage();
				if (isset($product)) {
					$result['product'] = $product->getId() . ' / ' . $product->code1;
				}

				foreach ($this->importHelper->onError as $onError) {
					$onError();
				}
				break;
			}
		}
		Debugger::log($result, 'import');


		$tmp = array_merge($newIds, array_keys($updatedProducts));
		if (!empty($tmp)) {
			$this->eventDispatcher->dispatch(new Event(['products' => $tmp]), 'eshopCatalog.category.checkProductsCount');
		}

		try {
			if ((int) $import->getSyncOpt('onlyUpdate') !== 1) {
				$conn = $this->em->getConnection();

				foreach (['relatedProducts1', 'relatedProducts2', 'relatedProducts3', 'relatedProducts4', 'relatedProducts5', 'relatedProducts6'] as $pairingK) {
					$pairKey      = $pairing[$pairingK]['key'] ?? null;
					$applyOn      = $pairing[$pairingK]['applyOn'];
					$groupName    = $pairing[$pairingK]['groupName'] ?? null;
					$groupNameKey = $pairing[$pairingK]['groupNameKey'] ?? null;

					if ($pairKey && $groupName && $groupNameKey) {
						$relatedGroupId = $this->importHelper->findRelatedGroupByName($groupName);

						foreach ($chunks as $chunk) {
							foreach ($chunk as $rowKey => $row) {
								$relatedList = $row[$pairKey] ?? null;
								$productId   = null;

								if (!$relatedList) {
									continue;
								}

								if ($groupNameKey === 'supplierCode') {
									$code = $this->fixString((string) $row[$pairing['supplierCode']['key']]);

									if ($code) {
										$productId = $conn->fetchOne("SELECT id_product FROM eshop_catalog__product_supplier 
											WHERE id_supplier = ? AND code = ?", [$supplier->getId(), $code]);
									}
								} else {
									$code = $row[$pairing[$groupNameKey]['key']];

									if ($code) {
										$productId = $conn->fetchOne("SELECT id FROM eshop_catalog__product
											WHERE `{$groupNameKey}` = ?", [$code]);
									}
								}

								if ($productId && ($applyOn !== 'new' || Arrays::contains($newIds, $productId))) {
									$relatedIds = [];

									if ($groupNameKey === 'supplierCode') {
										foreach ($conn->fetchAllAssociative("SELECT id_product FROM eshop_catalog__product_supplier 
											WHERE id_supplier = ? AND code IN ('" . implode("','", $relatedList) . "')", [$supplier->getId()]) as $v) {
											/** @var array $v */
											$relatedIds[] = (int) $v['id_product'];
										}
									} else {
										foreach ($conn->fetchAllAssociative("SELECT id FROM eshop_catalog__product 
											WHERE `{$groupNameKey}` IN ('" . implode("','", $relatedList) . "')") as $v) {
											/** @var array $v */
											$relatedIds[] = (int) $v['id_product'];
										}
									}

									if (!empty($relatedIds)) {
										$this->importHelper->addProductsToRelatedGroup($relatedGroupId, $productId, $relatedIds);
									}
								}
							}
						}
					}
				}
			}
		} catch (Exception $e) {
			Debugger::log($e, 'import_related');
			LogNotifier::toDevelopers($e->getMessage(), 'failed set related');
		}

		try {
			$this->importHelper->mergeVariantGalleries();
		} catch (Exception $e) {
			Debugger::log($e, 'import');
			LogNotifier::toDevelopers($e->getMessage(), 'failed to merge variant galleries');
		}

		AvailabilityService::$allowCheckCountInCategories = true;

		return $result;
	}

	/**
	 * TODO udělat globální třídu
	 *
	 *
	 * @return mixed
	 */
	protected function fixString(mixed $str)
	{
		if (!is_string($str)) {
			return $str;
		}

		$str = htmlspecialchars_decode($str, ENT_QUOTES);
		$str = html_entity_decode($str);

		return Strings::trim($str);
	}

	/**
	 * @param Album[] $galleries
	 */
	public function downloadGallery(
		array         &$row,
		string        &$lang,
		array         &$pairing,
		bool          &$exist,
		Product       $product,
		?ProductTexts $productTexts,
		array         &$galleries
	): void
	{
		if (ImportDebugger::$allowImagesDownload && (isset($pairing['images']) && (!$exist || $pairing['images']['applyOn'] != 'new'))) {
			$gallery = $this->importHelper->getGallery($product);
			foreach ($gallery->getImages()->toArray() as $k => $img) {
				if (!file_exists($img->getFile())) {
					$this->em->remove($img);
					$gallery->getImages()->remove($k);
				}
			}
			$galleryText = $gallery->getText($lang);
			if (!$galleryText instanceof AlbumText) {
				if ($gallery->getId()) {
					$galleryText = $this->em->getRepository(AlbumText::class)->createQueryBuilder('at')
						->where('at.album = :album')->andWhere('at.lang = :lang')
						->setParameters(new ArrayCollection([new Parameter('album', $gallery->getId()), new Parameter('lang', $lang)]))->getQuery()->getOneOrNullResult();
				}

				if (!$galleryText) {
					$galleryText = new AlbumText($gallery, $lang);
				}
			}
			if ($productTexts instanceof ProductTexts) {
				$galleryText->title = $productTexts->name;
			}
			$this->em->persist($galleryText);
			// Zjištění prefixu pokud je nastaven
			$urlPrefix = (string) ($pairing['imagesUrlPrefix']['key'] ?? $pairing['imagesUrlPrefix']['fixedValue']);
			$product->setGallery($gallery);
			$galleries[] = $gallery;
			// Získání obrázků z importu a přidání do galerie
			foreach ($pairing['images']['key'] as $k) {
				foreach ($this->importHelper->getImages($row[$k], $urlPrefix) as $img) {
					if ($img) {
						$img = $this->parseImgUrl($img, $row);
						$this->importHelper->addImage($gallery, $img);
					}
				}
			}
		}
	}

	protected function parseImgUrl(string $img, array $row): string
	{
		if (!\str_starts_with($img, 'http') && !\str_starts_with($img, 'ftp')) {
			$img = 'http://' . $img;
		}

		return $img;
	}

	protected function getCodesForSkip(int $supplierId): array
	{
		if (!array_key_exists($supplierId, $this->cSkipCodes)) {
			$this->cSkipCodes[$supplierId] = [];

			foreach ($this->em->getConnection()->fetchAllAssociative("SELECT code FROM eshop_catalog__product_supplier_skip WHERE id_supplier = ?", [$supplierId]) as $row) {
				$this->cSkipCodes[$supplierId][$row['code']] = $row['code'];
			}
		}

		return $this->cSkipCodes[$supplierId];
	}
}
