<?php declare(strict_types = 1);

namespace EshopCatalog\AdminModule\Model;

use Core\Model\Helpers\BaseEntityService;
use Core\Model\Helpers\Traits\TPublish;
use EshopCatalog\Model\Config;
use EshopCatalog\Model\Entities\ProductChangePlan;
use EshopCatalog\Model\Entities\VatRate;
use Nette\Utils\DateTime;
use Tracy\Debugger;

/**
 * @method ProductChangePlan|null getReference($id)
 * @method ProductChangePlan[] getAll()
 * @method ProductChangePlan|null get($id)
 */
class ProductsChangePlan extends BaseEntityService
{
	use TPublish;

	protected $entityClass = ProductChangePlan::class;

	protected array $changeForProductList = [];

	public function __construct(
		protected Products $productServices,
	)
	{
	}

	/**
	 * @return ProductChangePlan[]
	 */
	public function getPlanForProduct(int $productId): array
	{
		if (!array_key_exists($productId, $this->changeForProductList)) {
			$this->changeForProductList[$productId] = array_merge(
				$this->getEr()->createQueryBuilder('p')
					->where('p.product = :productId')
					->andWhere('p.changed IS NULL')
					->setParameter('productId', $productId)
					->orderBy('p.changeAt', 'ASC')
					->getQuery()
					->getResult(),
				$this->getEr()->createQueryBuilder('p')
					->where('p.product = :productId')
					->andWhere('p.changed IS NOT NULL')
					->setParameter('productId', $productId)
					->orderBy('p.changeAt', 'DESC')
					->getQuery()
					->getResult(),
			);
		}

		return $this->changeForProductList[$productId];
	}

	public function runUpdate(): void
	{
		if (!Config::load('product.allowPlan', false)) {
			return;
		}

		$cacheIds = [];

		foreach ($this->getEr()->createQueryBuilder('p')
			         ->where('p.changed IS NULL')
			         ->andWhere('p.changeAt <= :now')
			         ->setParameter('now', new \DateTime)
			         ->getQuery()
			         ->getResult() as $plan) {
			if ($plan->getColumn() === ProductChangePlan::columnVatRate) {
				/** @var VatRate|null $tmp */
				$tmp = $this->em->getReference(VatRate::class, $plan->getValue());
				if ($tmp) {
					$plan->getProduct()->vatRate = $tmp;
					$this->em->persist($plan->getProduct());

					$plan->change();
					$this->em->persist($plan);
					$cacheIds[] = (int) $plan->getProduct()->getId();

					Debugger::log("Zmena DPH u produktu {$plan->getProduct()->getId()} na {$tmp->getId()} ({$tmp->rate})", 'productChangePlan');
				}
			}
		}

		$this->em->flush();

		foreach ($cacheIds as $id) {
			$this->productServices->clearProductCache($id);
		}
	}

	public function runInitialVatChange(): void
	{
		if (!Config::load('product.allowPlan', false)) {
			return;
		}

		// Spustit az na novy rok
		if ((new DateTime())->format('Y-m-d H:i:s') < '2024-01-01 00:00:00') {
			return;
		}

		// Kontrola jestli uz bylo spusteni
		$checkFile = LOG_DIR . '/2024VatChanged.txt';
		if (file_exists($checkFile)) {
			return;
		}

		// Najduti DPH 12
		/** @var VatRate|null $newRate */
		$newRate = $this->em->getRepository(VatRate::class)->findOneBy(['rate' => 12]);
		if (!$newRate) {
			return;
		}

		// Zapis do souboru pro kontrolu spusteni
		file_put_contents($checkFile, (new DateTime())->format('Y-m-d H:i:s'));

		// Nacteni DPH pro zmenu
		$ratesForChange = [];
		foreach ($this->em->getConnection()->fetchAllAssociative("SELECT id FROM eshop_catalog__vat_rate WHERE rate = 10 OR rate = 15") as $row) {
			$ratesForChange[] = $row['id'];
		}

		if ($ratesForChange === []) {
			return;
		}

		// Zmena DPH u produktu
		$count = $this->em->getConnection()->executeStatement("UPDATE eshop_catalog__product SET id_vat_rate = {$newRate->getId()} WHERE id_vat_rate IN (" . implode(',', $ratesForChange) . ")");
		Debugger::log("Zmena DPH u produktu: $count", 'vatChange2024');

		// Smazani cache
		foreach ($this->em->getConnection()->fetchAllAssociative("SELECT id FROM eshop_catalog__product WHERE id_vat_rate = " . $newRate->getId()) as $row) {
			$this->productServices->clearProductCache((int) $row['id']);
		}
	}
}
