<?php declare(strict_types = 1);

namespace EshopCatalog\AdminModule\Model\Subscribers;

use Core\Model\Entities\EntityManagerDecorator;
use Core\Model\Event\Event;
use Core\Model\Event\FormSuccessEvent;
use Core\Model\Sites;
use EshopCatalog\AdminModule\Components\Categories\CategoryForm;
use EshopCatalog\Model\Entities\Category;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use Tracy\Debugger;

class CategorySubscriber implements EventSubscriberInterface
{
	protected EntityManagerDecorator $em;
	protected Sites                  $sites;

	public function __construct(
		EntityManagerDecorator $em,
		Sites                  $sites
	)
	{
		$this->em    = $em;
		$this->sites = $sites;
	}

	public static function getSubscribedEvents(): array
	{
		return [
			'eshopCatalog.category.checkProductsCount'     => 'checkProductsCount',
			CategoryForm::class . '::formSuccessAfterSave' => 'formSuccessAfterSave',
		];
	}

	public function checkProductsCount(Event $event): void
	{
		$conn      = $this->em->getConnection();
		$site      = $this->sites->getCurrentSite();
		$siteIdent = $site->getIdent();

		try {
			$allInserts = [];
			$allParams  = [];

			// Připravit všechny categories
			foreach ($event->data['categories'] ?? [] as $v) {
				$allInserts[] = "(?, ?, NOW())";
				$allParams[]  = $v;
				$allParams[]  = $siteIdent;
			}

			// Připravit všechny products
			foreach ($event->data['products'] ?? [] as $v) {
				$allInserts[] = "(?, NULL, NOW())";
				$allParams[]  = 'product_' . $v;
			}

			// Jeden velký INSERT
			if (!empty($allInserts)) {
				$sql = "INSERT IGNORE INTO eshop_catalog__category_check_products_count (category_id, site_ident, created) VALUES "
					. implode(', ', $allInserts);
				$conn->executeStatement($sql, $allParams);
			}
		} catch (\Exception $e) {
			Debugger::log($e, '_checkproductscount');
		}
	}

	public function formSuccessAfterSave(FormSuccessEvent $event): void
	{
		/** @var Category $entity */
		$entity = $event->custom['entity'];

		$this->checkProductsCount(new Event(['categories' => [$entity->getId()]]));
	}
}
