<?php declare(strict_types = 1);

namespace EshopCatalog\ApiModule\Api\V1\Controllers;

use Apitte\Core\Annotation\Controller\Method;
use Apitte\Core\Annotation\Controller\Path;
use Apitte\Core\Annotation\Controller\RequestParameter;
use Apitte\Core\Annotation\Controller\RequestParameters;
use Apitte\Core\Http\ApiRequest;
use Apitte\Core\Http\ApiResponse;
use EshopCatalog\FrontModule\Model\Categories;
use EshopCatalog\FrontModule\Model\Dao\Category;
use EshopCatalog\FrontModule\Model\Products;
use EshopCatalog\FrontModule\Model\ProductsFacade;

/**
 * @Path("/category")
 */
class CategoriesController extends BaseController
{
	public function __construct(
		protected Categories     $categoriesDaoService,
		protected Products       $productsDaoService,
		protected ProductsFacade $productsFacade,
	)
	{
		ProductsFacade::setMode(ProductsFacade::MODE_CHECKOUT);
	}

	/**
	 * @Path("/")
	 * @Method("GET")
	 */
	public function index(ApiRequest $request, ApiResponse $response): array
	{
		$result = [];

		foreach ($this->categoriesDaoService->getAllRootIds() as $siteIdent => $id) {
			$result[$siteIdent] = $this->categoriesDaoService->getStructured($id);
		}

		return $result;
	}

	/**
	 * TODO
	 * Nekdy az bude chodit lepsi DAO dodelat.
	 *
	 * @Path("/{id}/subcategories")
	 * @Method("GET")
	 * @RequestParameters({
	 *      @RequestParameter(name="id", type="int", description="category id", deprecated=true)
	 * })
	 */
	public function subcategories(ApiRequest $request, ApiResponse $response): array
	{
		$id  = $request->getParameter('id');
		$cat = $this->categoriesDaoService->get((int) $id);

		return $cat instanceof Category ? array_values($cat->getChild()) : [];
	}

	/**
	 * @Path("/{id}")
	 * @Method("GET")
	 * @RequestParameters({
	 *      @RequestParameter(name="id", type="int", description="category id")
	 * })
	 */
	public function category(ApiRequest $request, ApiResponse $response): ApiResponse
	{
		$category = $this->categoriesDaoService->get((int) $request->getParameter('id'));
		if (!$category instanceof Category) {
			$this->sendError($response, "Category with this id doesn´t exist");
		}

		return $response->writeJsonBody(['data' => (array) $category]);
	}

	/**
	 * @Path("/{id}/products")
	 *
	 * @Method("GET")
	 * @RequestParameters({
	 *      @RequestParameter(name="id", type="int", description="Category id"),
	 *      @RequestParameter(name="offset", type="int", description="Start", in="query", required=false),
	 *      @RequestParameter(name="limit", type="int", description="End", in="query", required=false),
	 *      @RequestParameter(name="sort", type="string", description="Sort", in="query", required=false)
	 * })
	 */
	public function categoryProducts(ApiRequest $request, ApiResponse $response): array
	{

		$id     = $request->getParameter('id');
		$offset = $request->getQueryParam('offset', 0);
		$limit  = $request->getQueryParam('limit', 0);
		$sort   = $request->getQueryParam('sort', null);

		$productIds = $this->productsDaoService->getProductsIdInCategory($id, $offset, $limit, $sort);
		$products   = $productIds !== [] ? $this->productsFacade->getProducts($productIds) : [];

		if ($products === []) {
			$this->sendError($response, "Category with this id doesn´t exist or the request failed");
		}

		return array_values($products);
	}

}
