<?php declare(strict_types = 1);

namespace EshopCatalog\Console;

use Core\Model\Entities\EntityManagerDecorator;
use Doctrine\DBAL\Logging\Middleware;
use EshopCatalog\AdminModule\Model\ProductReplenishmentPlanner;
use EshopCatalog\Model\Config;
use Exception;
use Psr\Log\NullLogger;
use Symfony\Component\Console\Attribute\AsCommand;
use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Output\OutputInterface;
use Override;
use Tracy\Debugger;

#[AsCommand(name: 'eshop-catalog:calculate-product-replenishment')]
class CalculateProductReplenishment extends Command
{
	public function __construct(
		protected EntityManagerDecorator      $em,
		protected ProductReplenishmentPlanner $planner,
	)
	{
		parent::__construct();
	}

	#[Override]
	protected function configure(): void
	{
		$this->setName('eshop-catalog:calculate-product-replenishment')
			->setDescription('Calculate product replenishment');
	}

	#[Override]
	protected function execute(
		InputInterface  $input,
		OutputInterface $output,
	): int
	{
		if (!Config::load('productReplenishmentPlanner.enable')) {
			$output->writeLn('<error>[ERROR] - Not enabled</error>');

			return Command::FAILURE;
		}

		ini_set('memory_limit', '5G');
		set_time_limit(7000);
		$this->em->getConnection()->getConfiguration()->setMiddlewares([new Middleware(new NullLogger)]);
		$output->writeln('Calculate product replenishment');

		try {
			Debugger::dispatch();

			$this->planner->runCalc();

			$this->em->clear();
			gc_collect_cycles();

			$output->writeLn('<info>[OK] - Calculated</info>');
		} catch (Exception $e) {
			Debugger::log($e);
			$output->writeln('<error>[ERROR] - ' . $e->getMessage() . '</error>');

			return Command::FAILURE;
		}

		return Command::SUCCESS;
	}
}
