<?php declare(strict_types = 1);

namespace EshopCatalog\Console;

use Contributte\Translation\Translator;
use Core\Model\Sites;
use Doctrine;
use Doctrine\DBAL\Logging\Middleware;
use EshopCatalog\AdminModule\Model\Categories;
use EshopCatalog\FrontModule\Model\Categories as FrontCategories;
use EshopCatalog\FrontModule\Model\ProductsFacade;
use Exception;
use Override;
use Psr\Log\NullLogger;
use Symfony\Component\Console\Attribute\AsCommand;
use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Output\OutputInterface;
use Tracy\Debugger;

#[AsCommand(name: 'eshop-catalog:check-category-products')]
class CheckCategoryProducts extends Command
{
	public function __construct(
		protected Translator      $translator,
		protected FrontCategories $frontCategories,
		protected Categories      $categories,
		protected ProductsFacade  $productsFacade,
		protected Sites           $sites,
	)
	{
		parent::__construct();
	}

	#[Override]
	protected function configure(): void
	{
		$this->setName('eshop-catalog:check-category-products')
			->setDescription('Check category products count');
	}

	#[Override]
	protected function execute(
		InputInterface  $input,
		OutputInterface $output,
	): int
	{
		ini_set('memory_limit', '5G');
		$this->categories->em->getConnection()->getConfiguration()->setMiddlewares([new Middleware(new NullLogger)]);
		$output->writeln('Check category products count');

		try {
			Debugger::dispatch();

			foreach ($this->sites->getSites() as $site) {
				$siteIdent = $site->getIdent();
				$lang      = $site->currentLang;

				$output->writeLn('<info>Checking - ' . $siteIdent . '</info>');

				$this->translator->setLocale($lang);
				Sites::$currentIdentOverride = $siteIdent;
				Sites::$currentLangOverride  = $lang;
				ProductsFacade::$forceLocale = $lang;

				$this->categories->checkVirtualGroup($site);

				$rootId = $this->frontCategories->getRootIdForSite($siteIdent);
				if ($rootId) {
					$categories = array_keys($this->frontCategories->getCategories($rootId));

					if (!empty($categories)) {
						$this->categories->checkProductsCountInCategories($categories);
					}
				}

				$this->productsFacade->clearTemp();
				$this->categories->em->clear();
				gc_collect_cycles();
			}

			$output->writeLn('<info>[OK] - Checked</info>');

			return 0;
		} catch (Exception $e) {
			$output->writeln('<error>[ERROR] - ' . $e->getMessage() . '</error>');
		}

		return 1;
	}

}
