<?php declare(strict_types = 1);

namespace EshopCatalog\Console;

use Contributte\Translation\Translator;
use Core\Model\Entities\EntityManagerDecorator;
use Core\Model\Helpers\Strings;
use Core\Model\Sites;
use Doctrine;
use Doctrine\DBAL\Logging\Middleware;
use EshopCatalog\AdminModule\Model\Categories;
use EshopCatalog\FrontModule\Model\ProductsFacade;
use Exception;
use Override;
use Psr\Log\NullLogger;
use Symfony\Component\Console\Attribute\AsCommand;
use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Output\OutputInterface;
use Tracy\Debugger;

#[AsCommand(name: 'eshop-catalog:check-category-products-from-list')]
class CheckCategoryProductsFromList extends Command
{
	public function __construct(
		protected Translator             $translator,
		protected Categories             $categories,
		protected ProductsFacade         $productsFacade,
		protected Sites                  $sites,
		protected EntityManagerDecorator $em,
	)
	{
		parent::__construct();
	}

	#[Override]
	protected function configure(): void
	{
		$this->setName('eshop-catalog:check-category-products-from-list')
			->setDescription('Check category products count');
	}

	#[Override]
	protected function execute(
		InputInterface  $input,
		OutputInterface $output,
	): int
	{
		$tmpFile = TMP_DIR . '/eshopCatalogCheckCategoryProductsFromList.txt';

		$lastRun = file_exists($tmpFile) ? (int) file_get_contents($tmpFile) : null;
		$now     = time();

		if ($lastRun && ($now - $lastRun) < 1300) {
			$output->writeln('Multihub DropShip - category update - already running');

			return 0;
		}

		ini_set('memory_limit', '5G');
		$this->em->getConnection()->getConfiguration()->setMiddlewares([new Middleware(new NullLogger)]);
		$output->writeln('Check category products count');

		try {
			Debugger::dispatch();
			$conn = $this->em->getConnection();

			$categories         = [];
			$categoriesWithSite = [];
			$products           = [];

			$processed = [];

			foreach ($conn->executeQuery('SELECT category_id, site_ident FROM `eshop_catalog__category_check_products_count` LIMIT 100')->iterateAssociative() as $row) {
				if (Strings::startsWith((string) $row['category_id'], 'product_')) {
					$products[] = (int) substr($row['category_id'], 8);
				} else if ($row['site_ident']) {
					$categoriesWithSite[$row['site_ident']][] = (int) $row['category_id'];
				} else {
					$categories[] = (int) $row['category_id'];
				}
			}

			if (!empty($products)) {
				foreach ($conn->executeQuery('SELECT id_category FROM eshop_catalog__category_product WHERE id_product IN (' . implode(',', $products) . ')')->iterateAssociative() as $row) {
					/** @var array $row */
					$categories[] = (int) $row['id_category'];
				}

				foreach ($conn->executeQuery('SELECT category_id, site FROM eshop_catalog__product_in_site WHERE product_id IN (' . implode(',', $products) . ')')->iterateAssociative() as $row) {
					/** @var array $row */
					if ($row['site']) {
						$categoriesWithSite[$row['site']][] = (int) $row['category_id'];
					} else {
						$categories[] = (int) $row['category_id'];
					}
				}
			}

			if (!empty($categories) || !empty($categoriesWithSite)) {
				foreach ($this->sites->getSites() as $site) {
					$siteIdent = $site->getIdent();
					$lang      = $site->currentLang;

					$output->writeLn('<info>Checking - ' . $siteIdent . '</info>');

					$this->translator->setLocale($lang);
					Sites::$currentIdentOverride = $siteIdent;
					Sites::$currentLangOverride  = $lang;
					ProductsFacade::$forceLocale = $lang;

					$cats = array_unique(array_merge($categories, $categoriesWithSite[$siteIdent] ?? []));

					if (!empty($cats)) {
						$this->categories->checkProductsCountInCategories($cats);

						$this->productsFacade->clearTemp();
						$this->categories->em->clear();
						gc_collect_cycles();
					}

					foreach ($cats as $catId) {
						$processed[$catId] = true;
					}
				}
			}

			if (!empty($processed)) {
				$conn->executeQuery('DELETE FROM `eshop_catalog__category_check_products_count` WHERE category_id IN (' . implode(',', array_keys($processed)) . ')');
			}

			$output->writeLn('<info>[OK] - Checked</info>');

			return 0;
		} catch (Exception $e) {
			$output->writeln('<error>[ERROR] - ' . $e->getMessage() . '</error>');
		}

		unlink($tmpFile);

		return 1;
	}

}
