<?php declare(strict_types = 1);

namespace EshopCatalog\Console;

use Contributte\Translation\Translator;
use Core\Model\Lang\Langs;
use Core\Model\Sites;
use Currency\Model\Currencies;
use Doctrine;
use EshopCatalog\CronModule\Model\ProductsFeed;
use EshopCatalog\FrontModule\Model\ProductsFacade;
use Exception;
use Override;
use Symfony\Component\Console\Attribute\AsCommand;
use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Input\InputOption;
use Symfony\Component\Console\Output\OutputInterface;
use Tracy\Debugger;

#[AsCommand(name: 'eshop-catalog:feed')]
class Feed extends Command
{
	public function __construct(
		protected Langs        $langs,
		protected Translator   $translator,
		protected ProductsFeed $productsFeed,
		protected Sites        $sites,
	)
	{
		parent::__construct();
	}

	#[Override]
	protected function configure(): void
	{
		$this->setName('eshop-catalog:feed')
			->addOption('site', null, InputOption::VALUE_REQUIRED)
			->addOption('lang', null, InputOption::VALUE_REQUIRED)
			->addOption('currency', null, InputOption::VALUE_OPTIONAL)
			->addOption('onlyStock', null, InputOption::VALUE_OPTIONAL)
			->setDescription('Generate feed');
	}

	#[Override]
	protected function execute(
		InputInterface  $input,
		OutputInterface $output,
	): int
	{
		$output->writeln('GENERATE FEED');

		try {
			ini_set('memory_limit', '5G');
			Debugger::dispatch();

			$site      = $input->getOption('site');
			$currency  = $input->getOption('currency') ?: 'CZK';
			$lang      = $input->getOption('lang');
			$onlyStock = $input->getOption('onlyStock') === 'true';

			$this->translator->setLocale($lang);
			Sites::$currentIdentOverride     = $site;
			Sites::$currentLangOverride      = $lang;
			ProductsFacade::$forceLocale     = $lang;
			Currencies::$currentCodeOverride = $currency;
			Currencies::clearConvertedProducts($currency);
			$result = $this->productsFeed->getProducts(false, $onlyStock);
			$output->writeln($result['count']);

			$output->writeLn('<info>[OK] - feed generated</info>');

			return 0;
		} catch (Exception $e) {
			$output->writeln('<error>[ERROR] - ' . $e->getMessage() . '</error>');
		}

		return 1;
	}

}
