<?php declare(strict_types = 1);

namespace EshopCatalog\FrontModule\Components\Personalization;

use Core\Model\Helpers\Strings;
use Core\Model\UI\Form\BaseForm;
use EshopCatalog\FrontModule\Components\IProductPreviewFactory;
use EshopCatalog\FrontModule\Model\Dao\Category;
use EshopCatalog\FrontModule\Model\Dao\Product;
use EshopCatalog\FrontModule\Model\Products;
use EshopCatalog\Model\Config;
use EshopCatalog\Model\Personalization\Personalization;
use EshopCatalog\Model\Personalization\PersonalizationAccessControl;
use EshopOrders\Model\Entities\Customer;
use Nette\Application\Attributes\Persistent;
use Nette\ComponentModel\IComponent;
use Nette\Utils\Paginator;
use Override;

class ProductListingFilter extends PersonalizationControl
{
	#[Persistent]
	public ?int $limit = null;

	public ?int         $overrideLimit = null;
	public int          $page          = 1;
	public int          $start         = 0;
	public int          $count         = 0;
	protected Paginator $paginator;
	protected string    $query         = '';

	/** @var Product[] */
	protected array $result = [];

	/** @var Category[] */
	protected array $categories = [];

	#[Persistent]
	public ?array $filter = [];

	public function __construct(
		protected Config                       $config,
		protected Products                     $productsService,
		protected Personalization              $personalization,
		protected PersonalizationAccessControl $personalizationAccessControl,
		protected IProductPreviewFactory       $productPreviewFactory,
	)
	{
	}

	#[Override]
	protected function attached(IComponent $presenter): void
	{
		parent::attached($presenter);

		if (!$this->isAllowed()) {
			return;
		}

		$params = $this->getNormlizeParams();

		/** @var Customer|null $customer */
		$customer = $this->template->customer;
		$query    = $this->getQuery();
		$data     = Strings::length($query) !== 0
			? $this->personalization->getProductsByFilter($query, $this->getLimit(), $this->getOffset(), array_key_exists('fc', $params) ? $params['fc'] : [], $customer)
			: ['result' => [], 'total' => 0];

		$this->result = [];
		foreach ($data['result'] as $product) {
			$this->result[$product->getId()] = $product;
		}

		$this->count      = $data['total'];
		$this->categories = $this->personalization->getCategoriesByFilter($this->getQuery())['result'];

		if ($this->result) {
			$this->productsService->loadQuantity($this->result);
		}
		$this->getPaginator();
	}

	public function render(): void
	{
		$list = [];
		foreach ((array) Config::load('productsList.itemsPerPageList') as $v) {
			$list[$v] = $this->link('limit!', $v);
		}

		$this->template->limit      = $this->getLimit();
		$this->template->listLimits = $list;
		$this->template->paginator  = $this->getPaginator();
		$this->template->products   = $this->result;
		$this->template->categories = $this->categories;
		$this->template->filterForm = $this->getComponent('filter');
		$this->template->render($this->getTemplateFile());
	}

	public function createComponentFilter(): BaseForm
	{
		$form = $this->createForm();

		$categories = [];
		foreach ($this->categories as $cat) {
			$categories[$cat->getId()] = $cat->nameH1 ?: $cat->name;
		}

		$form->addCheckboxList('fc', $this->t('eshopCatalogFront.advicio.productListingFilter.chooseCategory'), $categories);

		$form->onAnchor[] = function() use ($form): void {
			$params = $this->getNormlizeParams();
			$form   = $this->getComponent('filter');

			$form->setDefaults($params);
		};

		return $form;
	}

	public function handleSet(): void
	{
		if ($this->getPresenter()->isAjax()) {
			$this->redrawControl('list');
			$this->redrawControl('filters');
			$this->redrawControl('listWrap');
			$this->redrawControl('paginator');
		}
	}

	public function handlePaginator(int $page): void
	{
		$this->page  = $page;
		$this->start = $page > 1 ? $this->getLimit() * ($page - 1) : 0;

		if ($this->getPresenter()->isAjax()) {
			$this->redrawControl('list');
			$this->redrawControl('filters');
			$this->redrawControl('listWrap');
			$this->redrawControl('paginator');
		}
	}

	public function handleLimit(int $limit): void
	{
		if ($limit === (int) Config::load('productsList.itemsPerPage') || ($this->overrideLimit && $limit === $this->overrideLimit)) {
			$this->limit = null;
		} else {
			$this->limit = $limit;
		}
	}

	protected function getNormlizeParams(): array
	{
		$normalizeParamArray = static function(?string $value) {
			if (!$value) {
				return [];
			}

			if (!\str_contains($value, '|')) {
				return [$value];
			}

			return explode('|', $value);
		};

		$result = [];
		foreach ($this->filter as $k => $f) {
			if (is_array($f)) {
				$result[$k] = $normalizeParamArray($f[0]);
			}
			// dalsi typy
		}

		return $result;
	}

	protected static function getDenormalizeParams(array $params): array
	{
		$result = [];
		foreach ($params as $k => $f) {
			if (is_array($f)) {
				$result[$k][0] = implode('|', $f);
			}
			// dalsi typy
		}

		return $result;
	}

	protected function getQuery(): string
	{
		return trim($this->presenter->getParameter('q') ?? '');
	}

	protected function getOffset(): int
	{
		$page = $this->getParameter('page');

		return $page ? ($page - 1) * $this->getLimit() : 0;
	}

	protected function getLimit(): int
	{
		if ($this->overrideLimit) {
			return $this->overrideLimit;
		}

		return $this->limit ?: (int) Config::load('productsList.itemsPerPage');
	}

	public function getPaginator(): Paginator
	{
		$itemsPerPage = $this->getLimit();

		if ($this->page === 1) {
			$pathParts      = explode('/', (string) $_SERVER['REQUEST_URI']);
			$paginatorIndex = array_search($this->translator->translate('default.urlPart.page'), $pathParts);

			if ($paginatorIndex !== false) {
				$this->page = (int) $pathParts[$paginatorIndex + 1];
			}
		}

		$paginator = new Paginator;
		$paginator->setItemCount($this->count);
		$paginator->setItemsPerPage($itemsPerPage);
		$paginator->setPage($this->page);

		return $paginator;
	}

	public function isAllowed(): bool
	{
		return $this->personalizationAccessControl->isProductListingFilterAllowed();
	}

	public function showFilter(): bool
	{
		return $this->categories !== [];
	}

	public function createFilterLink(string $key, string $value): string
	{
		$filter = $this->getNormlizeParams();

		if (array_key_exists($key, $filter) && in_array($value, $filter[$key])) {
			unset($filter[$key][(int) array_search($value, $filter[$key])]);
		} else {
			$filter[$key][] = $value;
		}

		return $this->link('set!', ['filter' => self::getDenormalizeParams($filter)]);
	}

}
